package edu.unika.aifb.kaon.virtualoimodel;

import java.util.Set;
import java.util.Map;
import java.util.HashMap;
import java.util.Iterator;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.vocabulary.*;

/**
 * Implementation of the LexicalEntry interface that relies on the RDF API as the model.
 *
 * @author Ljiljana Stojanovic (Ljiljana.Stojanovic@fzi.de)
 */
public class VirtualLexicalEntry extends VirtualInstance implements LexicalEntry {
    /**
     * Creates an instance of this class and attaches it to the virtual OI-model.
     *
     * @param oimodel                   virtual OI-model
     */
    public VirtualLexicalEntry(VirtualOIModel oimodel) {
        super(oimodel);
    }
    /**
     * Retruns the set of URI determining the type of this entry.
     *
     * @return                      URI determining the type of this entry
     */
    public String getTypeURI() throws KAONException {
        Set set=getParentConcepts();
        if (set.size()!=2)
            throw new KAONException("Lexical entry must have exactly one parent that is not root.");
        Iterator iterator=set.iterator();
        while (iterator.hasNext()) {
            Concept concept=(Concept)iterator.next();
            if (!concept.equals(m_oimodel.getRootConcept()))
                return concept.getURI();
        }
        throw new KAONException("Lexical entry must have exactly one parent that is not root.");
    }
    /**
     * Returns all named entities referenced by this instance.
     *
     * @return                      set of named entities referenced by this entry
     */
    public Set getReferencedEntities() throws KAONException {
        Property references=m_oimodel.getProperty(KAONVocabularyAdaptor.INSTANCE.getReferences());
        if (!references.isInOIModel())
            throw new KAONException("Lexical layer is not included in associated OI-model");
        return getFromPropertyValues(references);
    }
    /**
     * Returns the map of attribute values indexed by appropriate URIs. Elements are either <code>String</code> or <code>Instance</code> objects.
     *
     * @return                      map of attribute values
     */
    public Map getAttributeValues() throws KAONException {
        Map result=new HashMap();
        Map map=getFromPropertyValues();
        Iterator keys=map.keySet().iterator();
        while (keys.hasNext()) {
            Property key=(Property)keys.next();
            if (!key.getURI().equals(KAONVocabularyAdaptor.INSTANCE.getReferences())) {
                Set set=(Set)map.get(key);
                result.put(key.getURI(),set.iterator().next());
            }
        }
        return result;
    }
    /**
     * Returns the value of attribute of this object. Return value may be either a <code>String</code> or an <code>Instance</code> object.
     *
     * @param attributeURI          URI of the attribute
     * @return                      value of the attribute with given URI (may be a <code>String</code>, <code>Instance</code> of <code>null</code>)
     */
    public Object getAttribute(String attributeURI) throws KAONException {
        Property property=m_oimodel.getProperty(attributeURI);
        if (!property.isInOIModel())
            return null;
        return getFromPropertyValue(property);
    }
    /**
     * Returns the language attribute of this lexical entry.
     *
     * @return                      language of this lexical entry
     */
    public String getInLanguage() throws KAONException {
        Object value=getAttribute(KAONVocabularyAdaptor.INSTANCE.getInLanguage());
        if (value instanceof Instance)
            return ((Instance)value).getURI();
        else
            return null;
    }
    /**
     * Returns the value attribute of this lexical entry.
     *
     * @return                      value of this lexical entry
     */
    public String getValue() throws KAONException {
        return (String)getAttribute(KAONVocabularyAdaptor.INSTANCE.getValue());
    }
    /**
     * Accepts an entity visitor.
     *
     * @param visitor                   visitor to apply
     */
    public void accept(EntityVisitor visitor) throws KAONException {
        visitor.visit(this);
    }
}
