package edu.unika.aifb.kaon.virtualoimodel;

import java.util.Iterator;
import java.util.Set;
import java.util.HashSet;
import java.util.List;
import java.util.ArrayList;
import java.util.Map;
import java.util.HashMap;
import java.util.Collections;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

/**
 * Implementation of the Virtual Instance
 *
 * @author Ljiljana Stojanovic (Ljiljana.Stojanovic@fzi.de)
 */
public class VirtualInstance extends VirtualEntity implements Instance {
    /** Map tracker of parent concepts. */
    protected MapTracker m_parentConcepts;
    /** Tracker of property value from this instance. */
    protected PropertyValueTracker m_propertiesFrom;
    /** Tracker of property values to this instance. */
    protected PropertyValueTracker m_propertiesTo;

    /**
     * Creates an instance of this class and attaches it to the OI-model.
     *
     * @param oimodel                       virtual OI-model
     */
    public VirtualInstance(VirtualOIModel oimodel) {
        super(oimodel);
    }
    /**
     * Returns the original instance for the virtual instance.
     *
     * @return                              original instance of this virtual instance
     */
    public Instance getInstance() {
        return (Instance)m_entity;
    }
    /**
     * Returns set tracker of <code>Concepts</code> objects that specify which concepts is this instance instance of.
     *
     * @return                                  set of concepts that this instance is instance of (may be empty)
     */
    public MapTracker getParentConceptsTracker() {
        if (m_parentConcepts==null)
            m_parentConcepts=new MapTracker() {
                protected Object getMissingValue(Object key) throws KAONException {
                    return getInstance().getConceptInstanceOIModel(((VirtualConcept)key).getConcept());
                }
                protected Map loadMap() throws KAONException {
                    return getVirtualConceptsMap(getInstance().getParentConcepts(),true);
                }
            };
        return m_parentConcepts;
    }
    /**
     * Returns set of <code>Concepts</code> objects that specify which concepts is this instance instance of.
     *
     * @return                                  set of concepts that this instance is instance of (may be empty)
     */
    public Set getParentConcepts() throws KAONException {
        return getParentConceptsTracker().keySet();
    }
    /**
     * Returns set of all <code>Concepts</code> objects that this instance is instance of.
     *
     * @return                                  set of all concepts that this instance is instance of (may be empty)
     */
    public Set getAllParentConcepts() throws KAONException {
        // if there are hierarcy cycles, this method will loop forever
        Set parentConcepts=getParentConcepts();
        Set result=new HashSet(parentConcepts);
        Iterator iterator=parentConcepts.iterator();
        while (iterator.hasNext()) {
            Concept concept=(Concept)iterator.next();
            result.addAll(concept.getAllSuperConcepts());
        }
        return result;
    }
    /**
     * Tests if given concept is a direct parent of this instance.
     *
     * @param potentialParent           concept being tested
     * @return                          <code>true</code> if given concept is a direct parent of this instance
     */
    public boolean isDirectParent(Concept potentialParent) throws KAONException {
        return getParentConcepts().contains(m_oimodel.getConcept(potentialParent));
    }
    /**
     * Returns the OI-model containing the this concept - instance association.
     *
     * @param concept                   the concept of this instance
     * @return                          the OI-model in which the concept-instance association is declared
     */
    public OIModel getConceptInstanceOIModel(Concept concept) throws KAONException {
        if (isDirectParent(concept))
            return m_oimodel;
        else
            return null;
    }
    /**
     * Returns the original OI-model containing the this concept - instance association.
     *
     * @param concept                   the concept of this instance
     * @return                          the OI-model in which the concept-instance association is declared
     */
    public OIModel getConceptInstanceOIModelOriginal(VirtualConcept concept) throws KAONException {
        return (OIModel)getParentConceptsTracker().getValue(concept);
    }
    /**
     * Returns <code>true</code> if the supplied concept has been declared a parent of this instance in this OI-model.
     *
     * @param concept                   the concept of this instance
     * @return                          <code>true</code> if supplied concept has been declared a parent of this instance in this OI-model
     */
    public boolean isConceptInstanceDeclaredLocally(Concept concept) throws KAONException {
        return isDirectParent(concept);
    }
    /**
     * Tests if given concept is a parent of this concept.
     *
     * @param potentialParent           concept being tested
     * @return                          <code>true</code> if given concept is a parent of this instance
     */
    public boolean isParent(Concept potentialParent) throws KAONException {
        return getAllParentConcepts().contains(m_oimodel.getConcept(potentialParent));
    }
    /**
     * Returns the set of <code>PropertyInstnace</code> objects that point out from this instance.
     *
     * @return                                  set of property instances pointing from this instance (may be empty)
     */
    public Set getFromPropertyInstances() throws KAONException {
        Set set=new HashSet();
        Map propertyValues=getFromPropertyValues();
        Iterator properties=propertyValues.keySet().iterator();
        while (properties.hasNext()) {
            VirtualProperty virtualProperty=(VirtualProperty)properties.next();
            Iterator values=((Set)propertyValues.get(virtualProperty)).iterator();
            while (values.hasNext()) {
                Object value=values.next();
                PropertyInstance propertyInstance=m_oimodel.getPropertyInstance(virtualProperty,this,value);
                set.add(propertyInstance);
            }
        }
        return set;
    }
    /**
     * Returns a property value tracker for properties from this instance.
     *
     * @return                                  property value tracker for properties from this instance
     */
    public PropertyValueTracker getPropertiesFromTracker() {
        if (m_propertiesFrom==null)
            m_propertiesFrom=new PropertyValueTracker() {
                protected Map loadMap() throws KAONException {
                    return getVirtualPropertyValues(getInstance().getFromPropertyValues(),true);
                }
            };
        return m_propertiesFrom;
    }
    /**
     * Returns the map of sets of property values going out of the instance. The map is indexed by <code>Property</code> objects.
     *
     * @return                                  map of sets of property values
     */
    public Map getFromPropertyValues() throws KAONException {
        return getPropertiesFromTracker().getMap();
    }
    /**
     * Returns the value of a given property for this instance. If there is no value for a property, <code>null</code> is
     * returned. If there are multiple values for a property, one is picked arbitrarily and returned. Returned value is either
     * an <code>Instance</code> or a <code>String</code>.
     *
     * @param property                          property
     * @return                                  value of the property or <code>null</code> if there is no value
     */
    public Object getFromPropertyValue(Property property) throws KAONException {
        Set set=getFromPropertyValues(property);
        if (set.isEmpty())
            return null;
        else
            return set.iterator().next();
    }
    /**
     * Returns the set of values of a given property. It there are no values, an empty set is returned. Elements of the set
     * may be <code>Instance</code> or <code>String</code> objects.
     *
     * @param property                          property
     * @return                                  set of values of the property
     */
    public Set getFromPropertyValues(Property property) throws KAONException {
        VirtualProperty virtualProperty=m_oimodel.getProperty(property);
        Set set=(Set)getFromPropertyValues().get(virtualProperty);
        if (set==null)
            return Collections.EMPTY_SET;
        else
            return set;
    }
    /**
     * Tests whether this instance points to given value using specified property.
     *
     * @param property                          property
     * @param targetValue                       the value (can be a string or an instance)
     * @return                                  <code>true</code> if this instance points to given value through given property (by including all inferred facts)
     */
    public boolean pointsToValue(Property property,Object targetValue) throws KAONException {
        throw new KAONException("Virtual ontologies don't support inferences.");
    }
    /**
     * Returns the values of a property for this instance, including the inferred ones.
     *
     * @param property                          property for which the value is requested
     * @return                                  set of property values for this given property
     */
    public Set getAllFromPropertyValues(Property property) throws KAONException {
        throw new KAONException("Virtual ontologies don't support inferences.");
    }
    /**
     * Returns the values of all properties for this instance, including the inferred ones.
     *
     * @return                                  set of all property values for this instance
     */
    public Map getAllFromPropertyValues() throws KAONException {
        throw new KAONException("Virtual ontologies don't support inferences.");
    }
    /**
     * Adds all elements to from source to destination that are not in the control set as well.
     *
     * @param source                            source set
     * @param destination                       destination set
     * @param control                           control set
     */
    protected void addElements(Set source,Set destination,Set control) {
        Iterator iterator=source.iterator();
        while (iterator.hasNext()) {
            Object element=iterator.next();
            if (!control.contains(element))
                destination.add(element);
        }
    }
    /**
     * Returns the set of <code>PropertyInstnace</code> objects that point to this instance.
     *
     * @return                                  set of property instances pointing to this instance (may be empty)
     */
    public Set getToPropertyInstances() throws KAONException {
        Set set=new HashSet();
        Map propertyValues=getToPropertyValues();
        Iterator properties=propertyValues.keySet().iterator();
        while (properties.hasNext()) {
            VirtualProperty virtualProperty=(VirtualProperty)properties.next();
            Iterator values=((Set)propertyValues.get(virtualProperty)).iterator();
            while (values.hasNext()) {
                Instance value=(Instance)values.next();
                PropertyInstance propertyInstance=m_oimodel.getPropertyInstance(virtualProperty,value,this);
                set.add(propertyInstance);
            }
        }
        return set;
    }
    /**
     * Returns a property value tracker for properties to this instance.
     *
     * @return                                  property value tracker for properties to this instance
     */
    public PropertyValueTracker getPropertiesToTracker() {
        if (m_propertiesTo==null)
            m_propertiesTo=new PropertyValueTracker() {
                protected Map loadMap() throws KAONException {
                    return getVirtualPropertyValues(getInstance().getToPropertyValues(),true);
                }
            };
        return m_propertiesTo;
    }
    /**
     * Returns the map of sets of property values pointing to the instance. The map is indexed by <code>Property</code> objects.
     *
     * @return                                  map of sets of property values
     */
    public Map getToPropertyValues() throws KAONException {
        return getPropertiesToTracker().getMap();
    }
    /**
     * Returns the set of <code>Instnace</code> objects that point to this instance via given property.
     *
     * @param property                          property of interest
     * @return                                  set of <code>Instance</code> objects pointing to this instance via given property
     */
    public Set getToPropertyValues(Property property) throws KAONException {
        VirtualProperty virtualProperty=m_oimodel.getProperty(property);
        Set set=(Set)getToPropertyValues().get(virtualProperty);
        if (set==null)
            return Collections.EMPTY_SET;
        else
            return set;
    }
    /**
     * Returns the values of a property pointing to this instance, including the inferred ones.
     *
     * @param property                          property for which the value is requested
     * @return                                  set of all property values pointing to this instance for given property
     */
    public Set getAllToPropertyValues(Property property) throws KAONException {
        throw new KAONException("Virtual ontologies don't support inferences.");
    }
    /**
     * Returns the values of all properties pointing to this instance, including the inferred ones.
     *
     * @return                                  set of all property values pointing to this instance
     */
    public Map getAllToPropertyValues() throws KAONException {
        throw new KAONException("Virtual ontologies don't support inferences.");
    }
    /**
     * Returns the spanning instance for this instance.
     *
     * @return                                  spanning instance for this instance
     */
    public Instance getSpanningInstance() {
        return this;
    }
    /**
     * Returns the spanning concept for this instance.
     *
     * @return                                  spanning concept for this instance
     */
    public Concept getSpanningConcept() throws KAONException {
        return m_oimodel.getConcept(getURI());
    }
    /**
     * Returns the spanning property for this instance.
     *
     * @return                                  spanning property for this instance
     */
    public Property getSpanningProperty() throws KAONException {
        return m_oimodel.getProperty(getURI());
    }
    /**
     * Accepts an entity visitor.
     *
     * @param visitor                   visitor to apply
     */
    public void accept(EntityVisitor visitor) throws KAONException {
        visitor.visit(this);
    }
    /**
     * Adds a property value from this instance.
     *
     * @param virtualProperty           virtual property
     * @param targetValue               targetValue
     */
    void addFromPropertyValue(VirtualProperty virtualProperty,Object targetValue) {
        getPropertiesFromTracker().addPropertyValue(virtualProperty,targetValue);
    }
    /**
     * Sets a property value from this instance.
     *
     * @param virtualProperty           virtual property
     * @param targetValue               targetValue
     */
    void setFromPropertyValue(VirtualProperty virtualProperty,Object targetValue) {
        getPropertiesFromTracker().setPropertyValue(virtualProperty,targetValue);
    }
    /**
     * Removes a property value from this instance.
     *
     * @param virtualProperty           virtual property
     * @param targetValue               targetValue
     */
    void removeFromPropertyValue(VirtualProperty virtualProperty,Object targetValue) {
        getPropertiesFromTracker().removePropertyValue(virtualProperty,targetValue);
    }
    /**
     * Adds a property value to this instance.
     *
     * @param virtualProperty           virtual property
     * @param virtualSourceInstance     virtual source instance
     */
    void addToPropertyValue(VirtualProperty virtualProperty,VirtualInstance virtualSourceInstance) {
        getPropertiesToTracker().addPropertyValue(virtualProperty,virtualSourceInstance);
    }
    /**
     * Removes a property value to this instance.
     *
     * @param virtualProperty           virtual property
     * @param virtualSourceInstance     virtual source instance
     */
    void removeToPropertyValue(VirtualProperty virtualProperty,VirtualInstance virtualSourceInstance) {
        getPropertiesToTracker().removePropertyValue(virtualProperty,virtualSourceInstance);
    }
    /**
     * Sets the flag that determines whether this entity is in the model.
     *
     * @param isInOIModel               flag determining whether this entity is in the model
     * @param sourceOIModel             the source OI-model
     */
    void setIsInOIModel(boolean isInOIModel,OIModel sourceOIModel) {
        super.setIsInOIModel(isInOIModel,sourceOIModel);
        if (!isInOIModel) {
            getParentConceptsTracker().clear();
            getPropertiesFromTracker().clear();
            getPropertiesToTracker().clear();
        }
    }

    /**
     * Class that tracks the properties to and from an instance. It records changes to the map of properties, and synthesizes the
     * map only when the actual map is needed. This enables making changes to the virtual ontology more efficiently.
     */
    protected abstract static class PropertyValueTracker {
        /** The actual map being tracked. */
        protected Map m_map;
        /** The back-log of changes. */
        protected List m_backLog;

        /**
         * Returns the original map. It is doesn't exist, the map is loaded.
         */
        public Map getMap() throws KAONException {
            if (m_map==null) {
                m_map=loadMap();
                if (m_backLog!=null) {
                    Iterator iterator=m_backLog.iterator();
                    while (iterator.hasNext()) {
                        PropertyValueChange change=(PropertyValueChange)iterator.next();
                        switch (change.m_changeType) {
                        case PropertyValueChange.ADD:
                            addPropertyValue(change.m_property,change.m_value);
                            break;
                        case PropertyValueChange.REMOVE:
                            removePropertyValue(change.m_property,change.m_value);
                            break;
                        case PropertyValueChange.SET:
                            setPropertyValue(change.m_property,change.m_value);
                            break;
                        }
                    }
                    m_backLog=null;
                }
            }
            return m_map;
        }
        /**
         * Clears this tracker.
         */
        public void clear() {
            m_map=new HashMap();
            m_backLog=null;
        }
        /**
         * Returns the backlog.
         *
         * @return                                  the backlog
         */
        protected List getBackLog() {
            if (m_backLog==null)
                m_backLog=new ArrayList();
            return m_backLog;
        }
        /**
         * Adds a property value to the map.
         *
         * @param property                          property
         * @param value                             the value
         */
        public void addPropertyValue(Property property,Object value) {
            if (m_map!=null) {
                Set set=(Set)m_map.get(property);
                if (set==null) {
                    set=new HashSet();
                    m_map.put(property,set);
                }
                set.add(value);
            }
            else
                getBackLog().add(new PropertyValueChange(PropertyValueChange.ADD,property,value));
        }
        /**
         * Removes a property value from the map.
         *
         * @param property                          property
         * @param value                             the value
         */
        public void removePropertyValue(Property property,Object value) {
            if (m_map!=null) {
                Set set=(Set)m_map.get(property);
                if (set!=null) {
                    set.remove(value);
                    if (set.isEmpty())
                        m_map.remove(property);
                }
            }
            else
                getBackLog().add(new PropertyValueChange(PropertyValueChange.REMOVE,property,value));
        }
        /**
         * Sets a property value int the map.
         *
         * @param property                          property
         * @param value                             the value
         */
        public void setPropertyValue(Property property,Object value) {
            if (m_map!=null) {
                Set set=(Set)m_map.get(property);
                if (set==null) {
                    set=new HashSet();
                    m_map.put(property,set);
                }
                set.clear();
                set.add(value);
            }
            else
                getBackLog().add(new PropertyValueChange(PropertyValueChange.SET,property,value));
        }
        /**
         * Loads the original map.
         *
         * @return                              the original map
         */
        protected abstract Map loadMap() throws KAONException;

        /**
         * Represents a change applied to the original map.
         */
        protected static class PropertyValueChange {
            public static final int ADD=0;
            public static final int REMOVE=1;
            public static final int SET=2;

            public int m_changeType;
            public Property m_property;
            public Object m_value;
            public int m_hashCode;

            public PropertyValueChange(int changeType,Property property,Object value) {
                m_changeType=changeType;
                m_property=property;
                m_value=value;
                m_hashCode=property.hashCode()*7+value.hashCode();
            }
            public int hashCode() {
                return m_hashCode;
            }
            public boolean equals(Object that) {
                if (this==that)
                    return true;
                if (!(that instanceof PropertyValueChange))
                    return false;
                PropertyValueChange thatPropertyValueChange=(PropertyValueChange)that;
                return m_property.equals(thatPropertyValueChange.m_property) && m_value.equals(thatPropertyValueChange.m_value);
            }
        }
    }
}
