package edu.unika.aifb.kaon.virtualoimodel;

import java.util.Set;
import java.util.HashSet;
import java.util.Map;
import java.util.HashMap;
import java.util.Iterator;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.vocabulary.*;

/**
 * Virtual entity in an ontology.
 *
 * @author Ljiljana Stojanovic (Ljiljana.Stojanovic@fzi.de)
 */
public abstract class VirtualEntity implements Entity {
    /** Virtual OI-model. */
    protected VirtualOIModel m_oimodel;
    /** Original entity. */
    protected Entity m_entity;
    /** Flag specifying whether this entity is in OI-model. */
    protected boolean m_isInOIModel;
    /** Source original OI-model. */
    protected OIModel m_sourceOIModelOriginal;

    /**
     * Creates an instance of this class and attaches it to the virtual OI-model.
     *
     * @param oimodel                       virtual OI-model of this entity
     */
    public VirtualEntity(VirtualOIModel oimodel) {
        m_oimodel=oimodel;
    }
    /**
     * Initializes this instance. This method is used in cases when one knows whether an entity really is in OI-model,
     * to avoid a call that may be expensive.
     *
     * @param oimodel                       virtual OI-model of this entity
     * @param entity                        original entity
     * @param isInOIModel                   determines whether this entity is in OI-model
     */
    void initialize(Entity entity,boolean isInOIModel) {
        m_entity=entity;
        m_isInOIModel=isInOIModel;
    }
    /**
     * Returns the bit mask specifying which parts of the object have been loaded.
     *
     * @return                              the bit mask specifying which parts of the object have been loaded
     */
    public int getLoadedState() throws KAONException {
        return m_entity.getLoadedState();
    }
    /**
     * Erases the cached state of the object. The next time something is requested, it will be fetched from the storage.
     */
    public void unload() throws KAONException {
        m_entity.unload();
    }
    /**
     * Returns the URI of this property.
     *
     * @return                              URI of this property
     */
    public String getURI() throws KAONException {
        return m_entity.getURI();
    }
    /**
     * Retruns the version of this entity.
     *
     * @return                          the version of the entity
     */
    public int getVersion() throws KAONException {
        if (isInOIModel())
            return m_entity.getVersion();
        else
            return Integer.MAX_VALUE;
    }
    /**
     * Returns the original entity.
     *
     * @return                              the original entity
     */
    public Entity getEntity() {
        return m_entity;
    }
    /**
     * Returns an OI-model of this entity.
     *
     * @return                          OI-model of this entity
     */
    public OIModel getOIModel() {
        return m_oimodel;
    }
    /**
     * Tests whether this entity exists as part of its associated OI-model.
     *
     * @return                          <code>true</code> if entity is a member in the OI-model
     */
    public boolean isInOIModel() {
        return m_isInOIModel;
    }
    /**
     * Returns <code>true</code> if this entity has been declared in the OI-model as returned
     * by <code>getOIModel()</code> call. This method always returns <code>true</code> for virtual OI-model
     * entities, because the virtual OI-model (virtually) copies all entities.
     *
     * @return                          <code>true</code> if this entity case declared in the OI-model as returned by <code>getOIModel()</code> call
     */
    public boolean isDeclaredLocally() {
        return true;
    }
    /**
     * Returns the OI-model where this entity was declared originally. This method returns always
     * the same OI-model as <code>getOIModel()</code>, since virtual OI-model (virtually) copies all entities.
     *
     * @return                          the OI-model there this entity was declared originally
     */
    public OIModel getSourceOIModel() {
        if (isInOIModel())
            return m_oimodel;
        else
            return null;
    }
    /**
     * Returns the original OI-model where this entity was declared. For the difference of the method above,
     * this returns the original (not virtual) OI-model.
     *
     * @return                          the original OI-model where this entity was declared originally
     */
    public OIModel getSourceOIModelOriginal() throws KAONException {
        if (m_isInOIModel && m_sourceOIModelOriginal==null)
            m_sourceOIModelOriginal=getEntity().getSourceOIModel();
        return m_sourceOIModelOriginal;
    }
    /**
     * Returns the spanning instance of this entity.
     *
     * @return                          spanning instance of this entiry
     */
    public Instance getSpanningInstance() throws KAONException {
        return m_oimodel.getInstance(m_entity.getSpanningInstance());
    }
    /**
     * Returns all lexical entries associated with this entity.
     *
     * @return                              set of lexical entries associated with this entity
     */
    public Set getLexicalEntries() throws KAONException {
        Property references=m_oimodel.getProperty(KAONVocabularyAdaptor.INSTANCE.getReferences());
        Instance instance=getSpanningInstance();
        return instance.getToPropertyValues(references);
    }
    /**
     * Retruns all lexical entries of specified type associated with this entity.
     *
     * @param typeURI                       URI of the lexical entry type
     * @return                              set of lexical entries of given type
     */
    public Set getLexicalEntries(String typeURI) throws KAONException {
        Property references=m_oimodel.getProperty(KAONVocabularyAdaptor.INSTANCE.getReferences());
        Instance instance=getSpanningInstance();
        Set set=new HashSet(instance.getToPropertyValues(references));
        Iterator iterator=set.iterator();
        while (iterator.hasNext()) {
            LexicalEntry lexicalEntry=(LexicalEntry)iterator.next();
            if (!lexicalEntry.getTypeURI().equals(typeURI))
                iterator.remove();
        }
        return set;
    }
    /**
     * Retruns all lexical entries of specified type associated with this entity.
     *
     * @param typeURI                       URI of the lexical entry type
     * @param languageURI                   URI of the language
     * @return                              set of lexical entries of given type
     */
    public Set getLexicalEntries(String typeURI,String languageURI) throws KAONException {
        Property references=m_oimodel.getProperty(KAONVocabularyAdaptor.INSTANCE.getReferences());
        Instance instance=getSpanningInstance();
        Set set=new HashSet(instance.getToPropertyValues(references));
        Iterator iterator=set.iterator();
        while (iterator.hasNext()) {
            LexicalEntry lexicalEntry=(LexicalEntry)iterator.next();
            if (!lexicalEntry.getTypeURI().equals(typeURI) || !lexicalEntry.getInLanguage().equals(languageURI))
                iterator.remove();
        }
        return set;
    }
    /**
     * Retruns a lexical entry of specified type and in given language associated with this entity. If there are more
     * lexical entries that match the criteria, one is picked and returned. If there are no lexical entries, <code>null</code>
     * is returned.
     *
     * @param typeURI                       URI of the lexical entry type
     * @param languageURI                   URI of the language
     * @return                              lexical entries of given type and language, or <code>null</code> if no entry is found
     */
    public LexicalEntry getLexicalEntry(String typeURI,String languageURI) throws KAONException {
        Set set=getLexicalEntries(typeURI);
        Iterator iterator=set.iterator();
        while (iterator.hasNext()) {
            LexicalEntry lexicalEntry=(LexicalEntry)iterator.next();
            if (languageURI.equals(lexicalEntry.getInLanguage()))
                return lexicalEntry;
        }
        return null;
    }
    /**
     * Returns the value of the lexical entry of given type in given language with given attribute name. If no appropriate
     * information is found, <code>null</code> is returned.
     *
     * @param typeURI                       URI of the lexical entry type
     * @param languageURI                   URI of the language
     * @param attributeURI                  URI name of the attribute that is requested
     * @return                              value of given attribute
     */
    public String getLexicalAttribute(String typeURI,String languageURI,String attributeURI) throws KAONException {
        LexicalEntry lexicalEntry=getLexicalEntry(typeURI,languageURI);
        if (lexicalEntry==null)
            return null;
        else {
            Object value=lexicalEntry.getAttribute(attributeURI);
            if (value instanceof String)
                return (String)value;
            else
                return null;
        }
    }
    /**
     * Returns the label in given language.
     *
     * @param languageURI                   URI of the language
     * @return                              label in given language
     */
    public String getLabel(String languageURI) throws KAONException {
        return getLexicalAttribute(KAONVocabularyAdaptor.INSTANCE.getKAONLabel(),languageURI,KAONVocabularyAdaptor.INSTANCE.getValue());
    }
    /**
     * Converts a set of original concepts into a set of virtual concepts.
     *
     * @param input                         set of original concepts
     * @param isInOIModel                   determines whether concepts are in OI-model
     * @return                              set of virtual concepts
     */
    protected Set getVirtualConcepts(Set input,boolean isInOIModel) throws KAONException {
        Set output=new HashSet();
        Iterator iterator=input.iterator();
        while (iterator.hasNext()) {
            Concept concept=(Concept)iterator.next();
            output.add(m_oimodel.getConcept(concept,isInOIModel));
        }
        return output;
    }
    /**
     * Converts a set of original concepts into a map of virtual concepts (values are <code>null</code>).
     *
     * @param input                         set of original concepts
     * @param isInOIModel                   determines whether concepts are in OI-model
     * @return                              map of virtual concepts
     */
    protected Map getVirtualConceptsMap(Set input,boolean isInOIModel) throws KAONException {
        Map output=new HashMap();
        Iterator iterator=input.iterator();
        while (iterator.hasNext()) {
            Concept concept=(Concept)iterator.next();
            output.put(m_oimodel.getConcept(concept,isInOIModel),null);
        }
        return output;
    }
    /**
     * Converts a set of original properties into a set of virtual properties.
     *
     * @param input                         set of original properties
     * @param isInOIModel                   determines whether concepts are in OI-model
     * @return                              set of virtual properties
     */
    protected Set getVirtualProperties(Set input,boolean isInOIModel) throws KAONException {
        Set output=new HashSet();
        Iterator iterator=input.iterator();
        while (iterator.hasNext()) {
            Property property=(Property)iterator.next();
            output.add(m_oimodel.getProperty(property,isInOIModel));
        }
        return output;
    }
    /**
     * Converts a set of original properties into a map of virtual properties (value are <code>null</code>).
     *
     * @param input                         set of original properties
     * @param isInOIModel                   determines whether concepts are in OI-model
     * @return                              set of virtual properties
     */
    protected Map getVirtualPropertiesMap(Set input,boolean isInOIModel) throws KAONException {
        Map output=new HashMap();
        Iterator iterator=input.iterator();
        while (iterator.hasNext()) {
            Property property=(Property)iterator.next();
            output.put(m_oimodel.getProperty(property,isInOIModel),null);
        }
        return output;
    }
    /**
     * Converts a set of original instances into a set of virtual instances.
     *
     * @param input                         set of original instances
     * @param isInOIModel                   determines whether concepts are in OI-model
     * @return                              set of virtual instances
     */
    protected Set getVirtualInstances(Set input,boolean isInOIModel) throws KAONException {
        Set output=new HashSet();
        Iterator iterator=input.iterator();
        while (iterator.hasNext()) {
            Instance instance=(Instance)iterator.next();
            output.add(m_oimodel.getInstance(instance,isInOIModel));
        }
        return output;
    }
    /**
     * Converts a map of original property values into a map of virtual property values.
     *
     * @param input                         map of original property values
     * @param isInOIModel                   determined whether instances are in OI-model
     * @return                              map of virtual property values
     */
    protected Map getVirtualPropertyValues(Map input,boolean isInOIModel) throws KAONException {
        Map output=new HashMap();
        Iterator iterator=input.keySet().iterator();
        while (iterator.hasNext()) {
            Property property=(Property)iterator.next();
            VirtualProperty virtualProperty=m_oimodel.getProperty(property);
            Set result=new HashSet();
            Iterator values=((Set)input.get(property)).iterator();
            while (values.hasNext()) {
                Object value=values.next();
                Object virtualValue;
                if (value instanceof Instance)
                    virtualValue=m_oimodel.getInstance((Instance)value,isInOIModel);
                else
                    virtualValue=value;
                result.add(virtualValue);
            }
            output.put(virtualProperty,result);
        }
        return output;
    }
    /**
     * Converts a Set of original property instances into a map of virtual property instances.
     *
     * @param input                         set of original property instances
     * @param isInOIModel                   determines whether property instances are in the OI-model
     * @return                              set of virtual property instances
     */
    protected Set getVirtualPropertyInstances(Set input,boolean isInOIModel) throws KAONException {
        Set output=new HashSet();
        Iterator iterator=input.iterator();
        while (iterator.hasNext()) {
            PropertyInstance propertyInstance=(PropertyInstance)iterator.next();
            output.add(m_oimodel.getPropertyInstance(propertyInstance,isInOIModel));
        }
        return output;
    }
    /**
     * Sets the flag that determines whether this entity is in the model.
     *
     * @param isInOIModel               flag determining whether this entity is in the model
     * @param sourceOIModel             the source OI-model
     */
    void setIsInOIModel(boolean isInOIModel,OIModel sourceOIModel) {
        m_isInOIModel=isInOIModel;
        if (m_isInOIModel)
            m_sourceOIModelOriginal=sourceOIModel;
        else
            m_sourceOIModelOriginal=null;
    }

    /**
     * Tracks changes to a set. On demand, this class syntehsizes the set and applies the changes.
     */
    protected abstract static class SetTracker {
        /** The actual set. */
        protected Set m_set;
        /** Back-log of additions to the set. */
        protected Set m_addBackLog;
        /** Back-log of removals from the set. */
        protected Set m_removeBackLog;

        /**
         * Adds an element to this set.
         *
         * @param object                            object being added
         */
        public void add(Object object) {
            if (m_set!=null)
                m_set.add(object);
            else {
                if (m_removeBackLog!=null)
                    m_removeBackLog.remove(object);
                if (m_addBackLog==null)
                    m_addBackLog=new HashSet();
                m_addBackLog.add(object);
            }
        }
        /**
         * Removes an element from this set.
         *
         * @param object                            object being removed
         */
        public void remove(Object object) {
            if (m_set!=null)
                m_set.remove(object);
            else {
                if (m_addBackLog!=null)
                    m_addBackLog.remove(object);
                if (m_removeBackLog==null)
                    m_removeBackLog=new HashSet();
                m_removeBackLog.add(object);
            }
        }
        /**
         * Returns the actual set. If the set is not loaded, the set is loaded and all back-logged changes are applied to it.
         *
         * @return                                  actual set
         */
        protected Set getSet() throws KAONException {
            if (m_set==null) {
                m_set=loadSet();
                if (m_addBackLog!=null)
                    m_set.addAll(m_addBackLog);
                if (m_removeBackLog!=null)
                    m_set.removeAll(m_removeBackLog);
                m_addBackLog=null;
                m_removeBackLog=null;
            }
            return m_set;
        }
        /**
         * Clears this tracker.
         */
        public void clear() {
            m_set=new HashSet();
            m_addBackLog=null;
            m_removeBackLog=null;
        }
        /**
         * Must be provided by the instantiator to retrieve the original set.
         *
         * @return                                  the original set
         */
        protected abstract Set loadSet() throws KAONException;
    }

    /**
     * Tracks changes to a map. On demand, this class syntehsizes the map and applies the changes.
     */
    protected static abstract class MapTracker {
        /** The actual map. */
        protected Map m_map;
        /** Back-log of additions to the set. */
        protected Map m_addBackLog;
        /** Back-log of removals from the set. */
        protected Set m_removeBackLog;

        /**
         * Adds an element to this set.
         *
         * @param object                            object being added
         * @param value                             the value
         */
        public void add(Object object,Object value) {
            if (m_map!=null)
                m_map.put(object,value);
            else {
                if (m_removeBackLog!=null)
                    m_removeBackLog.remove(object);
                if (m_addBackLog==null)
                    m_addBackLog=new HashMap();
                m_addBackLog.put(object,value);
            }
        }
        /**
         * Removes an element from this set.
         *
         * @param object                            object being removed
         */
        public void remove(Object object) {
            if (m_map!=null)
                m_map.remove(object);
            else {
                if (m_addBackLog!=null)
                    m_addBackLog.remove(object);
                if (m_removeBackLog==null)
                    m_removeBackLog=new HashSet();
                m_removeBackLog.add(object);
            }
        }
        /**
         * Returns the actual map. If the map is not loaded, the map is loaded and all back-logged changes are applied to it.
         *
         * @return                                  actual set
         */
        protected Map getMap() throws KAONException {
            if (m_map==null) {
                m_map=loadMap();
                if (m_addBackLog!=null) {
                    Iterator iterator=m_addBackLog.keySet().iterator();
                    while (iterator.hasNext()) {
                        Object object=iterator.next();
                        m_map.put(object,m_addBackLog.get(object));
                    }
                }
                if (m_removeBackLog!=null) {
                    Iterator iterator=m_removeBackLog.iterator();
                    while (iterator.hasNext()) {
                        Object object=iterator.next();
                        m_map.remove(object);
                    }
                }
                m_addBackLog=null;
                m_removeBackLog=null;
            }
            return m_map;
        }
        /**
         * Returns the key set.
         *
         * @return                                  the key set
         */
        public Set keySet() throws KAONException {
            return getMap().keySet();
        }
        /**
         * Clears this tracker.
         */
        public void clear() {
            m_map=new HashMap();
            m_addBackLog=null;
            m_removeBackLog=null;
        }
        /**
         * Returns the value for given key.
         *
         * @param key                               the key for which the value is read
         * @return                                  the value for this key
         */
        public Object getValue(Object key) throws KAONException {
            if (m_addBackLog!=null && m_addBackLog.containsKey(key))
                return m_addBackLog.get(key);
            if (m_removeBackLog!=null && m_removeBackLog.contains(key))
                return null;
            Map map=getMap();
            Object value=map.get(key);
            if (value==null && map.containsKey(key)) {
                value=getMissingValue(key);
                map.put(key,value);
            }
            return value;
        }
        /**
         * Must be provided by the instantiator to provide value that is missing for given key.
         *
         * @param key                               the key for which the value is to be provided
         * @return                                  the value that is missing for given key
         */
        protected abstract Object getMissingValue(Object key) throws KAONException;
        /**
         * Must be provided by the instantiator to retrieve the original map.
         *
         * @return                                  the original map
         */
        protected abstract Map loadMap() throws KAONException;
    }
}
