package edu.unika.aifb.kaon.virtualoimodel;

import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.HashSet;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

/**
 * Implementation of the VirtualConcept.
 *
 * @author Ljiljana Stojanovic (Ljiljana.Stojanovic@fzi.de)
 */
public class VirtualConcept extends VirtualEntity implements Concept {
    /** Map tracker of direct superconcepts. */
    protected MapTracker m_superConcepts;
    /** Set tracker of direct subconcepts. */
    protected SetTracker m_subConcepts;
    /** Set tracker of property instances from this concept. */
    protected SetTracker m_propertiesFrom;
    /** Set tracker of property instances to this concept. */
    protected SetTracker m_propertiesTo;
    /** Set tracker of all direct instances of this concept. */
    protected SetTracker m_instances;

    /**
     * Creates an instance of this class and attaches it to the virtual OI-model.
     *
     * @param oimodel                       virtual OI-model
     */
    public VirtualConcept(VirtualOIModel oimodel) {
        super(oimodel);
    }
    /**
     * Returns the original cncept for the virtual concept.
     *
     * @return                          the original concept for this concept
     */
    public Concept getConcept() {
        return (Concept)m_entity;
    }
    /**
     * Returns the tracker for properties from concept.
     *
     * @return                          the tracker for properties from concept
     */
    public SetTracker getPropertiesFromConceptTracker() {
        if (m_propertiesFrom==null)
            m_propertiesFrom=new SetTracker() {
                protected Set loadSet() throws KAONException {
                    return getVirtualProperties(getConcept().getPropertiesFromConcept(),true);
                }
            };
        return m_propertiesFrom;
    }
    /**
     * Returns all virtual properties starting from this concept.
     *
     * @return                          set of properties starting from this virtual concept (may be empty)
     */
    public Set getPropertiesFromConcept() throws KAONException {
        return getPropertiesFromConceptTracker().getSet();
    }
    /**
     * Returns all virtual properties (including inherited ones) starting from this concept.
     *
     * @return                          set of all properties starting from this concept (may be empty)
     */
    public Set getAllPropertiesFromConcept() throws KAONException {
        // if there are hierarcy cycles, this method will loop forever
        Set result=new HashSet(getPropertiesFromConcept());
        Iterator superConcepts=getSuperConcepts().iterator();
        while (superConcepts.hasNext()) {
            Concept concept=(Concept)superConcepts.next();
            result.addAll(concept.getAllPropertiesFromConcept());
        }
        return result;
    }
    /**
     * Returns the tracker for all virtual properties pointing to this concept.
     *
     * @return                          set of properties pointing to this virtual concept (may be empty)
     */
    public SetTracker getPropertiesToConceptTracker() {
        if (m_propertiesTo==null)
            m_propertiesTo=new SetTracker() {
                protected Set loadSet() throws KAONException {
                    return getVirtualProperties(getConcept().getPropertiesToConcept(),true);
                }
            };
        return m_propertiesTo;
    }
    /**
     * Returns all virtual properties pointing to this concept.
     *
     * @return                          set of properties pointing to this virtual concept (may be empty)
     */
    public Set getPropertiesToConcept() throws KAONException {
        return getPropertiesToConceptTracker().getSet();
    }
    /**
     * Returns all properties (including inherited ones) pointing to this concept.
     *
     * @return                          set of all properties pointing to this concept (may be empty)
     */
    public Set getAllPropertiesToConcept() throws KAONException {
        // if there are hierarcy cycles, this method will loop forever
        Set result=new HashSet(getPropertiesToConcept());
        Iterator superConcepts=getSuperConcepts().iterator();
        while (superConcepts.hasNext()) {
            Concept concept=(Concept)superConcepts.next();
            result.addAll(concept.getAllPropertiesToConcept());
        }
        return result;
    }
    /**
     * Returns the tracker for direct subconcepts of this concept.
     *
     * @return                          subconcepts of this concept
     */
    public SetTracker getSubConceptsTracker() {
        if (m_subConcepts==null)
            m_subConcepts=new SetTracker() {
                protected Set loadSet() throws KAONException {
                    return getVirtualConcepts(getConcept().getSubConcepts(),true);
                }
            };
        return m_subConcepts;
    }
    /**
     * Returns direct subconcepts of this concept.
     *
     * @return                          subconcepts of this concept
     */
    public Set getSubConcepts() throws KAONException {
        return getSubConceptsTracker().getSet();
    }
    /**
     * Returns all subconcepts (inherited as well) of this concept.
     *
     * @return                          subconcepts of this concept
     */
    public Set getAllSubConcepts() throws KAONException {
        // if there are hierarcy cycles, this method will loop forever
        Set subConcepts=getSubConcepts();
        Set result=new HashSet(subConcepts);
        Iterator iterator=subConcepts.iterator();
        while (iterator.hasNext()) {
            Concept concept=(Concept)iterator.next();
            result.addAll(concept.getAllSubConcepts());
        }
        return result;
    }
    /**
     * Tests if this concept is a direct subconcept of given concept. If concepts are the same, this method returns <code>false</code>.
     *
     * @param potentialSuperConcept     concept being tested
     * @return                          <code>true</code> if this concept is a direct subconcept
     */
    public boolean isDirectSubConceptOf(Concept potentialSuperConcept) throws KAONException {
        return getSuperConcepts().contains(m_oimodel.getConcept(potentialSuperConcept));
    }
    /**
     * Returns the OI-model containing the this concept - subconcept association.
     *
     * @param superConcept              the superconcept of this concept
     * @return                          the OI-model in which the super-subconcept association is declared
     */
    public OIModel getSuperSubConceptOIModel(Concept superConcept) throws KAONException {
        if (isDirectSubConceptOf(superConcept))
            return m_oimodel;
        else
            return null;
    }
    /**
     * Returns the original OI-model containing the this concept - subconcept association.
     *
     * @param superConcept              the superconcept of this concept
     * @return                          the original OI-model in which the super-subconcept association is declared
     */
    public OIModel getSuperSubConceptOIModelOriginal(VirtualConcept superConcept) throws KAONException {
        return (OIModel)getSuperConceptsTracker().getValue(superConcept);
    }
    /**
     * Returns <code>true</code> if this concept has been declared a subconcept of given concept in this OI-model.
     *
     * @param superConcept              the superconcept of this concept
     * @return                          <code>true</code> if supplied concept has been declared a subconcept of this concept in this OI-model
     */
    public boolean isSuperSubConceptDeclaredLocally(Concept superConcept) throws KAONException {
        return isDirectSubConceptOf(superConcept);
    }
    /**
     * Tests if this concept is a subconcept of given concept. If concepts are the same, this method returns <code>false</code>.
     *
     * @param potentialSuperConcept     concept being tested
     * @return                          <code>true</code> if given concept is a subconcept
     */
    public boolean isSubConceptOf(Concept potentialSuperConcept) throws KAONException {
        return getAllSuperConcepts().contains(m_oimodel.getConcept(potentialSuperConcept));
    }
    /**
     * Returns the tracker for direct superconcepts of this concept.
     *
     * @return                          superconcepts of this concept
     */
    public MapTracker getSuperConceptsTracker() {
        if (m_superConcepts==null)
            m_superConcepts=new MapTracker() {
                protected Object getMissingValue(Object key) throws KAONException {
                    return getConcept().getSuperSubConceptOIModel(((VirtualConcept)key).getConcept());
                }
                protected Map loadMap() throws KAONException {
                    return getVirtualConceptsMap(getConcept().getSuperConcepts(),true);
                }
            };
        return m_superConcepts;
    }
    /**
     * Returns direct superconcepts of this concept.
     *
     * @return                          superconcepts of this concept
     */
    public Set getSuperConcepts() throws KAONException {
        return getSuperConceptsTracker().keySet();
    }
    /**
     * Returns all superconcepts (inherited as well) of this concept.
     *
     * @return                          superconcepts of this concept
     */
    public Set getAllSuperConcepts() throws KAONException {
        // if there are hierarcy cycles, this method will loop forever
        Set superConcepts=getSuperConcepts();
        Set result=new HashSet(superConcepts);
        Iterator iterator=superConcepts.iterator();
        while (iterator.hasNext()) {
            Concept concept=(Concept)iterator.next();
            result.addAll(concept.getAllSuperConcepts());
        }
        return result;
    }
    /**
     * Returns the set trecker for set of direct instances of this concept.
     *
     * @return                          set of instances of this concept
     */
    public SetTracker getInstancesTracker() {
        if (m_instances==null)
            m_instances=new SetTracker() {
                protected Set loadSet() throws KAONException {
                    return getVirtualInstances(getConcept().getInstances(),true);
                }
            };
        return m_instances;
    }
    /**
     * Returns the set of direct instances of this concept.
     *
     * @return                          set of instances of this concept
     */
    public Set getInstances() throws KAONException {
        return getInstancesTracker().getSet();
    }
    /**
     * Returns the set of all instances (even of subconcepts) of a given concept in the pool.
     *
     * @return                          set of all instances of this concept
     */
    public Set getAllInstances() throws KAONException {
        // if there are hierarcy cycles, this method will loop forever
        Set result=new HashSet(getInstances());
        Set subConcepts=getSubConcepts();
        Iterator iterator=subConcepts.iterator();
        while (iterator.hasNext()) {
            Concept subConcept=(Concept)iterator.next();
            result.addAll(subConcept.getAllInstances());
        }
        return result;
    }
    /**
     * Accepts an entity visitor.
     *
     * @param visitor                   visitor to apply
     */
    public void accept(EntityVisitor visitor) throws KAONException {
        visitor.visit(this);
    }
    /**
     * Sets the flag that determines whether this entity is in the model.
     *
     * @param isInOIModel               flag determining whether this entity is in the model
     * @param sourceOIModel             the source OI-model
     */
    void setIsInOIModel(boolean isInOIModel,OIModel sourceOIModel) {
        super.setIsInOIModel(isInOIModel,sourceOIModel);
        if (!isInOIModel) {
            getSuperConceptsTracker().clear();
            getSubConceptsTracker().clear();
            getPropertiesFromConceptTracker().clear();
            getPropertiesToConceptTracker().clear();
            getInstancesTracker().clear();
        }
    }
}
