package edu.unika.aifb.kaon.virtualoimodel;

import java.util.Iterator;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.oimodel.*;

/**
 * Visitor of ontology change events.
 *
 * @author Ljiljana Stojanovic (Ljiljana.Stojanovic@fzi.de)
 */
public class VirtualApplyChangeVisitor implements ChangeVisitor {
    /** OI-model. */
    protected VirtualOIModel m_oimodel;
    /** The current event. */
    protected ChangeEvent m_currentEvent;
    /** Visitor for adding. */
    protected EntityVisitor m_addVisitor;
    /** Visitor for removing. */
    protected EntityVisitor m_removeVisitor;

    /**
     * Creates an instance of this class.
     *
     * @param oimodel               virtual OI-model
     */
    public VirtualApplyChangeVisitor(VirtualOIModel oimodel) {
        m_oimodel=oimodel;
        m_addVisitor=new VirtualAddEntityVisitor();
        m_removeVisitor=new VirtualRemoveEntityVisitor();
    }
    /**
     * Visits AddEntity change.
     */
    public void visit(AddEntity event) throws KAONException {
        m_currentEvent=event;
        try {
            event.getEntity().accept(m_addVisitor);
        }
        finally {
            m_currentEvent=null;
        }
    }
    /**
     * Visits RemoveEntity change.
     */
    public void visit(RemoveEntity event) throws KAONException {
        m_currentEvent=event;
        try {
            event.getEntity().accept(m_removeVisitor);
        }
        finally {
            m_currentEvent=null;
        }
    }
    /**
     * Visits an event for creation of subconcepts.
     */
    public void visit(AddSubConcept event) throws KAONException {
        VirtualConcept superConcept=m_oimodel.getConcept(event.getSuperConcept());
        VirtualConcept subConcept=m_oimodel.getConcept(event.getSubConcept());
        superConcept.getSubConceptsTracker().add(subConcept);
        subConcept.getSuperConceptsTracker().add(superConcept,event.getOIModel());
    }
    /**
     * Visits an event for removal of subconcepts.
     */
    public void visit(RemoveSubConcept event) throws KAONException {
        VirtualConcept superConcept=m_oimodel.getConcept(event.getSuperConcept());
        VirtualConcept subConcept=m_oimodel.getConcept(event.getSubConcept());
        superConcept.getSubConceptsTracker().remove(subConcept);
        subConcept.getSuperConceptsTracker().remove(superConcept);
    }
    /**
     * Visits an event for adding a domain to the property.
     */
    public void visit(AddPropertyDomain event) throws KAONException {
        VirtualProperty virtualProperty=m_oimodel.getProperty(event.getProperty());
        VirtualConcept virtualConcept=m_oimodel.getConcept(event.getConcept());
        virtualProperty.addDomainConcept(virtualConcept,event.getOIModel());
        virtualConcept.getPropertiesFromConceptTracker().add(virtualProperty);
    }
    /**
     * Visits an event for removing a domain from the property.
     */
    public void visit(RemovePropertyDomain event) throws KAONException {
        VirtualProperty virtualProperty=m_oimodel.getProperty(event.getProperty());
        VirtualConcept virtualConcept=m_oimodel.getConcept(event.getConcept());
        virtualProperty.removeDomainConcept(virtualConcept);
        virtualConcept.getPropertiesFromConceptTracker().remove(virtualProperty);
    }
    /**
     * Visits an event for adding a range to the property.
     */
    public void visit(AddPropertyRange event) throws KAONException {
        VirtualProperty virtualProperty=m_oimodel.getProperty(event.getProperty());
        VirtualConcept virtualConcept=m_oimodel.getConcept(event.getConcept());
        virtualProperty.getRangeConceptsTracker().add(virtualConcept,event.getOIModel());
        virtualConcept.getPropertiesToConceptTracker().add(virtualProperty);
    }
    /**
     * Visits an event for removing a range from the property.
     */
    public void visit(RemovePropertyRange event) throws KAONException {
        VirtualProperty virtualProperty=m_oimodel.getProperty(event.getProperty());
        VirtualConcept virtualConcept=m_oimodel.getConcept(event.getConcept());
        virtualProperty.getRangeConceptsTracker().remove(virtualConcept);
        virtualConcept.getPropertiesToConceptTracker().remove(virtualProperty);
    }
    /**
     * Visits an event for determinig whether property is attirubte.
     */
    public void visit(SetPropertyIsAttribute event) throws KAONException {
        VirtualProperty virtualProperty=m_oimodel.getProperty(event.getProperty());
        virtualProperty.setIsAttribute(event.getIsAttribute());
    }
    /**
     * Visits an event for setting the minimum cardinality of a property for a concept.
     */
    public void visit(SetMinimumCardinality event) throws KAONException {
        VirtualProperty virtualProperty=m_oimodel.getProperty(event.getProperty());
        VirtualConcept virtualConcept=m_oimodel.getConcept(event.getConcept());
        virtualProperty.setMinimumCardinality(virtualConcept,event.getCardinality());
    }
    /**
     * Visits an event for setting the maximum cardinality of a property for a concept.
     */
    public void visit(SetMaximumCardinality event) throws KAONException {
        VirtualProperty virtualProperty=m_oimodel.getProperty(event.getProperty());
        VirtualConcept virtualConcept=m_oimodel.getConcept(event.getConcept());
        virtualProperty.setMaximumCardinality(virtualConcept,event.getCardinality());
    }
    /**
     * Visits an event for creation of subproperties.
     */
    public void visit(AddSubProperty event) throws KAONException {
        VirtualProperty superProperty=m_oimodel.getProperty(event.getSuperProperty());
        VirtualProperty subProperty=m_oimodel.getProperty(event.getSubProperty());
        superProperty.getSubPropertiesTracker().add(subProperty);
        subProperty.getSuperPropertiesTracker().add(superProperty,event.getOIModel());
    }
    /**
     * Visits an event for removal of subproperties.
     */
    public void visit(RemoveSubProperty event) throws KAONException {
        VirtualProperty superProperty=m_oimodel.getProperty(event.getSuperProperty());
        VirtualProperty subProperty=m_oimodel.getProperty(event.getSubProperty());
        superProperty.getSubPropertiesTracker().remove(subProperty);
        subProperty.getSuperPropertiesTracker().remove(superProperty);
    }
    /**
     * Visits an event for setting the inverse relationship between properties.
     */
    public void visit(SetInverseProperties event) throws KAONException {
        VirtualProperty property1=m_oimodel.getProperty(event.getProperty1());
        VirtualProperty property2=m_oimodel.getProperty(event.getProperty2());
        property1.setInverseProperty(property2,event.getOIModel());
        property2.setInverseProperty(property1,event.getOIModel());
    }
    /**
     * Visits an event for removing an inverse relationship between properties.
     */
    public void visit(SetNoInverseProperties event) throws KAONException {
        VirtualProperty property1=m_oimodel.getProperty(event.getProperty1());
        VirtualProperty property2=m_oimodel.getProperty(event.getProperty2());
        property1.setInverseProperty(null,null);
        property2.setInverseProperty(null,null);
    }
    /**
     * Visits an event for setting the symmetry flag of the property.
     */
    public void visit(SetPropertySymmetric event) throws KAONException {
        VirtualProperty property=m_oimodel.getProperty(event.getProperty());
        property.setSymmetric(event.getSymmetric());
    }
    /**
     * Visits an event for setting the transitivity flag of the property.
     */
    public void visit(SetPropertyTransitive event) throws KAONException {
        VirtualProperty property=m_oimodel.getProperty(event.getProperty());
        property.setTransitive(event.getTransitive());
    }
    /**
     * Visits an event for making an instance a subinstance of given concept.
     */
    public void visit(AddInstanceOf event) throws KAONException {
        VirtualConcept concept=m_oimodel.getConcept(event.getConcept());
        VirtualInstance instance=m_oimodel.getInstance(event.getInstance());
        concept.getInstancesTracker().add(instance);
        instance.getParentConceptsTracker().add(concept,event.getOIModel());
    }
    /**
     * Visits an event for making an instance not a subinstance of given concept.
     */
    public void visit(RemoveInstanceOf event) throws KAONException {
        VirtualConcept concept=m_oimodel.getConcept(event.getConcept());
        VirtualInstance instance=m_oimodel.getInstance(event.getInstance());
        concept.getInstancesTracker().remove(instance);
        instance.getParentConceptsTracker().remove(concept);
    }
    /**
     * Visits an event for adding a property instance.
     */
    public void visit(AddPropertyInstance event) throws KAONException {
        VirtualPropertyInstance propertyInstance=m_oimodel.getPropertyInstance(event.getPropertyInstance());
        propertyInstance.setIsInOIModel(true);
        VirtualProperty property=(VirtualProperty)propertyInstance.getProperty();
        VirtualInstance sourceInstance=(VirtualInstance)propertyInstance.getSourceInstance();
        Object targetValue=propertyInstance.getTargetValue();
        property.getPropertyInstancesTracker().add(propertyInstance);
        sourceInstance.addFromPropertyValue(property,targetValue);
        if (targetValue instanceof VirtualInstance)
            ((VirtualInstance)targetValue).addToPropertyValue(property,sourceInstance);
    }
    /**
     * Visits an event for removing a property instance.
     */
    public void visit(RemovePropertyInstance event) throws KAONException {
        VirtualPropertyInstance propertyInstance=m_oimodel.getPropertyInstance(event.getPropertyInstance());
        propertyInstance.setIsInOIModel(false);
        VirtualProperty property=(VirtualProperty)propertyInstance.getProperty();
        VirtualInstance sourceInstance=(VirtualInstance)propertyInstance.getSourceInstance();
        Object targetValue=propertyInstance.getTargetValue();
        property.getPropertyInstancesTracker().remove(propertyInstance);
        sourceInstance.removeFromPropertyValue(property,targetValue);
        if (targetValue instanceof VirtualInstance)
            ((VirtualInstance)targetValue).removeToPropertyValue(property,sourceInstance);
    }
    /**
     * Visits an event for settomg the value of a property instance.
     */
    public void visit(SetPropertyInstanceValue event) throws KAONException {
        VirtualPropertyInstance propertyInstance=m_oimodel.getPropertyInstance(event.getPropertyInstance());
        propertyInstance.setIsInOIModel(true);
        VirtualProperty property=(VirtualProperty)propertyInstance.getProperty();
        VirtualInstance sourceInstance=(VirtualInstance)propertyInstance.getSourceInstance();
        Object targetValue=propertyInstance.getTargetValue();
        Iterator values=sourceInstance.getFromPropertyValues(property).iterator();
        while (values.hasNext()) {
            Object toValue=values.next();
            VirtualPropertyInstance existingPropertyInstance=(VirtualPropertyInstance)m_oimodel.getPropertyInstance(property,sourceInstance,toValue);
            existingPropertyInstance.setIsInOIModel(false);
            property.getPropertyInstancesTracker().remove(existingPropertyInstance);
            if (toValue instanceof VirtualInstance)
                ((VirtualInstance)toValue).removeToPropertyValue(property,sourceInstance);
        }
        property.getPropertyInstancesTracker().add(propertyInstance);
        sourceInstance.setFromPropertyValue(property,targetValue);
        if (targetValue instanceof VirtualInstance)
            ((VirtualInstance)targetValue).addToPropertyValue(property,sourceInstance);
    }
    /**
     * Visits an event for changing the value of a property instance.
     */
    public void visit(ChangePropertyInstanceValue event) throws KAONException {
        VirtualPropertyInstance propertyInstance=m_oimodel.getPropertyInstance(event.getPropertyInstance());
        propertyInstance.setIsInOIModel(false);
        VirtualProperty property=(VirtualProperty)propertyInstance.getProperty();
        VirtualInstance sourceInstance=(VirtualInstance)propertyInstance.getSourceInstance();
        Object targetValue=propertyInstance.getTargetValue();
        property.getPropertyInstancesTracker().remove(propertyInstance);
        sourceInstance.removeFromPropertyValue(property,targetValue);
        if (targetValue instanceof VirtualInstance)
            ((VirtualInstance)targetValue).removeToPropertyValue(property,sourceInstance);
        VirtualPropertyInstance newPropertyInstance=(VirtualPropertyInstance)m_oimodel.getPropertyInstance(event.getPropertyInstance().getProperty(),event.getPropertyInstance().getSourceInstance(),event.getNewTargetValue());
        newPropertyInstance.setIsInOIModel(true);
        Object newTargetValue=newPropertyInstance.getTargetValue();
        property.getPropertyInstancesTracker().add(newPropertyInstance);
        sourceInstance.addFromPropertyValue(property,newTargetValue);
        if (newTargetValue instanceof VirtualInstance)
            ((VirtualInstance)newTargetValue).addToPropertyValue(property,sourceInstance);
    }
    /**
     * Visits an event for making a model included in another model.
     */
    public void visit(AddIncludedOIModel event) throws KAONException {
        throw new KAONException("Virtual OI-model doesn't support adding included OI-models.");
    }
    /**
     * Visits an event for making a model not included in another model.
     */
    public void visit(RemoveIncludedOIModel event) throws KAONException {
        throw new KAONException("Virtual OI-model doesn't support removing included OI-models.");
    }

    /**
     * Visitor that adds elements to the virtual ontology.
     *
     * @author Ljiljana Stojanovic (Ljiljana.Stojanovic@fzi.de)
     */
    protected class VirtualAddEntityVisitor implements EntityVisitor {
        /**
         * Visits a concept.
         */
        public void visit(Concept entity) throws KAONException {
            VirtualConcept virtualEntity=m_oimodel.getConcept(entity);
            virtualEntity.setIsInOIModel(true,m_currentEvent.getOIModel());
            visit(entity.getSpanningInstance());
        }
        /**
         * Visits a propety.
         */
        public void visit(Property entity) throws KAONException {
            VirtualProperty virtualEntity=m_oimodel.getProperty(entity);
            virtualEntity.setIsInOIModel(true,m_currentEvent.getOIModel());
            visit(entity.getSpanningInstance());
        }
        /**
         * Visits an instance.
         */
        public void visit(Instance entity) throws KAONException {
            VirtualInstance virtualEntity=m_oimodel.getInstance(entity);
            virtualEntity.setIsInOIModel(true,m_currentEvent.getOIModel());
            VirtualConcept root=(VirtualConcept)m_oimodel.getRootConcept();
            root.getInstancesTracker().add(virtualEntity);
            virtualEntity.getParentConceptsTracker().add(root,m_currentEvent.getOIModel());
        }
        /**
         * Visits a lexical entry.
         */
        public void visit(LexicalEntry entity) throws KAONException {
            visit((Instance)entity);
        }
    }

    /**
     * Visitor that removes elements to the virtual ontology.
     *
     * @author Ljiljana Stojanovic (Ljiljana.Stojanovic@fzi.de)
     */
    protected class VirtualRemoveEntityVisitor implements EntityVisitor {
        /**
         * Visits a concept.
         */
        public void visit(Concept entity) throws KAONException {
            VirtualConcept virtualEntity=m_oimodel.getConcept(entity);
            virtualEntity.setIsInOIModel(false,null);
            visit(entity.getSpanningInstance());
        }
        /**
         * Visits a propety.
         */
        public void visit(Property entity) throws KAONException {
            VirtualProperty virtualEntity=m_oimodel.getProperty(entity);
            virtualEntity.setIsInOIModel(false,null);
            visit(entity.getSpanningInstance());
        }
        /**
         * Visits an instance.
         */
        public void visit(Instance entity) throws KAONException {
            VirtualInstance virtualEntity=m_oimodel.getInstance(entity);
            virtualEntity.setIsInOIModel(false,null);
        }
        /**
         * Visits a lexical entry.
         */
        public void visit(LexicalEntry entity) throws KAONException {
            visit((Instance)entity);
        }
    }
}
