package edu.unika.aifb.kaon.evolutionlog;

import java.util.List;
import java.util.LinkedList;
import java.util.Collections;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.util.*;

/**
 * A transient implementation of the evolution log.
 *
 * @author Ljiljana Stojanovic (Ljiljana.Stojanovic@fzi.de)
 */
public class TransientEvolutionLog extends EvolutionLog {
    /** Stores the batches of changes that have been applied to the ontology. */
    protected List m_appliedChanges;
    /** Stores the batches of changes that have been undone. */
    protected List m_undoneChanges;

    /**
     * Creates an instance of this class.
     *
     * @param kaonConnectionEvolutionLogs       the manager of evolution logs
     * @param oimodel                           the original OI-model
     */
    public TransientEvolutionLog(KAONConnectionEvolutionLogs kaonConnectionEvolutionLogs,OIModel oimodel) {
        super(kaonConnectionEvolutionLogs,oimodel);
        m_appliedChanges=new LinkedList();
        m_undoneChanges=new LinkedList();
    }
    /**
     * Returns <code>true</code> if this log is persistent.
     *
     * @return                                  <code>true</code> if this log is persistent
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized boolean isPersistent() {
        return false;
    }
    /**
     * Closes this evolution log.
     *
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized void close() throws KAONException {
        if (m_kaonConnectionEvolutionLogs!=null) {
            m_appliedChanges=null;
            m_undoneChanges=null;
            super.close();
        }
    }
    /**
     * Read changes of given version.
     *
     * @param version                           the version whose changes are requested
     * @return                                  the list of performed changes
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized List readChanges(int version) {
        return Collections.EMPTY_LIST;
    }
    /**
     * Returns <code>true</code> if undo can be performed.
     *
     * @return                                  <code>true</code> if undo can be performed
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized boolean canUndoChanges() {
        return !m_appliedChanges.isEmpty();
    }
    /**
     * Returns the list of changes that must be applied to the model to undo the last change batch.
     *
     * @return                                  the list of reversed changes
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized List readUndoChanges() throws KAONException {
        if (m_appliedChanges.isEmpty())
            throw new KAONException("There are no changes to be undone.");
        List list=(List)m_appliedChanges.get(0);
        return ChangeReverse.reverseChanges(list);
    }
    /**
     * Returns <code>true</code> if redo can be performed.
     *
     * @return                              <code>true</code> if redo can be performed
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized boolean canRedoChanges() {
        return !m_undoneChanges.isEmpty();
    }
    /**
     * Returns the list of changes that must be applied to the model to redo the last undone change batch.
     *
     * @return                                  the list of redoed changes
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized List readRedoChanges() throws KAONException {
        if (m_undoneChanges.isEmpty())
            throw new KAONException("There are no changes to be redoed.");
        return (List)m_undoneChanges.get(0);
    }
    /**
     * Creates a logger.
     *
     * @param loggingType                       the type of logging
     * @return                                  the logger
     */
    public synchronized Logger createLogger(int loggingType) {
        return new TransientLogger(loggingType);
    }

    /**
     * The logger for transient log.
     */
    protected class TransientLogger implements Logger {
        protected int m_loggingType;
        protected List m_changeList;

        public TransientLogger(int loggingType) {
            m_loggingType=loggingType;
            m_changeList=new LinkedList();
        }
        public void logChange(ChangeEvent changeEvent) {
            m_changeList.add(changeEvent);
        }
        public void finishChangeBatch() throws KAONException {
            switch (m_loggingType) {
            case LOG_NORMAL:
                m_appliedChanges.add(0,m_changeList);
                m_undoneChanges.clear();
                break;
            case LOG_UNDO:
                if (m_appliedChanges.isEmpty())
                    throw new KAONException("Cannot undo changes.");
                else {
                    List list=(List)m_appliedChanges.remove(0);
                    m_undoneChanges.add(0,list);
                }
                break;
            case LOG_REDO:
                if (m_undoneChanges.isEmpty())
                    throw new KAONException("Cannot redo changes.");
                else {
                    List list=(List)m_undoneChanges.remove(0);
                    m_appliedChanges.add(0,list);
                }
                break;
            }
        }
    }
}
