package edu.unika.aifb.kaon.evolutionlog;

import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;
import java.util.Set;
import java.util.Map;
import java.util.HashMap;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;

/**
 * A logger for the OIModelEvolutionLog.
 *
 * @author Ljiljana Stojanovic (Ljiljana.Stojanovic@fzi.de)
 */
public class OIModelLogger implements EvolutionLog.Logger,ChangeVisitor {
    /** The namespace of the evolution log. */
    public static final String EVOLUTIONNS=KAONConnection.EVOLUTION_OIMODEL_URI+"#";

    /** OI-model evolution log. */
    protected OIModelEvolutionLog m_oimodelEvolutionLog;
    /** The type of logging. */
    protected int m_loggingType;
    /** List of original events. */
    protected List m_originalEvents;
    /** List of changes. */
    protected List m_changes;
    /** Map for previous changes. */
    protected Map m_mapOfChanges;
    /** The LOG instance. */
    protected Instance m_log;

    /**
     * Creates an instance of this class.
     *
     * @param oimodelEvolutionLog           the evolution log for this logger
     * @param loggingType                   the type of logger
     */
    public OIModelLogger(OIModelEvolutionLog oimodelEvolutionLog,int loggingType) throws KAONException {
        m_oimodelEvolutionLog=oimodelEvolutionLog;
        m_loggingType=loggingType;
        m_mapOfChanges=new HashMap();
        m_changes=new LinkedList();
        m_originalEvents=new LinkedList();
        loadLog();
    }
    /**
     * Logs one change.
     *
     * @param changeEvent                   the event to be logged
     * @throws KAONException                thrown if there is an error
     */
    public void logChange(ChangeEvent changeEvent) throws KAONException {
        changeEvent.accept(this);
    }
    /**
     * Finishes writing a batch of changes.
     *
     * @throws KAONException                    thrown if there is an error
     */
    public void finishChangeBatch() throws KAONException {
        Instance previousEventInstance=getLastChange();
        String versionNumber=String.valueOf(m_oimodelEvolutionLog.getKAONConnectionEvolutionLogs().getOIModelVersion(m_oimodelEvolutionLog.getDomainOIModel()));
        boolean first=true;
        Iterator iterator=m_originalEvents.iterator();
        while (iterator.hasNext()) {
            ChangeEvent changeEvent=(ChangeEvent)iterator.next();
            Instance eventInstance=(Instance)m_mapOfChanges.get(changeEvent);
            ChangeEvent causeChangeEvent=changeEvent.getCause();
            if (causeChangeEvent!=null) {
                Instance causeInstance=(Instance)m_mapOfChanges.get(causeChangeEvent);
                m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"causesChange"),causeInstance,eventInstance));
            }
            if (first) {
                m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"firstChangeInAGroup"),eventInstance,"true"));
                first=false;
            }
            if (previousEventInstance!=null){
                m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_previousChange"),eventInstance,previousEventInstance));
                m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_previousHistoryChange"),eventInstance,previousEventInstance));
            }
            m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"version"),eventInstance,versionNumber));
            previousEventInstance=eventInstance;
        }

        if (m_loggingType==EvolutionLog.LOG_NORMAL)
            setLastChange(previousEventInstance);
        else if (m_loggingType==EvolutionLog.LOG_UNDO)
            setUndoPointers(getLastChange());
        else
            setRedoPointers(getRedoListHead());
        setLastHistoryChange(previousEventInstance);

        getEvolutionOIModelInstance().applyChanges(m_changes);
        m_changes.clear();
        m_mapOfChanges.clear();
        m_originalEvents.clear();
    }
    protected OIModel getEvolutionOIModelInstance() {
        return m_oimodelEvolutionLog.getEvolutionOIModelInstance();
    }
    protected void loadLog() throws KAONException {
        getEvolutionOIModelInstance().refresh();
        Concept logConcept=getConcept(OIModelEvolutionLog.EVOLUTIONNS+"LOG");
        Set set=logConcept.getInstances();
        if (!set.isEmpty())
            m_log=(Instance)set.iterator().next();
        else
            try {
                m_log=getInstance(getEvolutionOIModelInstance().createNewURI());
                m_changes.add(new AddEntity(m_log));
                m_changes.add(new AddInstanceOf(getConcept(EVOLUTIONNS+"LOG"),m_log));
                getEvolutionOIModelInstance().applyChanges(m_changes);
            }
            finally {
                m_changes.clear();
            }
    }
    protected Concept getConcept(String uri) throws KAONException {
        return getEvolutionOIModelInstance().getConcept(uri);
    }
    protected Property getProperty(String uri) throws KAONException {
        return getEvolutionOIModelInstance().getProperty(uri);
    }
    protected Instance getInstance(String uri) throws KAONException {
        return getEvolutionOIModelInstance().getInstance(uri);
    }
    protected Instance getLastChange() throws KAONException {
        return (Instance)m_log.getFromPropertyValue(getProperty(EVOLUTIONNS+"lastChange"));
    }
    protected Instance getRedoListHead() throws KAONException {
        return (Instance)m_log.getFromPropertyValue(getProperty(EVOLUTIONNS+"redoListHead"));
    }
    protected void setLastChange(Instance currentChange) throws KAONException {
        if (currentChange!=null) {
            m_changes.add(new SetPropertyInstanceValue(getProperty(EVOLUTIONNS+"lastChange"),m_log,currentChange));
            if (getRedoListHead()!=null) {
                //remove redo list
                m_changes.addAll(removeRedoList(getRedoListHead()));
                m_changes.add(new RemovePropertyInstance(getProperty(EVOLUTIONNS+"redoListHead"),m_log,getRedoListHead()));
            }
        }
    }
    protected List removeRedoList(Instance instance) throws KAONException {
        List redoChanges=new LinkedList();
        Property propertyNext=getProperty(EVOLUTIONNS+"redoListNext");
        Property propertyChange=getProperty(EVOLUTIONNS+"listItemChange");
        while (instance!=null) {
            redoChanges.add(0,new RemoveEntity(instance));
            redoChanges.add(new RemoveInstanceOf(getConcept(EVOLUTIONNS+"RedoListItem"),instance));
            Instance instanceChange=(Instance)instance.getFromPropertyValue(propertyChange);
            redoChanges.add(0,new RemovePropertyInstance(propertyChange,instance,instanceChange));
            Instance instanceNext=(Instance)instance.getFromPropertyValue(propertyNext);
            if (instanceNext!=null)
                redoChanges.add(0,new RemovePropertyInstance(propertyNext,instance,instanceNext));
            instance=instanceNext;
        }
        return redoChanges;
    }
    protected void setLastHistoryChange(Instance currentChange) throws KAONException {
        if (currentChange!=null)
            m_changes.add(new SetPropertyInstanceValue(getProperty(EVOLUTIONNS+"lastHistoryChange"),m_log,currentChange));
    }
    protected void setUndoPointers(Instance lastChangeInstance) throws KAONException {
        if (lastChangeInstance==null)
            throw new KAONException("There is no previous change");
        Instance newLastChangeInstance = getFirstChangeInPreviousGroup(lastChangeInstance);

        // adapt lastChangeID
        if (newLastChangeInstance==null)
            m_changes.add(new RemovePropertyInstance(getProperty(EVOLUTIONNS+"lastChange"),m_log,lastChangeInstance));
        else
            m_changes.add(new SetPropertyInstanceValue(getProperty(EVOLUTIONNS+"lastChange"),m_log,newLastChangeInstance));
        // adapt redoList - add head
        Instance redoItem=getInstance(m_oimodelEvolutionLog.m_evolutionOIModelInstance.createNewURI());
        m_changes.add(new AddEntity(redoItem));
        m_changes.add(new AddInstanceOf(getConcept(EVOLUTIONNS+"RedoListItem"),redoItem));
        m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"listItemChange"),redoItem,lastChangeInstance));
        if (getRedoListHead()==null)
            m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"redoListHead"),m_log,redoItem));
        else {
            m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"redoListNext"),redoItem,getRedoListHead()));
            m_changes.add(new SetPropertyInstanceValue(getProperty(EVOLUTIONNS+"redoListHead"),m_log,redoItem));
        }
    }
   protected void setRedoPointers(Instance lastRedoItem) throws KAONException {
        if (lastRedoItem==null)
            throw new KAONException("Redo list is empty");
        Instance redoChange=(Instance)lastRedoItem.getFromPropertyValue(getProperty(EVOLUTIONNS+"listItemChange"));
        if (redoChange==null)
            throw new KAONException("Error in redo list");
        // adapt lastChangeID
        m_changes.add(new SetPropertyInstanceValue(getProperty(EVOLUTIONNS+"lastChange"),m_log,redoChange));
        //adapt redoList - remove head
        Instance nextRedoItem=(Instance)lastRedoItem.getFromPropertyValue(getProperty(EVOLUTIONNS+"redoListNext"));
        if (nextRedoItem==null)
            m_changes.add(new RemovePropertyInstance(getProperty(EVOLUTIONNS+"redoListHead"),m_log,lastRedoItem));
        else {
            m_changes.add(new RemovePropertyInstance(getProperty(EVOLUTIONNS+"redoListNext"),lastRedoItem,nextRedoItem));
            m_changes.add(new SetPropertyInstanceValue(getProperty(EVOLUTIONNS+"redoListHead"),m_log,nextRedoItem));
        }
        m_changes.add(new RemovePropertyInstance(getProperty(EVOLUTIONNS+"listItemChange"),lastRedoItem,redoChange));
        m_changes.add(new RemoveInstanceOf(getConcept(EVOLUTIONNS+"RedoListItem"),lastRedoItem));
        m_changes.add(new RemoveEntity(lastRedoItem));
   }
    protected Instance getFirstChangeInPreviousGroup(Instance startChange) throws KAONException {
        //find first change in a previous change group
        Property propertyPreviousChange=getProperty(EVOLUTIONNS+"has_previousChange");
        Property propertyFirstChangeInAGroup=getProperty(EVOLUTIONNS+"firstChangeInAGroup");
        Instance previousChange=startChange;
        boolean found=false;
        while (!found) {
            if (previousChange==null)
                throw new KAONException("Error in finding previous changes");
            String value=(String)previousChange.getFromPropertyValue(propertyFirstChangeInAGroup);
            if (value!=null && value.equals("true"))
                found=true;
            previousChange=(Instance)previousChange.getFromPropertyValue(propertyPreviousChange);
        }
        return previousChange;
    }
    protected Instance createEventInstance(ChangeEvent event,Concept eventConcept) throws KAONException {
        m_originalEvents.add(event);
        Instance eventInstance=getInstance(getEvolutionOIModelInstance().createNewURI());
        m_changes.add(new AddEntity(eventInstance));
        m_changes.add(new AddInstanceOf(eventConcept,eventInstance));
        m_changes.add(new AddPropertyInstance(getProperty(OIModelEvolutionLog.EVOLUTIONNS+"inOIModel"),eventInstance,event.getOIModel().getLogicalURI()));
        m_mapOfChanges.put(event,eventInstance);
        return eventInstance;
    }
    /**
     * Visits AddEntity change.
     */
    public void visit(AddEntity event) throws KAONException {
        Instance eventInstance=createEventInstance(event,getConcept(EVOLUTIONNS+"AddEntity"));
        m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+findEntityType(event.getEntity())),eventInstance,event.getEntity().getURI()));
    }
    /**
     * Visits RemoveEntity change.
     */
    public void visit(RemoveEntity event) throws KAONException {
          Instance eventInstance=createEventInstance(event,getConcept(EVOLUTIONNS+"RemoveEntity"));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+findEntityType(event.getEntity())),eventInstance,event.getEntity().getURI()));
    }
    /**
     * Returns type of the entity - Instance,Property,Concept
     */
    protected String findEntityType(Entity entity) {
          String typeOfEntity;
          if (entity instanceof Instance)
              typeOfEntity="has_referenceInstance";
          else if (entity instanceof Property)
              typeOfEntity="has_referenceProperty";
          else if (entity instanceof Concept)
              typeOfEntity="has_referenceConcept";
          else
              typeOfEntity="has_referenceEntity";
          return typeOfEntity;
    }
    /**
     * Visits an event for creation of subconcepts.
     */
    public void visit(AddSubConcept event) throws KAONException {
          Instance eventInstance=createEventInstance(event,getConcept(EVOLUTIONNS+"AddSubConcept"));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceSuperConcept"),eventInstance,event.getSuperConcept().getURI()));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceSubConcept"),eventInstance,event.getSubConcept().getURI()));
    }
    /**
     * Visits an event for removal of subconcepts.
     */
    public void visit(RemoveSubConcept event) throws KAONException {
          Instance eventInstance=createEventInstance(event,getConcept(EVOLUTIONNS+"RemoveSubConcept"));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceSuperConcept"),eventInstance,event.getSuperConcept().getURI()));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceSubConcept"),eventInstance,event.getSubConcept().getURI()));
    }
    /**
     * Visits an event for adding a domain to the property.
     */
    public void visit(AddPropertyDomain event) throws KAONException {
          Instance eventInstance=createEventInstance(event,getConcept(EVOLUTIONNS+"AddPropertyDomain"));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceConcept"),eventInstance,event.getConcept().getURI()));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceProperty"),eventInstance,event.getProperty().getURI()));
    }
    /**
     * Visits an event for removing a domain from the property.
     */
    public void visit(RemovePropertyDomain event) throws KAONException {
          Instance eventInstance=createEventInstance(event,getConcept(EVOLUTIONNS+"RemovePropertyDomain"));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceConcept"),eventInstance,event.getConcept().getURI()));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceProperty"),eventInstance,event.getProperty().getURI()));
    }
    /**
     * Visits an event for adding a range to the property.
     */
    public void visit(AddPropertyRange event) throws KAONException {
          Instance eventInstance=createEventInstance(event,getConcept(EVOLUTIONNS+"AddPropertyRange"));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceConcept"),eventInstance,event.getConcept().getURI()));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceProperty"),eventInstance,event.getProperty().getURI()));
    }
    /**
     * Visits an event for removing a range from the property.
     */
    public void visit(RemovePropertyRange event) throws KAONException {
          Instance eventInstance=createEventInstance(event,getConcept(EVOLUTIONNS+"RemovePropertyRange"));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceConcept"),eventInstance,event.getConcept().getURI()));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceProperty"),eventInstance,event.getProperty().getURI()));
    }
    /**
     * Visits an event for determinig whether property is attirubte.
     */
    public void visit(SetPropertyIsAttribute event) throws KAONException {
          Instance eventInstance=createEventInstance(event,getConcept(EVOLUTIONNS+"SetPropertyIsAttribute"));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceProperty"),eventInstance,event.getProperty().getURI()));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_value"),eventInstance,String.valueOf(event.getIsAttribute())));
    }
    /**
     * Visits an event for setting the minimum cardinality of a property for a concept.
     */
    public void visit(SetMinimumCardinality event) throws KAONException {
        throw new KAONException("Not yet supported.");
    }
    /**
     * Visits an event for setting the maximum cardinality of a property for a concept.
     */
    public void visit(SetMaximumCardinality event) throws KAONException {
        throw new KAONException("Not yet supported.");
    }
    /**
     * Visits an event for creation of subproperties.
     */
    public void visit(AddSubProperty event) throws KAONException {
          Instance eventInstance=createEventInstance(event,getConcept(EVOLUTIONNS+"AddSubProperty"));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceSuperProperty"),eventInstance,event.getSuperProperty().getURI()));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceSubProperty"),eventInstance,event.getSubProperty().getURI()));
    }
    /**
     * Visits an event for removal of subproperties.
     */
    public void visit(RemoveSubProperty event) throws KAONException {
          Instance eventInstance=createEventInstance(event,getConcept(EVOLUTIONNS+"RemoveSubProperty"));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceSuperProperty"),eventInstance,event.getSuperProperty().getURI()));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceSubProperty"),eventInstance,event.getSubProperty().getURI()));
    }
    /**
     * Visits an event for setting the inverse relationship between properties.
     */
    public void visit(SetInverseProperties event) throws KAONException {
          Instance eventInstance=createEventInstance(event,getConcept(EVOLUTIONNS+"SetInverseProperties"));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceProperty"),eventInstance,event.getProperty1().getURI()));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceProperty"),eventInstance,event.getProperty2().getURI()));
    }
    /**
     * Visits an event for removing an inverse relationship from a property.
     */
    public void visit(SetNoInverseProperties event) throws KAONException {
          Instance eventInstance=createEventInstance(event,getConcept(EVOLUTIONNS+"SetNoInverseProperties"));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceProperty"),eventInstance,event.getProperty1().getURI()));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceProperty"),eventInstance,event.getProperty2().getURI()));
    }
    /**
     * Visits an event for setting the symmetry flag of the property.
     */
    public void visit(SetPropertySymmetric event) throws KAONException {
          Instance eventInstance=createEventInstance(event,getConcept(EVOLUTIONNS+"SetPropertySymmetric"));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceProperty"),eventInstance,event.getProperty().getURI()));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_value"),eventInstance,String.valueOf(event.getSymmetric())));
    }
    /**
     * Visits an event for setting the transitivity flag of the property.
     */
    public void visit(SetPropertyTransitive event) throws KAONException {
          Instance eventInstance=createEventInstance(event,getConcept(EVOLUTIONNS+"SetPropertyTransitive"));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceProperty"),eventInstance,event.getProperty().getURI()));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_value"),eventInstance,String.valueOf(event.getTransitive())));
    }
    /**
     * Visits an event for making an instance a subinstance of given concept.
     */
    public void visit(AddInstanceOf event) throws KAONException {
          Instance eventInstance=createEventInstance(event,getConcept(EVOLUTIONNS+"AddInstanceOf"));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceConcept"),eventInstance,event.getConcept().getURI()));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceInstance"),eventInstance,event.getInstance().getURI()));
    }
    /**
     * Visits an event for making an instance not a subinstance of given concept.
     */
    public void visit(RemoveInstanceOf event) throws KAONException {
          Instance eventInstance=createEventInstance(event,getConcept(EVOLUTIONNS+"RemoveInstanceOf"));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceConcept"),eventInstance,event.getConcept().getURI()));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceInstance"),eventInstance,event.getInstance().getURI()));
    }
    /**
     * Visits an event for adding a property instance.
     */
    public void visit(AddPropertyInstance event) throws KAONException {
          Instance eventInstance=createEventInstance(event,getConcept(EVOLUTIONNS+"AddPropertyInstance"));
          PropertyInstance propertyInstance=event.getPropertyInstance();
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceSourceInstance"),eventInstance,propertyInstance.getSourceInstance().getURI()));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceProperty"),eventInstance,propertyInstance.getProperty().getURI()));
          if (propertyInstance.getTargetValue() instanceof Instance)
              m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceTargetInstance"),eventInstance,((Instance)propertyInstance.getTargetValue()).getURI()));
          else
              m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceTargetObject"),eventInstance,propertyInstance.getTargetValue()));
    }
    /**
     * Visits an event for removing a property instance.
     */
    public void visit(RemovePropertyInstance event) throws KAONException {
          Instance eventInstance=createEventInstance(event,getConcept(EVOLUTIONNS+"RemovePropertyInstance"));
          PropertyInstance propertyInstance=event.getPropertyInstance();
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceSourceInstance"),eventInstance,propertyInstance.getSourceInstance().getURI()));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceProperty"),eventInstance,propertyInstance.getProperty().getURI()));
          if (propertyInstance.getTargetValue() instanceof Instance)
              m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceTargetInstance"),eventInstance,((Instance)propertyInstance.getTargetValue()).getURI()));
          else
              m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceTargetObject"),eventInstance,propertyInstance.getTargetValue()));
    }
    /**
     * Visits an event for setting the value of a property instance.
     */
    public void visit(SetPropertyInstanceValue event) throws KAONException {
          Instance eventInstance=createEventInstance(event,getConcept(EVOLUTIONNS+"SetPropertyInstanceValue"));
          PropertyInstance propertyInstance=event.getPropertyInstance();
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceSourceInstance"),eventInstance,propertyInstance.getSourceInstance().getURI()));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceProperty"),eventInstance,propertyInstance.getProperty().getURI()));
          if (propertyInstance.getTargetValue() instanceof Instance)
              m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceTargetInstance"),eventInstance,((Instance)propertyInstance.getTargetValue()).getURI()));
          else
              m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceTargetObject"),eventInstance,propertyInstance.getTargetValue()));
    }
    /**
     * Visits an event for changing the value of a property instance.
     */
    public void visit(ChangePropertyInstanceValue event) throws KAONException {
          Instance eventInstance=createEventInstance(event,getConcept(EVOLUTIONNS+"ChangePropertyInstanceValue"));
          PropertyInstance propertyInstance=event.getPropertyInstance();
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceSourceInstance"),eventInstance,propertyInstance.getSourceInstance().getURI()));
          m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_referenceProperty"),eventInstance,propertyInstance.getProperty().getURI()));
          if (propertyInstance.getTargetValue() instanceof Instance)
              m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_oldReferenceTargetInstance"),eventInstance,((Instance)propertyInstance.getTargetValue()).getURI()));
          else
              m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_oldReferenceTargetObject"),eventInstance,propertyInstance.getTargetValue()));
          Object newValue=event.getNewTargetValue();
          if ( newValue instanceof Instance)
              m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_newReferenceTargetInstance"),eventInstance,((Instance)newValue).getURI()));
          else
              m_changes.add(new AddPropertyInstance(getProperty(EVOLUTIONNS+"has_newReferenceTargetObject"),eventInstance, (String)newValue));
    }
    /**
     * Visits an event for making a model included in another model.
     */
    public void visit(AddIncludedOIModel event) throws KAONException {
        throw new KAONException("Not yet supported.");
    }
    /**
     * Visits an event for making a model not included in another model.
     */
    public void visit(RemoveIncludedOIModel event) throws KAONException {
        throw new KAONException("Not yet supported.");
    }
}
