package edu.unika.aifb.kaon.evolutionlog;

import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;
import java.util.Set;
import java.util.Map;
import java.util.HashMap;
import java.util.Collections;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.util.*;

/**
 * An instance of this class is capable of storing a sequence of OI-model changes.
 *
 * @author Ljiljana Stojanovic (Ljiljana.Stojanovic@fzi.de)
 */
public class OIModelEvolutionLog extends EvolutionLog {
    /** The namespace of the evolution log. */
    public static final String EVOLUTIONNS=KAONConnection.EVOLUTION_OIMODEL_URI+"#";

    /** OI-model for evolution. */
    protected OIModel m_evolutionOIModelInstance;
    /** Map for the dependency between instances and changes. */
    protected Map m_mapInstanceChange;
    /** List of causes-caused changes. */
    protected List m_causesCausedChanges;

    /**
     * Creates an instance of this class.
     *
     * @param kaonConnectionEvolutionLogs       the manager of evolution logs
     * @param oimodel                           the original OI-model
     * @param evolutionOIModelInstance          the OI-model that is the instance of the evolution OI-model
     * @throws KAONException                    thrown if there is an error
     */
    public OIModelEvolutionLog(KAONConnectionEvolutionLogs kaonConnectionEvolutionLogs,OIModel oimodel,OIModel evolutionOIModelInstance) throws KAONException {
        super(kaonConnectionEvolutionLogs,oimodel);
        if (evolutionOIModelInstance.getIncludedOIModel(KAONConnection.EVOLUTION_OIMODEL_URI)==null) {
            OIModel evolutionOIModel=evolutionOIModelInstance.getKAONConnection().openOIModelLogical(KAONConnection.EVOLUTION_OIMODEL_URI);
            evolutionOIModelInstance.applyChanges(Collections.singletonList(new AddIncludedOIModel(evolutionOIModel)));
        }
        m_evolutionOIModelInstance=evolutionOIModelInstance;
        m_mapInstanceChange=new HashMap();
        m_causesCausedChanges=new LinkedList();
    }
    /**
     * Returns <code>true</code> if this log is persistent.
     *
     * @return                                  <code>true</code> if this log is persistent
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized boolean isPersistent() {
        return true;
    }
    /**
     * Returns <code>true</code> if this log is owned by the domain OI-model for which it was opened.
     *
     * @return                                  <code>true</code> if this log is owned by the domain OI-model
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized boolean isOwnedByDomainOIModel() throws KAONException {
        return !DistributedEvolutionStrategy.isReplica(m_oimodel);
    }
    /**
     * Returns the instance of the evolution.
     *
     * @return                                  the OI-model that is the instance of the evolution OI-model
     */
    public synchronized OIModel getEvolutionOIModelInstance() {
        return m_evolutionOIModelInstance;
    }
    /**
     * Saves this evolution log.
     *
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized void save() throws KAONException {
        m_evolutionOIModelInstance.save();
    }
    /**
     * Closes this evolution log.
     *
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized void close() throws KAONException {
        if (m_kaonConnectionEvolutionLogs!=null)
            try {
                if (m_oimodel.getKAONConnection()!=m_evolutionOIModelInstance.getKAONConnection())
                    m_evolutionOIModelInstance.getKAONConnection().close();
            }
            finally {
                m_evolutionOIModelInstance=null;
                super.close();
            }
    }
    /**
     * Deletes the evolution log.
     *
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized void delete() throws KAONException {
        if (!isOwnedByDomainOIModel())
            throw new KAONException("Evolution log can't be deleted since it isn't owned by the domain OI-model.");
        try {
            m_oimodel.setAttribute(KAONConnectionEvolutionLogs.EVOLUTION_LOG_ATTIBUTE,null);
            m_evolutionOIModelInstance.delete();
        }
        finally {
            super.delete();
        }
    }
    /**
     * Refreshes the evolution log.
     *
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized void refresh() throws KAONException {
        m_evolutionOIModelInstance.refresh();
    }
    /**
     * Returns <code>true</code> if the log has unsaved changes.
     *
     * @return                                  <code>true</code> if the log has unsaved changes
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized boolean hasUnsavedChanges() throws KAONException {
        return m_evolutionOIModelInstance.hasUnsavedChanges();
    }
    /**
     * Read changes of given version.
     *
     * @param version                           the version whose changes are requested
     * @return                                  the list of performed changes
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized List readChanges(int version) throws KAONException {
        if (version<0)
            throw new KAONException("Error in reading changes");
        List changes=new LinkedList();
        Instance log=loadLog();
        Instance lastHistoryChangeInstance=getLastHistoryChange(log);
        if (lastHistoryChangeInstance!=null) {
            Property propertyPreviousChange=getProperty(EVOLUTIONNS+"has_previousHistoryChange");
            Property propertyVersion=getProperty(EVOLUTIONNS+"version");
            Instance newLastChangeInstance=lastHistoryChangeInstance;
            boolean found=false;
            while (!found && newLastChangeInstance!=null) {
                String value=(String)newLastChangeInstance.getFromPropertyValue(propertyVersion);
                if (value!=null) {
                    int valueNum=Integer.parseInt(value);
                    if (valueNum<=version)
                        found=true;
                }
                if (!found) {
                    changes.add(0,transfromInstanceEvent(newLastChangeInstance));
                    newLastChangeInstance=(Instance)newLastChangeInstance.getFromPropertyValue(propertyPreviousChange);
                    if (newLastChangeInstance==null && version==0)
                        found=true;
                }
            }
        }
        return improveListOfChange(changes);
    }
    /**
     * Returns <code>true</code> if undo can be performed.
     *
     * @return                                  <code>true</code> if undo can be performed
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized boolean canUndoChanges() throws KAONException {
        Instance log=loadLog();
        return getLastChange(log)!=null;
    }
    /**
     * Returns the list of changes that must be applied to the model to undo last change batch.
     *
     * @return                                  the list of reversed changes
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized List readUndoChanges() throws KAONException {
        Instance log=loadLog();
        Instance lastChangeInstance=getLastChange(log);
        if (lastChangeInstance==null)
            throw new KAONException("There is no previous change");
        //find first change in a previous change group
        Property propertyPreviousChange=getProperty(EVOLUTIONNS+"has_previousChange");
        Property propertyFirstChangeInAGroup=getProperty(EVOLUTIONNS+"firstChangeInAGroup");
        Instance newLastChangeInstance=lastChangeInstance;
        List undoChanges=new LinkedList();
        boolean found=false;
        while (!found) {
            if (newLastChangeInstance==null)
                throw new KAONException("Error in undo changes");
            String value=(String)newLastChangeInstance.getFromPropertyValue(propertyFirstChangeInAGroup);
            if (value!=null && value.equals("true"))
                found=true;
            undoChanges.add(0,transfromInstanceEvent(newLastChangeInstance));
            newLastChangeInstance=(Instance)newLastChangeInstance.getFromPropertyValue(propertyPreviousChange);
        }
        undoChanges=improveListOfChange(undoChanges);
        ChangeReverse changeReverse=new ChangeReverse();
        return changeReverse.getReversedChanges(undoChanges);
    }
    /**
     * Returns <code>true</code> if redo can be performed.
     *
     * @return                                  <code>true</code> if redo can be performed
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized boolean canRedoChanges() throws KAONException {
        Instance log=loadLog();
        return getRedoListHead(log)!=null;
    }
    /**
     * Returns the list of changes that must be applied to the model to redo the last undone change batch.
     *
     * @return                                  the list of redoed changes
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized List readRedoChanges() throws KAONException {
        Instance log=loadLog();
        Instance lastRedoItem=getRedoListHead(log);
        if (lastRedoItem==null)
            throw new KAONException("Redo list is empty");
        Instance redoChange=(Instance)lastRedoItem.getFromPropertyValue(getProperty(EVOLUTIONNS+"listItemChange"));
        if (redoChange==null)
            throw new KAONException("Error in redo list");
        //find first change in a previous change group
        Property propertyPreviousChange=getProperty(EVOLUTIONNS+"has_previousChange");
        Property propertyFirstChangeInAGroup=getProperty(EVOLUTIONNS+"firstChangeInAGroup");
        Instance nextRedoChange=redoChange;
        List redoChanges=new LinkedList();
        boolean found=false;
        while (!found) {
            if (nextRedoChange==null)
                throw new KAONException("Error in redo changes");
            String value=(String)nextRedoChange.getFromPropertyValue(propertyFirstChangeInAGroup);
            if (value!=null && value.equals("true"))
                found=true;
            redoChanges.add(0,transfromInstanceEvent(nextRedoChange));
            nextRedoChange=(Instance)nextRedoChange.getFromPropertyValue(propertyPreviousChange);
        }
        return improveListOfChange(redoChanges);
    }
    /**
     * Creates a logger.
     *
     * @param loggingType                       the type of logger
     * @return                                  the logger
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized Logger createLogger(int loggingType) throws KAONException {
        return new OIModelLogger(this,loggingType);
    }
    protected Instance loadLog() throws KAONException {
        m_evolutionOIModelInstance.refresh();
        Instance log;
        Concept logConcept=getConcept(OIModelEvolutionLog.EVOLUTIONNS+"LOG");
        Set set=logConcept.getInstances();
        if (!set.isEmpty())
            log=(Instance)set.iterator().next();
        else {
            List changes=new LinkedList();
            log=getInstance(m_evolutionOIModelInstance.createNewURI());
            changes.add(new AddEntity(log));
            changes.add(new AddInstanceOf(getConcept(EVOLUTIONNS+"LOG"),log));
            m_evolutionOIModelInstance.applyChanges(changes);
        }
        return log;
    }
    protected ChangeEvent transfromInstanceEvent(Instance instance) throws KAONException {
        ChangeEvent changeEvent;
        String logicalURI=(String)instance.getFromPropertyValue(getProperty(OIModelEvolutionLog.EVOLUTIONNS+"inOIModel"));
        OIModel oimodel=m_oimodel.getIncludedOIModel(logicalURI);
        if (instance.getParentConcepts().contains(getConcept(EVOLUTIONNS+"AddEntity"))) {
            Entity entity=getDomainEntity(instance);
            changeEvent=new AddEntity(oimodel,null,entity);
        }
        else if (instance.getParentConcepts().contains(getConcept(EVOLUTIONNS+"RemoveEntity"))) {
            Entity entity=getDomainEntity(instance);
            changeEvent=new RemoveEntity(oimodel,null,entity);
        }
        else if (instance.getParentConcepts().contains(getConcept(EVOLUTIONNS+"AddSubConcept"))) {
            Property propertySuperConcept=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceSuperConcept");
            Property propertySubConcept=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceSubConcept");
            changeEvent=new AddSubConcept(oimodel,null,getDomainConcept(instance,propertySuperConcept),getDomainConcept(instance,propertySubConcept));
        }
        else if (instance.getParentConcepts().contains(getConcept(EVOLUTIONNS+"RemoveSubConcept"))) {
            Property propertySuperConcept=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceSuperConcept");
            Property propertySubConcept=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceSubConcept");
            changeEvent=new RemoveSubConcept(oimodel,null,getDomainConcept(instance,propertySuperConcept),getDomainConcept(instance,propertySubConcept));
        }
        else if (instance.getParentConcepts().contains(getConcept(EVOLUTIONNS+"AddPropertyDomain"))) {
            Property propertyReferenceProperty=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceProperty");
            Property propertyReferenceConcept=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceConcept");
            changeEvent=new AddPropertyDomain(oimodel,null,getDomainProperty(instance,propertyReferenceProperty),getDomainConcept(instance,propertyReferenceConcept));
        }
        else if (instance.getParentConcepts().contains(getConcept(EVOLUTIONNS+"RemovePropertyDomain"))) {
            Property propertyReferenceProperty=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceProperty");
            Property propertyReferenceConcept=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceConcept");
            changeEvent=new RemovePropertyDomain(oimodel,null,getDomainProperty(instance,propertyReferenceProperty),getDomainConcept(instance,propertyReferenceConcept));
        }
        else if (instance.getParentConcepts().contains(getConcept(EVOLUTIONNS+"AddPropertyRange"))) {
            Property propertyReferenceProperty=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceProperty");
            Property propertyReferenceConcept=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceConcept");
            changeEvent=new AddPropertyRange(oimodel,null,getDomainProperty(instance,propertyReferenceProperty),getDomainConcept(instance,propertyReferenceConcept));
        }
        else if (instance.getParentConcepts().contains(getConcept(EVOLUTIONNS+"RemovePropertyRange"))) {
            Property propertyReferenceProperty=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceProperty");
            Property propertyReferenceConcept=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceConcept");
            changeEvent=new RemovePropertyRange(oimodel,null,getDomainProperty(instance,propertyReferenceProperty),getDomainConcept(instance,propertyReferenceConcept));
        }
        else if (instance.getParentConcepts().contains(getConcept(EVOLUTIONNS+"SetPropertyIsAttribute"))) {
            Property propertyReferenceProperty=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceProperty");
            Property propertyValue=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_value");
            changeEvent=new SetPropertyIsAttribute(oimodel,null,getDomainProperty(instance,propertyReferenceProperty),Boolean.getBoolean((String)instance.getFromPropertyValue(propertyValue)));
        }
        else if (instance.getParentConcepts().contains(getConcept(EVOLUTIONNS+"SetMinimumCardinality"))) {
            throw new KAONException("SetMinimumCardinality is not supported.");
        }
        else if (instance.getParentConcepts().contains(getConcept(EVOLUTIONNS+"SetMaximumCardinality"))) {
            throw new KAONException("SetMaximumCardinality is not supported.");
        }
        else if (instance.getParentConcepts().contains(getConcept(EVOLUTIONNS+"AddSubProperty"))) {
            Property propertySuperProperty=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceSuperProperty");
            Property propertySubProperty=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceSubProperty");
            changeEvent=new AddSubProperty(oimodel,null,getDomainProperty(instance,propertySuperProperty),getDomainProperty(instance,propertySubProperty));
        }
        else if (instance.getParentConcepts().contains(getConcept(EVOLUTIONNS+"RemoveSubProperty"))) {
            Property propertySuperProperty=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceSuperProperty");
            Property propertySubProperty=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceSubProperty");
            changeEvent=new RemoveSubProperty(oimodel,null,getDomainProperty(instance,propertySuperProperty),getDomainProperty(instance,propertySubProperty));
        }
        else if (instance.getParentConcepts().contains(getConcept(EVOLUTIONNS+"SetInverseProperties"))) {
            Property propertyReferenceProperty=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceProperty");
            Set set=instance.getFromPropertyValues(propertyReferenceProperty);
            if (set.size()!=2)
                throw new KAONException("Error in SetInverseProperties.");
            Iterator iterator=set.iterator();
            String domainPropertyURI1=(String)iterator.next();
            String domainPropertyURI2=(String)iterator.next();
            changeEvent=new SetInverseProperties(oimodel,null,m_oimodel.getProperty(domainPropertyURI1),m_oimodel.getProperty(domainPropertyURI2));
        }
        else if (instance.getParentConcepts().contains(getConcept(EVOLUTIONNS+"SetNoInverseProperties"))) {
            Property propertyReferenceProperty=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceProperty");
            Set set=instance.getFromPropertyValues(propertyReferenceProperty);
            if (set.size()!=2)
                throw new KAONException("Error in SetNoInverseProperties.");
            Iterator iterator=set.iterator();
            String domainPropertyURI1=(String)iterator.next();
            String domainPropertyURI2=(String)iterator.next();
            changeEvent=new SetNoInverseProperties(oimodel,null,m_oimodel.getProperty(domainPropertyURI1),m_oimodel.getProperty(domainPropertyURI2));
        }
        else if (instance.getParentConcepts().contains(getConcept(EVOLUTIONNS+"SetPropertySymmetric"))) {
            Property propertyReferenceProperty=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceProperty");
            Property propertyValue=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_value");
            changeEvent=new SetPropertySymmetric(oimodel,null,getDomainProperty(instance,propertyReferenceProperty),Boolean.getBoolean((String)instance.getFromPropertyValue(propertyValue)));
        }
        else if (instance.getParentConcepts().contains(getConcept(EVOLUTIONNS+"SetPropertyTransitive"))) {
            Property propertyReferenceProperty=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceProperty");
            Property propertyValue=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_value");
            changeEvent=new SetPropertyTransitive(oimodel,null,getDomainProperty(instance,propertyReferenceProperty),Boolean.getBoolean((String)instance.getFromPropertyValue(propertyValue)));
        }
        else if (instance.getParentConcepts().contains(getConcept(EVOLUTIONNS+"AddInstanceOf"))) {
            Property propertyConcept=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceConcept");
            Property propertyInstance=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceInstance");
            changeEvent=new AddInstanceOf(oimodel,null,getDomainConcept(instance,propertyConcept),getDomainInstance(instance,propertyInstance));
        }
        else if (instance.getParentConcepts().contains(getConcept(EVOLUTIONNS+"RemoveInstanceOf"))) {
            Property propertyConcept=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceConcept");
            Property propertyInstance=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceInstance");
            changeEvent=new RemoveInstanceOf(oimodel,null,getDomainConcept(instance,propertyConcept),getDomainInstance(instance,propertyInstance));
        }
        else if (instance.getParentConcepts().contains(getConcept(EVOLUTIONNS+"AddPropertyInstance"))) {
            Property propertyReferenceProperty=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceProperty");
            Property propertyReferenceSourceInstance=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceSourceInstance");
            Property propertyReferenceTargetInstance=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceTargetInstance");
            Property propertyReferenceTargetObject=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceTargetObject");
            String domainEntityURI=(String)instance.getFromPropertyValue(propertyReferenceTargetInstance);
            if (domainEntityURI!=null) {
                Instance targetInstance=m_oimodel.getInstance(domainEntityURI);
                changeEvent=new AddPropertyInstance(oimodel,null,getDomainProperty(instance,propertyReferenceProperty),getDomainInstance(instance,propertyReferenceSourceInstance),targetInstance);
            }
            else
                changeEvent=new AddPropertyInstance(oimodel,null,getDomainProperty(instance,propertyReferenceProperty),getDomainInstance(instance,propertyReferenceSourceInstance),(String)instance.getFromPropertyValue(propertyReferenceTargetObject));
        }
        else if (instance.getParentConcepts().contains(getConcept(EVOLUTIONNS+"RemovePropertyInstance"))) {
            Property propertyReferenceProperty=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceProperty");
            Property propertyReferenceSourceInstance=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceSourceInstance");
            Property propertyReferenceTargetInstance=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceTargetInstance");
            Property propertyReferenceTargetObject=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceTargetObject");
            String domainEntityURI=(String)instance.getFromPropertyValue(propertyReferenceTargetInstance);
            if (domainEntityURI!=null) {
                Instance targetInstance=m_oimodel.getInstance(domainEntityURI);
                changeEvent=new RemovePropertyInstance(oimodel,null,getDomainProperty(instance,propertyReferenceProperty),getDomainInstance(instance,propertyReferenceSourceInstance),targetInstance);
            }
            else
                changeEvent=new RemovePropertyInstance(oimodel,null,getDomainProperty(instance,propertyReferenceProperty),getDomainInstance(instance,propertyReferenceSourceInstance),(String)instance.getFromPropertyValue(propertyReferenceTargetObject));
        }
        else if (instance.getParentConcepts().contains(getConcept(EVOLUTIONNS+"SetPropertyInstanceValue"))) {
            Property propertyReferenceProperty=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceProperty");
            Property propertyReferenceSourceInstance=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceSourceInstance");
            Property propertyReferenceTargetInstance=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceTargetInstance");
            Property propertyReferenceTargetObject=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceTargetObject");
            String domainEntityURI=(String)instance.getFromPropertyValue(propertyReferenceTargetInstance);
            if (domainEntityURI!=null) {
                Instance targetInstance=m_oimodel.getInstance(domainEntityURI);
                changeEvent=new SetPropertyInstanceValue(oimodel,null,getDomainProperty(instance,propertyReferenceProperty),getDomainInstance(instance,propertyReferenceSourceInstance),targetInstance);
            }
            else
                changeEvent=new SetPropertyInstanceValue(oimodel,null,getDomainProperty(instance,propertyReferenceProperty),getDomainInstance(instance,propertyReferenceSourceInstance),(String)instance.getFromPropertyValue(propertyReferenceTargetObject));
        }
        else if (instance.getParentConcepts().contains(getConcept(EVOLUTIONNS+"ChangePropertyInstanceValue"))) {
            Property propertyReferenceProperty=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceProperty");
            Property propertyReferenceSourceInstance=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceSourceInstance");
            Property propertyReferenceOldTargetInstance=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_oldReferenceTargetInstance");
            Property propertyReferenceOldTargetObject=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_oldReferenceTargetObject");
            Property propertyReferenceNewTargetInstance=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_newReferenceTargetInstance");
            Property propertyReferenceNewTargetObject=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_newReferenceTargetObject");
            String oldDomainEntityURI=(String)instance.getFromPropertyValue(propertyReferenceOldTargetInstance);
            String newDomainEntityURI=(String)instance.getFromPropertyValue(propertyReferenceNewTargetInstance);
            if (oldDomainEntityURI!=null) {
                Instance oldTargetInstance=m_oimodel.getInstance(oldDomainEntityURI);
                if (newDomainEntityURI!=null) {
                    Instance newTargetInstance=m_oimodel.getInstance(newDomainEntityURI);
                    changeEvent=new ChangePropertyInstanceValue(oimodel,null,getDomainProperty(instance,propertyReferenceProperty),getDomainInstance(instance,propertyReferenceSourceInstance),oldTargetInstance, newTargetInstance);
                }
                else
                    changeEvent=new ChangePropertyInstanceValue(oimodel,null,getDomainProperty(instance,propertyReferenceProperty),getDomainInstance(instance,propertyReferenceSourceInstance),oldTargetInstance,(String)instance.getFromPropertyValue(propertyReferenceNewTargetObject));
            }
            else {
                if (newDomainEntityURI!=null) {
                    Instance newTargetInstance=m_oimodel.getInstance(newDomainEntityURI);
                    changeEvent=new ChangePropertyInstanceValue(oimodel,null,getDomainProperty(instance,propertyReferenceProperty),getDomainInstance(instance,propertyReferenceSourceInstance),(String)instance.getFromPropertyValue(propertyReferenceOldTargetObject),newTargetInstance);
                }
                else
                    changeEvent=new ChangePropertyInstanceValue(oimodel,null,getDomainProperty(instance,propertyReferenceProperty),getDomainInstance(instance,propertyReferenceSourceInstance),(String)instance.getFromPropertyValue(propertyReferenceOldTargetObject),(String)instance.getFromPropertyValue(propertyReferenceNewTargetObject));
            }
        }
        else if (instance.getParentConcepts().contains(getConcept(EVOLUTIONNS+"AddIncludedOIModel"))) {
            throw new KAONException("AddIncludedOIModel is not supported.");
        }
        else if (instance.getParentConcepts().contains(getConcept(EVOLUTIONNS+"RemoveIncludedOIModel"))) {
            throw new KAONException("RemoveIncludedOIModel is not supported.");
        }
        else
            throw new KAONException("Unknown change");

        Set set=instance.getFromPropertyValues(getProperty(EVOLUTIONNS+"causesChange"));
        Iterator iter=set.iterator();
        while(iter.hasNext()) {
            Instance causedInstance=(Instance)iter.next();
            List list=new LinkedList();
            list.add(instance);
            list.add(causedInstance);
            m_causesCausedChanges.add(list);
        }
        m_mapInstanceChange.put(instance,changeEvent);
        return changeEvent;
    }
    protected List improveListOfChange(List changes)  {
        Iterator iter=m_causesCausedChanges.iterator();
        while(iter.hasNext()) {
            List list=(List)iter.next();
            Instance instance1=(Instance)list.get(0);
            Instance instance2=(Instance)list.get(1);
            ChangeEvent event1=(ChangeEvent)m_mapInstanceChange.get(instance1);
            ChangeEvent event2=(ChangeEvent)m_mapInstanceChange.get(instance2);
            int index=changes.indexOf(event2);
            event2.setCause(event1);
            changes.set(index,event2);
        }
        return changes;
    }
    protected Concept getConcept(String uri) throws KAONException {
        return m_evolutionOIModelInstance.getConcept(uri);
    }
    protected Property getProperty(String uri) throws KAONException {
        return m_evolutionOIModelInstance.getProperty(uri);
    }
    protected Instance getInstance(String uri) throws KAONException {
        return m_evolutionOIModelInstance.getInstance(uri);
    }
    protected Entity getDomainEntity(Instance instance) throws KAONException {
        Property property=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceInstance");
        String domainEntityURI=(String)instance.getFromPropertyValue(property);
        if (domainEntityURI!=null)
            return m_oimodel.getInstance(domainEntityURI);
        property=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceProperty");
        domainEntityURI=(String)instance.getFromPropertyValue(property);
        if (domainEntityURI!=null)
            return m_oimodel.getProperty(domainEntityURI);
        property=getProperty(OIModelEvolutionLog.EVOLUTIONNS+"has_referenceConcept");
        domainEntityURI=(String)instance.getFromPropertyValue(property);
        if (domainEntityURI!=null)
            return m_oimodel.getConcept(domainEntityURI);
        throw new KAONException("Unknown entity type");
    }
    protected Concept getDomainConcept(Instance instance,Property property) throws KAONException {
        return m_oimodel.getConcept((String)instance.getFromPropertyValue(property));
    }
    protected Property getDomainProperty(Instance instance,Property property) throws KAONException {
        return m_oimodel.getProperty((String)instance.getFromPropertyValue(property));
    }
    protected Instance getDomainInstance(Instance instance,Property property) throws KAONException {
        return m_oimodel.getInstance((String)instance.getFromPropertyValue(property));
    }
    protected Instance getLastChange(Instance log) throws KAONException {
        return (Instance)log.getFromPropertyValue(getProperty(EVOLUTIONNS+"lastChange"));
    }
    protected Instance getLastHistoryChange(Instance log) throws KAONException {
        return (Instance)log.getFromPropertyValue(getProperty(EVOLUTIONNS+"lastHistoryChange"));
    }
    protected Instance getRedoListHead(Instance log) throws KAONException {
        return (Instance)log.getFromPropertyValue(getProperty(EVOLUTIONNS+"redoListHead"));
    }
}
