package edu.unika.aifb.kaon.evolutionlog;

import java.util.List;
import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.oimodel.*;

/**
 * An instance of this class manages all evolution logs within a KAON connection.
 *
 * @author Ljiljana Stojanovic (Ljiljana.Stojanovic@fzi.de)
 */
public class KAONConnectionEvolutionLogs {
    /** The attribute of the evolution log. */
    public static final String EVOLUTION_LOG_ATTIBUTE="evolutionLog.physicalURI";

    /** Stores the open evolution logs indexed by the OI-model. */
    protected Map m_evolutionLogsByOIModel;

    /**
     * Creates an instance of this class.
     */
    public KAONConnectionEvolutionLogs() {
        m_evolutionLogsByOIModel=new HashMap();
    }
    /**
     * Saves all open logs.
     *
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized void save() throws KAONException {
        KAONException error=null;
        Iterator iterator=m_evolutionLogsByOIModel.values().iterator();
        while (iterator.hasNext()) {
            EvolutionLog evolutionLog=(EvolutionLog)iterator.next();
            try {
                evolutionLog.save();
            }
            catch (KAONException e) {
                error=e;
            }
        }
        // This rethrows only the last exception, but I don't care.
        if (error!=null)
            throw error;
    }
    /**
     * Closes all open logs.
     *
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized void close() throws KAONException {
        KAONException error=null;
        Iterator iterator=new HashSet(m_evolutionLogsByOIModel.values()).iterator();
        while (iterator.hasNext()) {
            EvolutionLog evolutionLog=(EvolutionLog)iterator.next();
            try {
                evolutionLog.close();
            }
            catch (KAONException e) {
                error=e;
            }
        }
        // This rethrows only the last exception, but I don't care.
        if (error!=null)
            throw error;
    }
    /**
     * Refreshes all logs.
     *
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized void refresh() throws KAONException {
        KAONException error=null;
        Iterator iterator=m_evolutionLogsByOIModel.values().iterator();
        while (iterator.hasNext()) {
            EvolutionLog evolutionLog=(EvolutionLog)iterator.next();
            try {
                evolutionLog.refresh();
            }
            catch (KAONException e) {
                error=e;
            }
        }
        // This rethrows only the last exception, but I don't care.
        if (error!=null)
            throw error;
    }
    /**
     * Returns <code>true</code> if some of the logs has unsaved changes.
     *
     * @return                                  <code>true</code> if some of the logs has unsaved changes
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized boolean hasUnsavedChanges() throws KAONException {
        Iterator iterator=m_evolutionLogsByOIModel.values().iterator();
        while (iterator.hasNext()) {
            EvolutionLog evolutionLog=(EvolutionLog)iterator.next();
            if (evolutionLog.hasUnsavedChanges())
                return true;
        }
        return false;
    }
    /**
     * Returns an evolution log for given OI-model.
     *
     * @param oimodel                           the OI-model for which the evolution log is requested
     * @return                                  the evolution log for supplied OI-model or <code>null</code> if the evolution log doesn't exist for the OI-model
     * @throws KAONException                    thrown if the evolution log cannot be opened
     */
    public synchronized EvolutionLog getEvolutionLog(OIModel oimodel) throws KAONException {
        EvolutionLog evolutionLog=(EvolutionLog)m_evolutionLogsByOIModel.get(oimodel);
        if (evolutionLog!=null)
            return evolutionLog;
        else {
            String evolutionLogPhysicalURI=oimodel.getAttribute(EVOLUTION_LOG_ATTIBUTE);
            if (evolutionLogPhysicalURI==null)
                return new TransientEvolutionLog(this,oimodel);
            else {
                KAONConnection oimodelConnection=oimodel.getKAONConnection();
                Map parameters=KAONManager.getDefaultParametersForPhysicalURI(evolutionLogPhysicalURI,oimodelConnection.getParameters());
                boolean reuseOIModelConnection=KAONManager.canUseConnectionForParameters(oimodelConnection,parameters);
                KAONConnection evolutionConnection=oimodelConnection;
                if (!reuseOIModelConnection)
                    evolutionConnection=KAONManager.getKAONConnection(parameters);
                try {
                    OIModel evolutionOIModelInstance=evolutionConnection.openOIModelPhysical(evolutionLogPhysicalURI);
                    return new OIModelEvolutionLog(this,oimodel,evolutionOIModelInstance);
                }
                catch (KAONException e) {
                    if (!reuseOIModelConnection)
                        evolutionConnection.close();
                    throw e;
                }
            }
        }
    }
    /**
     * @param oimodel                           the OI-model
     * @param parameters                        the parameters
     * @param evolutionLogPhysicalURI           the physical URI of the evolution log
     * @param evolutionLogLogicalURI            the logical URI of the evolution log
     * @return                                  the evolution log for given OI-model
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized EvolutionLog createEvolutionLog(OIModel oimodel,Map parameters,String evolutionLogPhysicalURI,String evolutionLogLogicalURI) throws KAONException {
        if (DistributedEvolutionStrategy.isReplica(oimodel))
            throw new KAONException("Can't create an evolution log for an OI-model that is a replica.");
        if (oimodel.getAttribute(EVOLUTION_LOG_ATTIBUTE)!=null)
            throw new KAONException("Evolution log already exists for supplied OI-model.");
        EvolutionLog existingEvolutionLog=(EvolutionLog)m_evolutionLogsByOIModel.get(oimodel);
        if (existingEvolutionLog!=null)
            existingEvolutionLog.close();
        KAONConnection oimodelConnection=oimodel.getKAONConnection();
        boolean reuseOIModelConnection=KAONManager.canUseConnectionForParameters(oimodelConnection,parameters);
        KAONConnection evolutionConnection=oimodelConnection;
        if (!reuseOIModelConnection)
            evolutionConnection=KAONManager.getKAONConnection(parameters);
        try {
            OIModel evolutionOIModelInstance=evolutionConnection.createOIModel(evolutionLogPhysicalURI,evolutionLogLogicalURI);
            EvolutionLog evolutionLog=new OIModelEvolutionLog(this,oimodel,evolutionOIModelInstance);
            oimodel.setAttribute(EVOLUTION_LOG_ATTIBUTE,evolutionOIModelInstance.getPhysicalURI());
            return evolutionLog;
        }
        catch (KAONException e) {
            if (!reuseOIModelConnection)
                evolutionConnection.close();
            throw e;
        }
    }

    /**
     * Logs changes and increases the version numbers of OI-models appropriately.
     *
     * @param changeList                        the list of changes
     * @param loggingType                       the type of logging
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized void logChanges(List changeList,int loggingType) throws KAONException {
        Map loggers=new HashMap();
        Set changedOIModels=new HashSet();
        Iterator iterator=changeList.iterator();
        while (iterator.hasNext()) {
            ChangeEvent changeEvent=(ChangeEvent)iterator.next();
            OIModel oimodel=changeEvent.getOIModel();
            logChange(changeEvent,oimodel,loggers,changedOIModels,loggingType);
            Iterator oimodels=oimodel.getAllIncludedByOIModels().iterator();
            while (oimodels.hasNext()) {
                OIModel inclduedByOIModel=(OIModel)oimodels.next();
                logChange(changeEvent,inclduedByOIModel,loggers,changedOIModels,loggingType);
            }
        }
        iterator=loggers.keySet().iterator();
        while (iterator.hasNext()) {
            OIModel oimodel=(OIModel)iterator.next();
            EvolutionLog.Logger logger=(EvolutionLog.Logger)loggers.get(oimodel);
            logger.finishChangeBatch();
        }
        iterator=changedOIModels.iterator();
        while (iterator.hasNext()) {
            OIModel oimodel=(OIModel)iterator.next();
            setOIModelVersion(oimodel,getOIModelVersion(oimodel));
        }
    }
    /**
     * Pushes the changes into a log.
     *
     * @param changeEvent                       the change event
     * @param oimodel                           the OI-model
     * @param loggers                           the map of loggers
     * @param changedOIModels                   the set of all OI-models that were changed
     * @param loggingType                       the type of logging
     * @throws KAONException                    thrown if there is an error
     */
    protected void logChange(ChangeEvent changeEvent,OIModel oimodel,Map loggers,Set changedOIModels,int loggingType) throws KAONException {
        changedOIModels.add(oimodel);
        if (!DistributedEvolutionStrategy.isReplica(oimodel)) {
            EvolutionLog.Logger logger=(EvolutionLog.Logger)loggers.get(oimodel);
            if (logger==null) {
                EvolutionLog evolutionLog=getEvolutionLog(oimodel);
                logger=evolutionLog.createLogger(loggingType);
                loggers.put(oimodel,logger);
            }
            logger.logChange(changeEvent);
        }
    }
    /**
     * Notifies this object that an evolution log has been loaded for an OI-model.
     *
     * @param evolutionLog                      the evolution log that is put into memory
     */
    void notifyEvolutionLogLoaded(EvolutionLog evolutionLog) {
        m_evolutionLogsByOIModel.put(evolutionLog.getDomainOIModel(),evolutionLog);
    }
    /**
     * Notifies this object that an evolution log has been unloaded for an OI-model.
     *
     * @param evolutionLog                      the evolution log that is removed from memory (closed or deleted)
     */
    void notifyEvolutionLogUnloaded(EvolutionLog evolutionLog) {
        m_evolutionLogsByOIModel.remove(evolutionLog.getDomainOIModel());
    }
    /**
     * Returns the version number of the oimodel.
     *
     * @param oimodel                           the OI-model
     * @return                                  the version of the OI-model
     * @throws KAONException                    thrown if there is an error
     */
    public int getOIModelVersion(OIModel oimodel) throws KAONException {
        String version=oimodel.getAttribute("OIModel.version");
        if (version==null)
            return 1;
        else
            return Integer.parseInt(version)+1;
    }
    /**
     * Sets the version number of the oimodel.
     *
     * @param oimodel                           the OI-model
     * @param version                           the new version of the associated OI-model
     * @throws KAONException                    thrown if there is an error
     */
    public void setOIModelVersion(OIModel oimodel,int version) throws KAONException {
        if (DistributedEvolutionStrategy.isReplica(oimodel)) {
            Map parameters=new HashMap();
            parameters.put(KAONManager.KAON_CONNECTION,"edu.unika.aifb.kaon.apionrdf.KAONConnectionImpl");
            KAONConnection connection=KAONManager.getKAONConnection(parameters);
            OIModel originalOIModel=connection.openOIModelPhysical(oimodel.getAttribute("OIModel.replicatedFromPhysicalURI"));
            String versionText=originalOIModel.getAttribute("OIModel.version");
            int originalVersion=versionText==null ? 0 : Integer.parseInt(versionText);
            oimodel.setAttribute("OIModel.version", String.valueOf(originalVersion));
        }
        else
            oimodel.setAttribute("OIModel.version", String.valueOf(version));
    }
}
