package edu.unika.aifb.kaon.evolutionlog;

import java.util.List;
import java.util.Iterator;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;

/**
 * An abstract base class for all evolution log types.
 *
 * @author Ljiljana Stojanovic (Ljiljana.Stojanovic@fzi.de)
 */
public abstract class EvolutionLog {
    /** Specifies that the log request was due to the normal request. */
    public static final int LOG_NORMAL=0;
    /** Specifies that the log request was due to an undo request. */
    public static final int LOG_UNDO=1;
    /** Specifies that the log request was due to a redo request. */
    public static final int LOG_REDO=2;

    /** The KAON connection evolution. */
    protected KAONConnectionEvolutionLogs m_kaonConnectionEvolutionLogs;
    /** The origial OI-model. */
    protected OIModel m_oimodel;

    /**
     * Creates an instance of this class.
     *
     * @param kaonConnectionEvolutionLogs       the manager of evolution logs
     * @param oimodel                           the original OI-model
     */
    public EvolutionLog(KAONConnectionEvolutionLogs kaonConnectionEvolutionLogs,OIModel oimodel) {
        m_kaonConnectionEvolutionLogs=kaonConnectionEvolutionLogs;
        m_oimodel=oimodel;
        m_kaonConnectionEvolutionLogs.notifyEvolutionLogLoaded(this);
    }
    /**
     * Returns the evolution log manager.
     *
     * @return                                  the evolution log manager
     */
    public synchronized KAONConnectionEvolutionLogs getKAONConnectionEvolutionLogs() {
        return m_kaonConnectionEvolutionLogs;
    }
    /**
     * Returns <code>true</code> if this log is owned by the domain OI-model for which it was opened.
     *
     * @return                                  <code>true</code> if this log is owned by the domain OI-model
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized boolean isOwnedByDomainOIModel() throws KAONException {
        return true;
    }
    /**
     * Returns the domain OI-model.
     *
     * @return                                  the domain OI-model
     */
    public synchronized OIModel getDomainOIModel() {
        return m_oimodel;
    }
    /**
     * Saves this evolution log.
     *
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized void save() throws KAONException {
    }
    /**
     * Closes this evolution log.
     *
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized void close() throws KAONException {
        if (m_kaonConnectionEvolutionLogs!=null) {
            m_kaonConnectionEvolutionLogs.notifyEvolutionLogUnloaded(this);
            m_oimodel=null;
            m_kaonConnectionEvolutionLogs=null;
        }
    }
    /**
     * Deletes the evolution log.
     *
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized void delete() throws KAONException {
        if (!isOwnedByDomainOIModel())
            throw new KAONException("Evolution log can't be deleted since it isn't owned by the domain OI-model.");
        close();
    }
    /**
     * Refreshes the evolution log.
     *
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized void refresh() throws KAONException {
    }
    /**
     * Returns <code>true</code> if the log has unsaved changes.
     *
     * @return                                  <code>true</code> if the log has unsaved changes
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized boolean hasUnsavedChanges() throws KAONException {
        return false;
    }
    /**
     * Log changes.
     *
     * @param loggingType                       the type of changes
     * @param changeList                        the list of changes
     * @throws KAONException                    thrown if there is an error
     */
    public synchronized void logChanges(List changeList,int loggingType) throws KAONException {
        Logger logger=createLogger(loggingType);
        Iterator iterator=changeList.iterator();
        while (iterator.hasNext()) {
            ChangeEvent changeEvent=(ChangeEvent)iterator.next();
            logger.logChange(changeEvent);
        }
        logger.finishChangeBatch();
    }
    /**
     * Returns <code>true</code> if this log is persistent.
     *
     * @return                                  <code>true</code> if this log is persistent
     * @throws KAONException                    thrown if there is an error
     */
    public abstract boolean isPersistent() throws KAONException;
    /**
     * Read changes of given version.
     *
     * @param version                           the version whose changes are requested
     * @return                                  the list of performed changes
     * @throws KAONException                    thrown if there is an error
     */
    public abstract List readChanges(int version) throws KAONException;
    /**
     * Returns <code>true</code> if undo can be performed.
     *
     * @return                                  <code>true</code> if undo can be performed
     * @throws KAONException                    thrown if there is an error
     */
    public abstract boolean canUndoChanges() throws KAONException;
    /**
     * Returns the list of changes that must be applied to the model to undo last change batch.
     *
     * @return                                  the list of reversed changes
     * @throws KAONException                    thrown if there is an error
     */
    public abstract List readUndoChanges() throws KAONException;
    /**
     * Returns <code>true</code> if redo can be performed.
     *
     * @return                              <code>true</code> if redo can be performed
     * @throws KAONException                    thrown if there is an error
     */
    public abstract boolean canRedoChanges() throws KAONException;
    /**
     * Returns the list of changes that must be applied to the model to redo the last undone change batch.
     *
     * @return                                  the list of redoed changes
     * @throws KAONException                    thrown if there is an error
     */
    public abstract List readRedoChanges() throws KAONException;
    /**
     * Creates a logger.
     *
     * @param loggingType                       the type of logging
     * @return                                  the logger
     * @throws KAONException                    thrown if there is an error
     */
    public abstract Logger createLogger(int loggingType) throws KAONException;

    /**
     * An instance of this class is responsible for logging a batch of changes.
     */
    public static interface Logger {
        /**
         * Logs one change.
         *
         * @param changeEvent                   the event to be logged
         * @throws KAONException                thrown if there is an error
         */
        void logChange(ChangeEvent changeEvent) throws KAONException;
        /**
         * Finishes the logging of changes.
         *
         * @throws KAONException                thrown if there is an error
         */
        void finishChangeBatch() throws KAONException;
    }
}
