package edu.unika.aifb.kaon.defaultevolution;

/**
 * Represents a set of parameters for the evolution.
 *
 * @author Ljiljana Stojanovic (Ljiljana.Stojanovic@fzi.de)
 */
public class EvolutionParameters {
    /** Determines how to handle orphaned concepts - those concepts that don't have parents any more. */
    protected int m_orphanedConcepts;
    /** Orphaned concepts are deleted. */
    public static final int ORPHANED_CONCEPTS_DELETE=0;
    /** Orphaned concepts are reconnected to the kaon:Root concept. */
    public static final int ORPHANED_CONCEPTS_RECONNECT_TO_ROOT=1;
    /** Orphaned concepts are reconnected to their parents. */
    public static final int ORPHANED_CONCEPTS_RECONNECT_TO_SUPER=2;

    /** Determines how to handle orphaned properties - those properties that don't have parents any more. */
    protected int m_orphanedProperties;
    /** Orphaned properties are deleted. */
    public static final int ORPHANED_PROPERTIES_DELETE=0;
    /** Orphaned properties are reconnected to parent. */
    public static final int ORPHANED_PROPERTIES_RECONNECT_TO_SUPER=1;
    /** Orphaned properties are left alone. */
    public static final int ORPHANED_PROPERTIES_LEAVE_ALONE=2;

    /** Determines how to propagate properties to concept whose parent changes. */
    protected int m_propertyPropagation;
    /** Don't propagate any properties of superconcepts. */
    public static final int PROPERTY_PROPAGATION_NO=0;
    /** Propagate all inherited properties form superconcepts to the concept whose parent changes. */
    public static final int PROPERTY_PROPAGATION_ALL=1;
    /** Propagate only properties of the parent concept to the concept whose parent changes. */
    public static final int PROPERTY_PROPAGATION_EXPLICIT=2;

    /** Determines whether a domain (range) of a property can contain a concept that is at the same time a subconcept of some other domain (range) concept. */
    protected int m_domainRangeShape;
    /** Property range cannot contain a concept whose superconcept is also a domain concept. */
    public static final int DOMAIN_RANGE_SHAPE_ALLOW_REPEATED_SUBCONCEPTS=0;
    /** Property range can contain any concept. */
    public static final int DOMAIN_RANGE_SHAPE_DISALLOW_REPEATED_SUBCONCEPTS=1;

    /** Determines what constitutes a valid domain of some property. */
    protected int m_propertyDomainEmpty;
    /**  Property can exist without a domain concept. */
    public static final int PROPERTY_DOMAIN_EMPTY_ALLOW=0;
    /**  Property cannot exist without a domain concept. */
    public static final int PROPERTY_DOMAIN_EMPTY_DISALLOW=1;

    /** Determines what constitutes a valid range of some property. */
    protected int m_propertyRangeEmpty;
    /**  Property can exist without a range concept. */
    public static final int PROPERTY_RANGE_EMPTY_ALLOW=0;
    /**  Property cannot exist without a range concept. */
    public static final int PROPERTY_RANGE_EMPTY_DISALLOW=1;

    /** Determines whether instances must be consistent with the ontology. */
    protected int m_instanceConsistency;
    /** Instances must be consistent with the ontology. */
    public static final int INSTANCE_CONSISTENCY_ENFORCE=0;
    /** Instances must not be consistent with the ontology. */
    public static final int INSTANCE_CONSISTENCY_DO_NOT_ENFORCE=1;

    /** Determines the allowed shape of the concept hierarchy. */
    protected int m_conceptHierarchyShape;
    /** There is no constraint for a concept hierarchy shape. */
    public static final int CONCEPT_HIERARCHY_SHAPE_NO_CONSTRAINT=0;
    /** Remove multiple paths to a superconcept in the hierarchy. */
    public static final int CONCEPT_HIERARCHY_SHAPE_REMOVE_MULTIPLE_PATHS=1;
    /** Do not allow multiple to a superconcept in the hierarchy. */
    public static final int CONCEPT_HIERARCHY_SHAPE_DO_NOT_ALLOW_MULTIPLE_PATHS=2;

    /** Determines how to handle orphaned instances - those instances whose parent concept is deleted. */
    protected int m_orphanedInstances;
    /** Orphaned instances are deleted. */
    public static final int ORPHANED_INSTANCES_DELETE=0;
    /** Orphaned instances are reconnected to parent. */
    public static final int ORPHANED_INSTANCES_RECONNECT_TO_SUPER=1;
    /** Orphaned instances are left unchanged. */
    public static final int ORPHANED_INSTANCES_LEFT_UNCHANGED=2;

    /** Determines how to handle orphaned property instances - those property instances that don't have property any more. */
    protected int m_orphanedPropertyInstances;
    /** Orphaned property instances are deleted. */
    public static final int ORPHANED_PROPERTYINSTANCES_DELETE=0;
    /** Orphaned property instances are reconnected to parent property. */
    public static final int ORPHANED_PROPERTYINSTANCES_RECONNECT_TO_SUPER=1;

    /**
     * Creates an instance of this class.
     */
    public EvolutionParameters() {
        setDefaultParameters();
    }
    /**
     * Sets default values for the parameters
     */
    protected void setDefaultParameters() {
        setOrphanedConcepts(ORPHANED_CONCEPTS_RECONNECT_TO_SUPER);
        setOrphanedProperties(ORPHANED_PROPERTIES_LEAVE_ALONE);
        setPropertyPropagation(PROPERTY_PROPAGATION_NO);
        setDomainRangeShape(DOMAIN_RANGE_SHAPE_ALLOW_REPEATED_SUBCONCEPTS);
        setPropertyDomainEmpty(PROPERTY_DOMAIN_EMPTY_DISALLOW);
        setPropertyRangeEmpty(PROPERTY_RANGE_EMPTY_DISALLOW);
        setInstanceConsistency(INSTANCE_CONSISTENCY_ENFORCE);
        setConceptHierarchyShape(CONCEPT_HIERARCHY_SHAPE_NO_CONSTRAINT);
        setOrphanedInstances(ORPHANED_INSTANCES_RECONNECT_TO_SUPER);
        setOrphanedPropertyInstances(ORPHANED_PROPERTYINSTANCES_RECONNECT_TO_SUPER);
    }
    /**
     * Determines how to handle orphaned concepts - those concepts that don't have parents any more.
     *
     * @param input                     orphaned concept mode
     */
    public void setOrphanedConcepts(int input) {
        m_orphanedConcepts=input;
    }
    /**
     * Returns how to handle orphaned concepts - those concepts that don't have parents any more.
     *
     * @return                          orphaned concept mode
     */
    public int getOrphanedConcepts() {
        return m_orphanedConcepts;
    }
   /**
    * Determines how to handle orphaned properties - those properties that don't have parents any more.
    *
    * @param input                     mode for the orphaned properties
    */
    public void setOrphanedProperties(int input) {
        m_orphanedProperties=input;
    }
    /**
     * Returns how to handle orphaned properties - those properties that don't have parents any more.
     *
     * @return                          mode for the orphaned properties
     */
    public int getOrphanedProperties() {
        return m_orphanedProperties;
    }
   /**
     * Set the property propagation mode.
     *
     * @param input                     property propagation mode
     */
    public void setPropertyPropagation(int input) {
        m_propertyPropagation=input;
    }
    /**
     * Returns the property propagation mode.
     *
     * @return                          property propagation mode
     */
    public int getPropertyPropagation() {
        return m_propertyPropagation;
    }
   /**
     * Determines whether a domain (range) of a property can contain a concept that is at the same time a subconcept of some other domain (range) concept.
     *
     * @param input                     mode for the domain concept
     */
    public void setDomainRangeShape(int input) {
        m_domainRangeShape=input;
    }
    /**
     * Returns whether a domain (range) of a property can contain a concept that is at the same time a subconcept of some other domain (range) concept.
     *
     * @return                          mode for a domain range set
     */
    public int getDomainRangeShape() {
        return m_domainRangeShape;
    }
   /**
     * Determines whether property domain can be empty.
     *
     * @param input                     mode for the property domain
     */
    public void setPropertyDomainEmpty(int input) {
        m_propertyDomainEmpty=input;
    }
    /**
     * Returns whether property domain can be empty.
     *
     * @return                          mode for the property domain
     */
    public int getPropertyDomainEmpty() {
        return m_propertyDomainEmpty;
    }
   /**
     * Determines whether property range can be empty.
     *
     * @param input                     mode for the property range
     */
    public void setPropertyRangeEmpty(int input) {
        m_propertyRangeEmpty=input;
    }
    /**
     * Returns whether property range can be empty.
     *
     * @return                          mode for the property range
     */
    public int getPropertyRangeEmpty() {
        return m_propertyRangeEmpty;
    }
   /**
     * Determines whether instances must be consistent with the ontology.
     *
     * @param input                     mode for instances
     */
    public void setInstanceConsistency(int input) {
        m_instanceConsistency=input;
    }
    /**
     * Returns whether instances must be consistent with the ontology.
     *
     * @return                          mode for instances
     */
    public int getInstanceConsistency() {
        return m_instanceConsistency;
    }
   /**
     * Determines the allowed shape of the concept hierarchy.
     *
     * @param input                     mode for the concept hierarchy
     */
    public void setConceptHierarchyShape(int input) {
        m_conceptHierarchyShape=input;
    }
    /**
     * Returns the allowed shape of the concept hierarchy.
     *
     * @return                          mode for the concept hierarchy
     */
    public int getConceptHierarchyShape() {
        return m_conceptHierarchyShape;
    }
    /**
     * Determines how to handle orphaned instances.
     *
     * @param input                     orphaned instance mode
     */
    public void setOrphanedInstances(int input) {
        m_orphanedInstances=input;
    }
    /**
     * Returns how to handle orphaned instances.
     *
     * @return                          orphaned instance mode
     */
    public int getOrphanedInstances() {
        return m_orphanedInstances;
    }
    /**
     * Determines how to handle orphaned property instances.
     *
     * @param input                     orphaned property instance mode
     */
    public void setOrphanedPropertyInstances(int input) {
        m_orphanedPropertyInstances=input;
    }
    /**
     * Returns how to handle orphaned property instances.
     *
     * @return                          orphaned property instance mode
     */
    public int getOrphanedPropertyInstances() {
        return m_orphanedPropertyInstances;
    }

}
