package edu.unika.aifb.kaon.defaultevolution;

import java.util.List;
import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.oimodel.*;

/**
 * Visits evolution events and creates a set of objects to be loaded.
 *
 * @author Ljiljana Stojanovic (Ljiljana.Stojanovic@fzi.de)
 */
public class EvolutionLoadVisitor implements ChangeVisitor {
    /** OI-model for which evolution is performed. */
    protected OIModel m_oimodel;
    /** Set of concepts to load. */
    protected Set m_conceptsToLoad;
    /** Set of properties to load. */
    protected Set m_propertiesToLoad;
    /** Set of instances to load. */
    protected Set m_instancesToLoad;
    /** Set of instances for which lexicon should be loaded. */
    protected Set m_instancesToLoadLexicon;
    /** Set of properties that will be removed. */
    protected Set m_propertiesToRemove;
    /** Set of concepts that will be removed. */
    protected Set m_conceptsToRemove;
    /** Visitor for adding elements. */
    protected EntityVisitor m_addEntityVisitor;
    /** Visitor for element removal. */
    protected EntityVisitor m_removeEntityVisitor;

    /**
     * Creates an instance of this class.
     *
     * @param oimodel                   OI-model for which loading is done
     */
    public EvolutionLoadVisitor(OIModel oimodel) {
        m_oimodel=oimodel;
        m_conceptsToLoad=new HashSet();
        m_propertiesToLoad=new HashSet();
        m_instancesToLoad=new HashSet();
        m_instancesToLoadLexicon=new HashSet();
        m_conceptsToRemove=new HashSet();
        m_propertiesToRemove=new HashSet();
        m_addEntityVisitor=new AddEntityVisitor();
        m_removeEntityVisitor=new RemoveEntityVisitor();
    }
    /**
     * Adds a concept to the set of objects to load.
     *
     * @param concept                   concept to load
     */
    protected void loadConcept(Concept concept) throws KAONException {
        Concept conceptInThisModel=m_oimodel.getConcept(concept.getURI());
        m_conceptsToLoad.add(conceptInThisModel);
        m_instancesToLoad.add(conceptInThisModel.getSpanningInstance());
    }
    /**
     * Adds a property to the set of objects to load.
     *
     * @param property                  property to load
     */
    protected void loadProperty(Property property) throws KAONException {
        Property propertyInThisModel=m_oimodel.getProperty(property.getURI());
        m_propertiesToLoad.add(propertyInThisModel);
        m_instancesToLoad.add(propertyInThisModel.getSpanningInstance());
    }
    /**
     * Adds an instance to the set of instances to load.
     *
     * @param instance                  instance to load
     */
    protected void loadInstance(Instance instance) throws KAONException {
        Instance instanceInThisModel=m_oimodel.getInstance(instance.getURI());
        m_instancesToLoad.add(instanceInThisModel);
        m_conceptsToLoad.add(instanceInThisModel.getSpanningConcept());
        m_propertiesToLoad.add(instanceInThisModel.getSpanningProperty());
    }
    /**
     * Adds an entity to the set of entities for which the lexicon must be loaded.
     *
     * @param entity                    the entity for which the lexicon should be loaded
     */
    protected void loadLexicon(Entity entity) throws KAONException {
        Instance spanningInstanceInThisModel=m_oimodel.getInstance(entity.getURI());
        m_instancesToLoadLexicon.add(spanningInstanceInThisModel);
    }
    /**
     * Preloads given objects.
     *
     * @param changeList                list of change events
     * @throws KAONException            thrown in case of error
     */
    public void preloadObjects(List changeList) throws KAONException {
        Iterator iterator=changeList.iterator();
        while (iterator.hasNext()) {
            ChangeEvent changeEvent=(ChangeEvent)iterator.next();
            changeEvent.accept(this);
        }
        performLoading();
    }
    /**
     * Performs the loading of objects.
     *
     * @throws KAONException            thrown in case of error
     */
    public void performLoading() throws KAONException {
        m_oimodel.loadObjects(m_conceptsToLoad,OIModel.LOAD_CONCEPT_BASICS | OIModel.LOAD_SUPER_CONCEPTS | OIModel.LOAD_PROPERTIES_FROM | OIModel.LOAD_PROPERTIES_TO);
        m_oimodel.loadObjects(m_propertiesToLoad,OIModel.LOAD_PROPERTY_BASICS | OIModel.LOAD_SUPER_PROPERTIES | OIModel.LOAD_PROPERTY_DOMAINS | OIModel.LOAD_PROPERTY_RANGES);
        if (!m_instancesToLoad.isEmpty()) {
            m_oimodel.loadObjects(m_instancesToLoad,OIModel.LOAD_INSTANCE_BASICS | OIModel.LOAD_INSTANCE_PARENT_CONCEPTS | OIModel.LOAD_INSTANCE_FROM_PROPERTY_VALUES | OIModel.LOAD_INSTANCE_TO_PROPERTY_VALUES);
            Set properties=new HashSet();
            Iterator iterator=m_instancesToLoad.iterator();
            while (iterator.hasNext()) {
                Instance instance=(Instance)iterator.next();
                properties.addAll(instance.getFromPropertyValues().keySet());
                properties.addAll(instance.getToPropertyValues().keySet());
            }
            m_oimodel.loadObjects(properties,OIModel.LOAD_CONCEPT_BASICS | OIModel.LOAD_PROPERTY_BASICS | OIModel.LOAD_INSTANCE_BASICS);
        }
        m_oimodel.loadObjects(m_instancesToLoadLexicon,OIModel.LOAD_LEXICON);
        if (!m_conceptsToRemove.isEmpty()) {
            m_oimodel.loadObjects(m_conceptsToRemove,OIModel.LOAD_CONCEPT_BASICS | OIModel.LOAD_SUPER_CONCEPTS | OIModel.LOAD_SUB_CONCEPTS | OIModel.LOAD_PROPERTIES_FROM | OIModel.LOAD_PROPERTIES_TO | OIModel.LOAD_CONCEPT_INSTANCES);
            Set instances=new HashSet();
            Iterator iterator=m_conceptsToRemove.iterator();
            while (iterator.hasNext()) {
                Concept concept=(Concept)iterator.next();
                instances.addAll(concept.getInstances());
            }
            m_oimodel.loadObjects(instances,OIModel.LOAD_CONCEPT_BASICS | OIModel.LOAD_PROPERTY_BASICS | OIModel.LOAD_INSTANCE_ALL | OIModel.LOAD_LEXICON);
        }
        m_oimodel.loadObjects(m_propertiesToRemove,OIModel.LOAD_PROPERTY_INSTANCES);
    }
    /**
     * Visits AddEntity change.
     */
    public void visit(AddEntity event) throws KAONException {
        event.getEntity().accept(m_addEntityVisitor);
    }
    /**
     * Visits RemoveEntity change.
     */
    public void visit(RemoveEntity event) throws KAONException {
        event.getEntity().accept(m_removeEntityVisitor);
    }
    /**
     * Visits an event for creation of subconcepts.
     */
    public void visit(AddSubConcept event) throws KAONException {
        loadConcept(event.getSuperConcept());
        loadConcept(event.getSubConcept());
    }
    /**
     * Visits an event for removal of subconcepts.
     */
    public void visit(RemoveSubConcept event) throws KAONException {
        loadConcept(event.getSuperConcept());
        loadConcept(event.getSubConcept());
    }
    /**
     * Visits an event for adding a domain to the property.
     */
    public void visit(AddPropertyDomain event) throws KAONException {
        loadProperty(event.getProperty());
        loadConcept(event.getConcept());
    }
    /**
     * Visits an event for removing a domain from the property.
     */
    public void visit(RemovePropertyDomain event) throws KAONException  {
        loadProperty(event.getProperty());
        loadConcept(event.getConcept());
    }
    /**
     * Visits an event for adding a range to the property.
     */
    public void visit(AddPropertyRange event) throws KAONException {
        loadProperty(event.getProperty());
        loadConcept(event.getConcept());
    }
    /**
     * Visits an event for removing a range from the property.
     */
    public void visit(RemovePropertyRange event) throws KAONException {
        loadProperty(event.getProperty());
        loadConcept(event.getConcept());
    }
    /**
     * Visits an event for determinig whether property is attirubte.
     */
    public void visit(SetPropertyIsAttribute event) throws KAONException {
        loadProperty(event.getProperty());
    }
    /**
     * Visits an event for creation of subproperties.
     */
    public void visit(AddSubProperty event) throws KAONException {
        loadProperty(event.getSuperProperty());
        loadProperty(event.getSubProperty());
    }
    /**
     * Visits an event for removal of subproperties.
     */
    public void visit(RemoveSubProperty event) throws KAONException {
        loadProperty(event.getSuperProperty());
        loadProperty(event.getSubProperty());
    }
    /**
     * Visits an event for setting the inverse propertyship between properties.
     */
    public void visit(SetInverseProperties event) throws KAONException {
        loadProperty(event.getProperty1());
        loadProperty(event.getProperty2());
    }
    /**
     * Visits an event for removing an inverse propertyship between properties.
     */
    public void visit(SetNoInverseProperties event) throws KAONException {
        loadProperty(event.getProperty1());
        loadProperty(event.getProperty2());
    }
    /**
     * Visits an event for setting the symmetry flag of the property.
     */
    public void visit(SetPropertySymmetric event) throws KAONException {
        loadProperty(event.getProperty());
    }
    /**
     * Visits an event for setting the transitivity flag of the property.
     */
    public void visit(SetPropertyTransitive event) throws KAONException {
        loadProperty(event.getProperty());
    }
    /**
     * Visits an event for setting the minimum cardinality of a property for a concept.
     */
    public void visit(SetMinimumCardinality event) throws KAONException {
        loadProperty(event.getProperty());
        loadConcept(event.getConcept());
    }
    /**
     * Visits an event for setting the maximum cardinality of a property for a concept.
     */
    public void visit(SetMaximumCardinality event) throws KAONException {
        loadProperty(event.getProperty());
        loadConcept(event.getConcept());
    }
    /**
     * Visits an event for making an instance a subinstance of given concept.
     */
    public void visit(AddInstanceOf event) throws KAONException  {
        loadConcept(event.getConcept());
        loadInstance(event.getInstance());
    }
    /**
     * Visits an event for making an instance not a subinstance of given concept.
     */
    public void visit(RemoveInstanceOf event) throws KAONException {
        loadConcept(event.getConcept());
        loadInstance(event.getInstance());
    }
    /**
     * Visits an event for adding a property instance.
     */
    public void visit(AddPropertyInstance event) throws KAONException {
        PropertyInstance propertyInstance=event.getPropertyInstance();
        loadProperty(propertyInstance.getProperty());
        loadInstance(propertyInstance.getSourceInstance());
        if (propertyInstance.getTargetValue() instanceof Instance)
            loadInstance((Instance)propertyInstance.getTargetValue());
    }
    /**
     * Visits an event for removing a property instance.
     */
    public void visit(RemovePropertyInstance event) throws KAONException {
        PropertyInstance propertyInstance=event.getPropertyInstance();
        loadProperty(propertyInstance.getProperty());
        loadInstance(propertyInstance.getSourceInstance());
        if (propertyInstance.getTargetValue() instanceof Instance)
            loadInstance((Instance)propertyInstance.getTargetValue());
    }
    /**
     * Visits an event for setting the value of a property instance.
     */
    public void visit(SetPropertyInstanceValue event) throws KAONException {
        PropertyInstance propertyInstance=event.getPropertyInstance();
        loadProperty(propertyInstance.getProperty());
        loadInstance(propertyInstance.getSourceInstance());
        if (propertyInstance.getTargetValue() instanceof Instance)
            loadInstance((Instance)propertyInstance.getTargetValue());
    }
    /**
     * Visits an event for changing the value of a property instance.
     */
    public void visit(ChangePropertyInstanceValue event) throws KAONException {
        PropertyInstance propertyInstance=event.getPropertyInstance();
        loadProperty(propertyInstance.getProperty());
        loadInstance(propertyInstance.getSourceInstance());
        if (propertyInstance.getTargetValue() instanceof Instance)
            loadInstance((Instance)propertyInstance.getTargetValue());
        if (event.getNewTargetValue() instanceof Instance)
            loadInstance((Instance)event.getNewTargetValue());
    }
    /**
     * Visits an event for making a model included in another model.
     */
    public void visit(AddIncludedOIModel event) {
    }
    /**
     * Visits an event for making a model not included in another model.
     */
    public void visit(RemoveIncludedOIModel event) {
    }

    /**
     * Visitor of events that adds entities.
     */
    protected class AddEntityVisitor implements EntityVisitor {
        /**
         * Visits a concpet.
         */
        public void visit(Concept entity) throws KAONException {
            loadConcept(entity);
        }
        /**
         * Visits a property.
         */
        public void visit(Property entity) throws KAONException {
            loadProperty(entity);
        }
        /**
         * Visits an instance.
         */
        public void visit(Instance entity) throws KAONException {
            loadInstance(entity);
        }
        /**
         * Visits a lexical entry.
         */
        public void visit(LexicalEntry entity) throws KAONException {
            loadInstance(entity);
        }
    }

    /**
     * Visitor of events that remove entities.
     */
    protected class RemoveEntityVisitor implements EntityVisitor {
        /**
         * Visits a concpet.
         */
        public void visit(Concept entity) throws KAONException {
            loadConcept(entity);
            loadLexicon(entity);
            m_conceptsToRemove.add(m_oimodel.getConcept(entity.getURI()));
        }
        /**
         * Visits a property.
         */
        public void visit(Property entity) throws KAONException {
            loadProperty(entity);
            loadLexicon(entity);
            m_propertiesToRemove.add(m_oimodel.getProperty(entity.getURI()));
        }
        /**
         * Visits an instance.
         */
        public void visit(Instance entity) throws KAONException {
            loadInstance(entity);
            loadConcept(entity.getSpanningConcept());
            loadProperty(entity.getSpanningProperty());
            loadLexicon(entity);
        }
        /**
         * Visits a lexical entry.
         */
        public void visit(LexicalEntry entity) throws KAONException {
            loadInstance(entity);
            loadConcept(entity.getSpanningConcept());
            loadProperty(entity.getSpanningProperty());
            loadLexicon(entity);
        }
    }
}
