package edu.unika.aifb.kaon.defaultevolution;

import java.util.List;
import java.util.Iterator;
import java.util.Set;
import java.util.HashSet;
import java.util.Map;
import java.util.Stack;
import java.util.Collections;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.vocabulary.*;

import edu.unika.aifb.kaon.virtualoimodel.*;

/**
 * Visitor of change events for evolution.
 *
 * @author Ljiljana Stojanovic (Ljiljana.Stojanovic@fzi.de)
 */
public class EvolutionChangeVisitor implements ChangeVisitor {
    /** Evolution parameters. */
    protected EvolutionParameters m_evolutionParameters;
    /** OI-model. */
    protected OIModel m_oimodel;
    /** List with changes to the ontology. */
    protected List m_changes;
    /** Visitor for element removal. */
    protected EntityVisitor m_removeEntityVisitor;
    /** Virtual OI-model. */
    protected VirtualOIModel m_virtualOIModel;
    /** Stack for previous changes. */
    protected Stack m_stackOfChanges;
    /** Map of change visitor holders. */
    protected Map m_changeVisitorHolders;

    /**
     * Creates an instance of this class.
     *
     * @param evolutionParameters               evolution parameters
     * @param oimodel                           the OI-model
     * @param changeVisitorHolders              the map of change visitor holders
     * @param changes                           the list receiving the changes
     */
    public EvolutionChangeVisitor(EvolutionParameters evolutionParameters,OIModel oimodel,Map changeVisitorHolders,List changes) throws KAONException {
        m_evolutionParameters=evolutionParameters;
        m_oimodel=oimodel;
        m_changeVisitorHolders=changeVisitorHolders;
        m_changes=changes;
        m_removeEntityVisitor=new RemoveEntityVisitor();
        m_virtualOIModel=new VirtualOIModel(m_oimodel);
        m_stackOfChanges=new Stack();
    }
    /**
     * Applies a change to the virtual OI-model and adds it to the list of changes.
     *
     * @param event                         event to be applied
     */
    protected void applyChange(ChangeEvent event) throws KAONException {
        m_virtualOIModel.applyChanges(Collections.singletonList(event));
        if (event.getOIModel()==m_oimodel) {
            if (m_changeVisitorHolders!=null) {
                ChangeVisitorHolder holder=(ChangeVisitorHolder)m_changeVisitorHolders.get(m_oimodel);
                holder=holder.m_next;
                while (holder!=null) {
                    EvolutionChangeVisitor newVisitor=holder.m_changeVisitor;
                    if (m_oimodel.getAllIncludedByOIModels().contains(newVisitor.getOIModel()))
                        event.accept(holder.m_changeVisitor);
                    holder=holder.m_next;
                }
            }
            m_changes.add(event);
        }
    }
    /**
     * Visits AddEntity change.
     */
    public void visit(AddEntity event) throws KAONException {
        m_stackOfChanges.push(event);
        VirtualEntity entity=getVirtualEntity(event.getEntity());
        assertEntityNotInOIModel(entity);
        applyChange(event);
        m_stackOfChanges.pop();
    }
    /**
     * Visits RemoveEntity change.
     */
    public void visit(RemoveEntity event) throws KAONException {
        m_stackOfChanges.push(event);
        VirtualEntity entity=getVirtualEntity(event.getEntity());
        assertEntityInOIModel(entity);

        if (!DistributedEvolutionStrategy.isReplica(m_oimodel)) {
            removeLexicalEntries(entity);
            // generate all removal events until the entity can be removed
            entity.accept(m_removeEntityVisitor);
        }
        applyChange(event);
        m_stackOfChanges.pop();
    }
    /**
     * Removes lexical entry references to given entity. If supplied entity becomes orphaned (with no references), then it is
     * removed as well.
     *
     * @param entity                entity whose lexical references are removed
     */
    protected void removeLexicalEntries(VirtualEntity entity) throws KAONException {
        Property references=m_oimodel.getProperty(KAONVocabularyAdaptor.INSTANCE.getReferences());
        Iterator iterator=new HashSet(entity.getLexicalEntries()).iterator();
        while (iterator.hasNext()) {
            VirtualLexicalEntry virtualLexicalEntry=(VirtualLexicalEntry)iterator.next();
            ChangeEvent event=new RemovePropertyInstance(m_oimodel,getCurrentEvent(),references,virtualLexicalEntry.getInstance(),((VirtualInstance)entity.getSpanningInstance()).getInstance());
            m_stackOfChanges.push(event);
            applyChange(event);
            if (virtualLexicalEntry.getReferencedEntities().isEmpty())
                visit(new RemoveEntity(m_oimodel,getCurrentEvent(),virtualLexicalEntry.getInstance()));
            m_stackOfChanges.pop();
        }
    }
    /**
     * Prepares the given instance for removal. This includes removing all of its property
     * instances and parent concepts.
     *
     * @param virtualInstance       the instance from which property instances are removed
     */
    protected void prepareInstanceForRemoval(VirtualInstance virtualInstance) throws KAONException {
        Iterator iterator=new HashSet(virtualInstance.getFromPropertyInstances()).iterator();
        while (iterator.hasNext()) {
            VirtualPropertyInstance virtualPropertyInstance=(VirtualPropertyInstance)iterator.next();
            applyChange(new RemovePropertyInstance(m_oimodel,getCurrentEvent(),virtualPropertyInstance.getPropertyInstance()));
        }
        iterator=new HashSet(virtualInstance.getToPropertyInstances()).iterator();
        while (iterator.hasNext()) {
            VirtualPropertyInstance virtualPropertyInstance=(VirtualPropertyInstance)iterator.next();
            applyChange(new RemovePropertyInstance(m_oimodel,getCurrentEvent(),virtualPropertyInstance.getPropertyInstance()));
        }
        Set set=new HashSet(virtualInstance.getParentConcepts());
        iterator=set.iterator();
        while (iterator.hasNext()) {
            VirtualConcept virtualConcept=(VirtualConcept)iterator.next();
            if (!virtualConcept.equals(m_virtualOIModel.getRootConcept()))
                applyChange(new RemoveInstanceOf(m_oimodel,getCurrentEvent(),virtualConcept.getConcept(),virtualInstance.getInstance()));
        }
    }
    /**
     * Visitor of events that remove entities.
     */
    protected class RemoveEntityVisitor implements EntityVisitor {
        /**
         * Visits a concpet.
         */
        public void visit(Concept entity) throws KAONException {
            VirtualConcept virtualConcept=(VirtualConcept)entity;
            if (m_virtualOIModel.getRootConcept().equals(virtualConcept))
                throw new KAONException("Root concept cannot be deleted");
            // first remove anything that is linked to the spanning instance
            prepareInstanceForRemoval((VirtualInstance)virtualConcept.getSpanningInstance());
            if (m_evolutionParameters.getOrphanedConcepts()==EvolutionParameters.ORPHANED_CONCEPTS_DELETE) {
                // For all subconcepts, if subconcept...
                // ...has exaclty one parent (the concept being deleted), then delete it recursively
                // ...has more than one parent, remove hierarchy link to the concept
                Iterator iterator=new HashSet(virtualConcept.getSubConcepts()).iterator();
                while (iterator.hasNext()) {
                    VirtualConcept virtualSubConcept=(VirtualConcept)iterator.next();
                    if (virtualSubConcept.getSuperConcepts().size()==1)
                        EvolutionChangeVisitor.this.visit(new RemoveEntity(m_oimodel,getCurrentEvent(),virtualSubConcept.getConcept()));
                    else
                        EvolutionChangeVisitor.this.visit(new RemoveSubConcept(m_oimodel,getCurrentEvent(),virtualConcept.getConcept(),virtualSubConcept.getConcept()));
                }
                prepareRemoveLeafConcept(virtualConcept);
            }
            else {
                // reconnect subconcepts
                propagateProperties(virtualConcept);
                propagateRanges(virtualConcept);
                reconnectChildren(virtualConcept);

                prepareRemoveLeafConcept(virtualConcept);

                detachSubConcepts(virtualConcept);
            }
        }
        /**
         * Visits a property.
         */
        public void visit(Property entity) throws KAONException {
            VirtualProperty virtualProperty=(VirtualProperty)entity;
            prepareInstanceForRemoval((VirtualInstance)virtualProperty.getSpanningInstance());

            // resolve all property instances
            if (m_evolutionParameters.getInstanceConsistency()==EvolutionParameters.INSTANCE_CONSISTENCY_ENFORCE)
                resolvePropertyInstances(virtualProperty);

            // resolve subproperties
            Iterator iterator=new HashSet(virtualProperty.getSubProperties()).iterator();
            while (iterator.hasNext()) {
                VirtualProperty virtualSubProperty=(VirtualProperty)iterator.next();
                applyChange(new RemoveSubProperty(m_oimodel,getCurrentEvent(),virtualProperty.getProperty(),virtualSubProperty.getProperty()));
                if (m_evolutionParameters.getOrphanedProperties()==EvolutionParameters.ORPHANED_PROPERTIES_DELETE)
                    EvolutionChangeVisitor.this.applyChange(new RemoveEntity(m_oimodel,getCurrentEvent(),virtualSubProperty.getProperty()));
                else if (m_evolutionParameters.getOrphanedProperties()==EvolutionParameters.ORPHANED_PROPERTIES_RECONNECT_TO_SUPER) {
                    Iterator superProperties=virtualProperty.getSuperProperties().iterator();
                    while (superProperties.hasNext()) {
                        VirtualProperty virtualSuperProperty=(VirtualProperty)superProperties.next();
                        applyChange(new AddSubProperty(m_oimodel,getCurrentEvent(),virtualSuperProperty.getProperty(),virtualSubProperty.getProperty()));
                    }
                }
            }
            // detach from superproperties
            iterator=new HashSet(virtualProperty.getSuperProperties()).iterator();
            while (iterator.hasNext()) {
                VirtualProperty virtualSuperProperty=(VirtualProperty)iterator.next();
                applyChange(new RemoveSubProperty(m_oimodel,getCurrentEvent(),virtualSuperProperty.getProperty(),virtualProperty.getProperty()));
            }
            // detach all domain concepts
            iterator=new HashSet(virtualProperty.getDomainConcepts()).iterator();
            while (iterator.hasNext()) {
                VirtualConcept virtualDomainConcept=(VirtualConcept)iterator.next();
                applyChange(new RemovePropertyDomain(m_oimodel,getCurrentEvent(),virtualProperty.getProperty(),virtualDomainConcept.getConcept()));
            }
            // detach all range concepts
            iterator=new HashSet(virtualProperty.getRangeConcepts()).iterator();
            while (iterator.hasNext()) {
                VirtualConcept virtualRangeConcept=(VirtualConcept)iterator.next();
                applyChange(new RemovePropertyRange(m_oimodel,getCurrentEvent(),virtualProperty.getProperty(),virtualRangeConcept.getConcept()));
            }
            // if it is an attribute, switch it off
            if (virtualProperty.isAttribute())
                applyChange(new SetPropertyIsAttribute(m_oimodel,getCurrentEvent(),virtualProperty.getProperty(),false));
            // remove metaproperites from a property
            if (virtualProperty.isSymmetric())
                applyChange(new SetPropertySymmetric(m_oimodel,getCurrentEvent(),virtualProperty.getProperty(),false));
            if (virtualProperty.isTransitive())
                applyChange(new SetPropertyTransitive(m_oimodel,getCurrentEvent(),virtualProperty.getProperty(),false));
            // remove an inverse propertyship from a property
            if (virtualProperty.getInverseProperty()!=null)
                applyChange(new SetNoInverseProperties(m_oimodel,getCurrentEvent(),virtualProperty.getProperty(),((VirtualProperty)virtualProperty.getInverseProperty()).getProperty()));
        }
        /**
         * Visits an instance.
         */
        public void visit(Instance entity) throws KAONException {
            VirtualInstance virtualInstance=(VirtualInstance)entity;
            if (virtualInstance.getSpanningConcept().isInOIModel())
                throw new KAONException("Cannot remove an instance from the model if it has a spanning concept in the model as well.");
            if (virtualInstance.getSpanningProperty().isInOIModel())
                throw new KAONException("Cannot remove an instance from the model if it has a spanning property in the model as well.");
            // remove all property instances and parent concepts
            prepareInstanceForRemoval(virtualInstance);
        }
        /**
         * Visits a lexical entry.
         */
        public void visit(LexicalEntry entity) throws KAONException {
            visit((VirtualInstance)entity);
        }
        /**
         * Performs propagation of properties to subconcepts.
         *
         * @param virtualConcept                virtualConcept that should be deleted
         */
        protected void propagateProperties(VirtualConcept virtualConcept) throws KAONException {
            Set propertiesToPropagate=null;
            if (m_evolutionParameters.getPropertyPropagation()==EvolutionParameters.PROPERTY_PROPAGATION_ALL && m_evolutionParameters.getOrphanedConcepts()==EvolutionParameters.ORPHANED_CONCEPTS_RECONNECT_TO_ROOT)
                propertiesToPropagate=virtualConcept.getAllPropertiesFromConcept();
            else if (m_evolutionParameters.getPropertyPropagation()!=EvolutionParameters.PROPERTY_PROPAGATION_NO)
                propertiesToPropagate=virtualConcept.getPropertiesFromConcept();
            if (propertiesToPropagate!=null) {
                Iterator properties=new HashSet(propertiesToPropagate).iterator();
                while(properties.hasNext()) {
                    VirtualProperty virtualProperty=(VirtualProperty)properties.next();
                    Iterator subConcepts=virtualConcept.getSubConcepts().iterator();
                    while (subConcepts.hasNext()) {
                        VirtualConcept virtualSubConcept=(VirtualConcept)subConcepts.next();
                        applyChange(new AddPropertyDomain(m_oimodel,getCurrentEvent(),virtualProperty.getProperty(),virtualSubConcept.getConcept()));
                    }
                }
            }
            // remove properties of the concept
            Iterator properties=new HashSet(virtualConcept.getPropertiesFromConcept()).iterator();
            while(properties.hasNext()) {
                VirtualProperty virtualProperty=(VirtualProperty)properties.next();
                EvolutionChangeVisitor.this.visit(new RemovePropertyDomain(m_oimodel,getCurrentEvent(),virtualProperty.getProperty(),virtualConcept.getConcept()));
            }
        }
        /**
         * Resolve propertiess whose range is concept that is being deleted
         *
         * @param virtualConcept                 concepts that is being deleted
         */
        protected void propagateRanges(VirtualConcept virtualConcept) throws KAONException {
            Iterator iterator=new HashSet(virtualConcept.getPropertiesToConcept()).iterator();
            while (iterator.hasNext()) {
                VirtualProperty virtualProperty=(VirtualProperty)iterator.next();
                EvolutionChangeVisitor.this.visit(new RemovePropertyRange(m_oimodel,getCurrentEvent(),virtualProperty.getProperty(),virtualConcept.getConcept()));
            }
        }
        /**
         * This method removes a concept from the ontology, provided that the concept doesn't have subconcept.
         *
         * @param virtualConcept                virtualConcept that should be deleted
         */
        protected void prepareRemoveLeafConcept(VirtualConcept virtualConcept) throws KAONException {
            // resolve the consistency of the instances
            if (m_evolutionParameters.getInstanceConsistency()==EvolutionParameters.INSTANCE_CONSISTENCY_ENFORCE) {
                // reconnect instances if needed
                if (m_evolutionParameters.getOrphanedInstances()==EvolutionParameters.ORPHANED_INSTANCES_RECONNECT_TO_SUPER) {
                    Iterator instancesIterator=new HashSet(virtualConcept.getInstances()).iterator();
                    while (instancesIterator.hasNext()) {
                        boolean found =false;
                        VirtualInstance virtualInstance=(VirtualInstance)instancesIterator.next();
                        Iterator parentsIterator=virtualConcept.getSuperConcepts().iterator();
                        while(parentsIterator.hasNext()) {
                            VirtualConcept virtualParent=(VirtualConcept) parentsIterator.next();
                            if (!virtualInstance.getParentConcepts().contains(virtualParent))
                                EvolutionChangeVisitor.this.visit(new AddInstanceOf(m_oimodel, getCurrentEvent(), virtualParent.getConcept(), virtualInstance.getInstance()));
                            else {
                                if (virtualParent.equals(m_virtualOIModel.getRootConcept()))
                                    found=true;
                            }
                        }
                        if (found) {
                            //the Root concept and the concept that has to be deleted are parents
                            if (virtualInstance.getParentConcepts().size()==2)
                                EvolutionChangeVisitor.this.visit(new RemoveEntity(m_oimodel,getCurrentEvent(), virtualInstance.getInstance()));
                            else
                                EvolutionChangeVisitor.this.visit(new RemoveInstanceOf(m_oimodel,getCurrentEvent(),virtualConcept.getConcept(), virtualInstance.getInstance()));
                        }
                        else
                            EvolutionChangeVisitor.this.visit(new RemoveInstanceOf(m_oimodel,getCurrentEvent(),virtualConcept.getConcept(), virtualInstance.getInstance()));
                    }
                }
                // remove instances if needed
                else {
                    Iterator instancesIterator=new HashSet(virtualConcept.getInstances()).iterator();
                    while (instancesIterator.hasNext()) {
                        VirtualInstance virtualInstance=(VirtualInstance)instancesIterator.next();
                        if (m_evolutionParameters.getOrphanedInstances()==EvolutionParameters.ORPHANED_INSTANCES_DELETE)
                            EvolutionChangeVisitor.this.visit(new RemoveEntity(m_oimodel,getCurrentEvent(),virtualInstance.getInstance()));
                        else {
                            // remove only instances that do not have any parent concept any more
                            Set concepts=virtualInstance.getParentConcepts();
                            if (concepts.size()==2)
                                EvolutionChangeVisitor.this.visit(new RemoveEntity(m_oimodel,getCurrentEvent(),virtualInstance.getInstance()));
                            else
                                EvolutionChangeVisitor.this.visit(new RemoveInstanceOf(m_oimodel,getCurrentEvent(),virtualConcept.getConcept(), virtualInstance.getInstance()));
                        }
                    }
                }
            }
            // remove properties that point to the concept
            Iterator properties=new HashSet(virtualConcept.getPropertiesToConcept()).iterator();
            while (properties.hasNext()) {
                VirtualProperty virtualProperty=(VirtualProperty)properties.next();
                EvolutionChangeVisitor.this.visit(new RemovePropertyRange(m_oimodel,getCurrentEvent(),virtualProperty.getProperty(),virtualConcept.getConcept()));
            }
            // remove properties that point from the concept
            properties=new HashSet(virtualConcept.getPropertiesFromConcept()).iterator();
            while (properties.hasNext()) {
                VirtualProperty virtualProperty=(VirtualProperty)properties.next();
                EvolutionChangeVisitor.this.visit(new RemovePropertyDomain(m_oimodel,getCurrentEvent(),virtualProperty.getProperty(),virtualConcept.getConcept()));
            }
            // resolve concept hierarchy
            Iterator iterator=new HashSet(virtualConcept.getSuperConcepts()).iterator();
            while (iterator.hasNext()) {
                VirtualConcept virtualParentConcept=(VirtualConcept)iterator.next();
                if (virtualConcept.getSuperSubConceptOIModelOriginal(virtualParentConcept)==m_oimodel)
                    applyChange(new RemoveSubConcept(m_oimodel,getCurrentEvent(),virtualParentConcept.getConcept(),virtualConcept.getConcept()));
            }
        }
        /**
         * Perform reconnections of children of a given concept according to evolution parameters.
         *
         * @param virtualConcept                concept whose children are being reconnected
         */
        protected void reconnectChildren(VirtualConcept virtualConcept) throws KAONException {
            Iterator iterator=new HashSet(virtualConcept.getSubConcepts()).iterator();
            while (iterator.hasNext()) {
                VirtualConcept virtualSubConcept=(VirtualConcept)iterator.next();
                if (virtualSubConcept.getSuperSubConceptOIModelOriginal(virtualConcept)==m_oimodel) {
                    applyChange(new RemoveSubConcept(m_oimodel,getCurrentEvent(),virtualConcept.getConcept(),virtualSubConcept.getConcept()));
                    if (m_evolutionParameters.getOrphanedConcepts()==EvolutionParameters.ORPHANED_CONCEPTS_RECONNECT_TO_ROOT) {
                        VirtualConcept virtualSuperConcept=(VirtualConcept)m_virtualOIModel.getRootConcept();
                        if (!virtualSubConcept.isSubConceptOf(virtualSuperConcept))
                            applyChange(new AddSubConcept(m_oimodel,getCurrentEvent(),virtualSuperConcept.getConcept(),virtualSubConcept.getConcept()));
                    }
                    else {
                        Iterator superConceptsIterator=virtualConcept.getSuperConcepts().iterator();
                        while (superConceptsIterator.hasNext()) {
                            VirtualConcept virtualSuperConcept=(VirtualConcept)superConceptsIterator.next();
                            if (!virtualSubConcept.isSubConceptOf(virtualSuperConcept))
                                applyChange(new AddSubConcept(m_oimodel,getCurrentEvent(),virtualSuperConcept.getConcept(),virtualSubConcept.getConcept()));
                        }
                    }
                }
            }
        }
        /**
         * Detach subconcepts of given concept.
         *
         * @param virtualConcept                concept whose subconcepts will be detached
         */
        protected void detachSubConcepts(VirtualConcept virtualConcept) throws KAONException {
            Iterator iterator=new HashSet(virtualConcept.getSuperConcepts()).iterator();
            while (iterator.hasNext()) {
                VirtualConcept virtualSuperConcept=(VirtualConcept)iterator.next();
                if (virtualConcept.getSuperSubConceptOIModelOriginal(virtualSuperConcept)==m_oimodel)
                    applyChange(new RemoveSubConcept(m_oimodel,getCurrentEvent(),virtualSuperConcept.getConcept(),virtualConcept.getConcept()));
            }
        }
    }
    /**
     * Visits an event for creation of subconcepts.
     */
    public void visit(AddSubConcept event) throws KAONException {
        m_stackOfChanges.push(event);
        VirtualConcept virtualSubConcept=m_virtualOIModel.getConcept(event.getSubConcept());
        assertEntityInOIModel(virtualSubConcept);
        VirtualConcept virtualSuperConcept=m_virtualOIModel.getConcept(event.getSuperConcept());
        assertEntityInOIModel(virtualSuperConcept);
        if (!DistributedEvolutionStrategy.isReplica(m_oimodel)) {
            // perform integrity check
            if (virtualSuperConcept.equals(virtualSubConcept))
                throw new KAONException("A concept cannot be a subconcept of itself.");
            if (virtualSubConcept.isDirectSubConceptOf(virtualSuperConcept))
                throw new KAONException("Concept '"+virtualSubConcept.getURI()+"' is already a subconcept of concept '"+virtualSuperConcept.getURI()+"'.");
            if (virtualSuperConcept.isSubConceptOf(virtualSubConcept))
                throw new KAONException("Making concept '"+virtualSubConcept.getURI()+"' a subconcept of concept '"+virtualSuperConcept.getURI()+"' would create a hierarchy cycle.");
            // check whether there already exists a path between a super- and subconcept
            if (m_evolutionParameters.getConceptHierarchyShape()!=EvolutionParameters.CONCEPT_HIERARCHY_SHAPE_NO_CONSTRAINT)
                resolveConceptHierarchy(virtualSubConcept,virtualSuperConcept);
        }
        // all was OK - apply the event
        applyChange(event);
        m_stackOfChanges.pop();
    }
    /**
     * Checks whether there are multiple paths between two concepts and applies events that bring the hierarchy into desired state.
     *
     * @param virtualSubConcept                 subconcept
     * @param virtualSuperConcept               superconcept
     */
    protected void resolveConceptHierarchy(VirtualConcept virtualSubConcept,VirtualConcept virtualSuperConcept) throws KAONException {
        // if new subconcept is already a child of the superconcept, then don't allow the change
        if (virtualSubConcept.isSubConceptOf(virtualSuperConcept))
            throw new KAONException("Concept '"+virtualSubConcept.getURI()+"' is already indirectly a subconcept of concept '"+virtualSuperConcept.getURI()+"'.");
        // now check whether there is a link that will be between subconcept and superconcept through an alternative path
        Iterator iterator=new HashSet(virtualSubConcept.getSuperConcepts()).iterator();
        while (iterator.hasNext()) {
            VirtualConcept virtualConcept=(VirtualConcept)iterator.next();
            if (virtualSuperConcept.isSubConceptOf(virtualConcept)) {
                if (m_evolutionParameters.getConceptHierarchyShape()==EvolutionParameters.CONCEPT_HIERARCHY_SHAPE_DO_NOT_ALLOW_MULTIPLE_PATHS)
                    throw new KAONException("Concept '"+virtualSubConcept.getURI()+"' is already a direct subconcept of concept '"+virtualConcept.getURI()+"', and making it a subconcept of '"+virtualSuperConcept.getURI()+"' would open antoher link.");
                else {
                    if (virtualSubConcept.getSuperSubConceptOIModelOriginal(virtualConcept)==m_oimodel)
                        applyChange(new RemoveSubConcept(m_oimodel,getCurrentEvent(),virtualConcept.getConcept(),virtualSubConcept.getConcept()));
                }
            }
        }
    }
    /**
     * Visits an event for removal of subconcepts.
     */
    public void visit(RemoveSubConcept event) throws KAONException {
        m_stackOfChanges.push(event);
        VirtualConcept virtualSubConcept=m_virtualOIModel.getConcept(event.getSubConcept());
        VirtualConcept virtualSuperConcept=m_virtualOIModel.getConcept(event.getSuperConcept());
        assertEntityInOIModel(virtualSubConcept);
        assertEntityInOIModel(virtualSuperConcept);
        if (!DistributedEvolutionStrategy.isReplica(m_oimodel)) {
            // test whether there is a direct link to be deleted
            if (!virtualSubConcept.isDirectSubConceptOf(virtualSuperConcept))
                throw new KAONException("Concept '"+virtualSubConcept.getURI()+"' is not a direct subconcept of concept '"+virtualSuperConcept.getURI()+"'.");
            if (virtualSubConcept.getSuperSubConceptOIModelOriginal(virtualSuperConcept)==m_oimodel) {
                // if root concept is the only parent, don't allow operation
                if (virtualSubConcept.getSuperConcepts().size()==1 && virtualSuperConcept.equals(m_virtualOIModel.getRootConcept()))
                    throw new KAONException("Cannot detach concept '"+virtualSubConcept.getURI()+"' from root, since root is the single parent.");
                resolveProperties(virtualSuperConcept,virtualSubConcept);
                applyChange(event);
                // if this was the only parent of the subconcept, then attach the subconcept to root
                if (virtualSubConcept.getSuperConcepts().size()==0)
                    applyChange(new AddSubConcept(m_oimodel,getCurrentEvent(),((VirtualConcept)m_virtualOIModel.getRootConcept()).getConcept(),virtualSubConcept.getConcept()));
            }
            else
                applyChange(event);
        }
        else
            applyChange(event);

        m_stackOfChanges.pop();
    }
    /**
     * Resolves properties when hierarchy link between concepts is deleted.
     *
     * @param virtualSuperConcept           superconcept
     * @param virtualSubConcept             subconcept
     */
    protected void resolveProperties(VirtualConcept virtualSuperConcept,VirtualConcept virtualSubConcept) throws KAONException {
        Set propertiesToPropagate=null;
        // propagate properties to subconcepts
        if (m_evolutionParameters.getPropertyPropagation()==EvolutionParameters.PROPERTY_PROPAGATION_ALL)
            propertiesToPropagate=virtualSuperConcept.getAllPropertiesFromConcept();
        else if (m_evolutionParameters.getPropertyPropagation()==EvolutionParameters.PROPERTY_PROPAGATION_EXPLICIT)
            propertiesToPropagate=virtualSuperConcept.getPropertiesFromConcept();
        if (propertiesToPropagate!=null) {
            Iterator iterator=propertiesToPropagate.iterator();
            while(iterator.hasNext()) {
                VirtualProperty virtualProperty=(VirtualProperty)iterator.next();
                applyChange(new AddPropertyDomain(m_oimodel,getCurrentEvent(),virtualProperty.getProperty(),virtualSubConcept.getConcept()));
            }
        }
        // remove adequate property instances
        if (m_evolutionParameters.getPropertyPropagation()!=EvolutionParameters.PROPERTY_PROPAGATION_ALL && m_evolutionParameters.getInstanceConsistency()==EvolutionParameters.INSTANCE_CONSISTENCY_ENFORCE) {
            // find properties that are not propagated
            Set notPropagatedProperties=new HashSet(virtualSuperConcept.getAllPropertiesFromConcept());
            if (m_evolutionParameters.getPropertyPropagation()!=EvolutionParameters.PROPERTY_PROPAGATION_NO)
                notPropagatedProperties.removeAll(virtualSuperConcept.getPropertiesFromConcept());
            // remove from the set of not propagated properties those properties that are inherited through some other parent
            Iterator iterator=virtualSubConcept.getSuperConcepts().iterator();
            while (iterator.hasNext()) {
                VirtualConcept virtualParentConcept=(VirtualConcept)iterator.next();
                if (!virtualParentConcept.equals(virtualSuperConcept))
                    notPropagatedProperties.removeAll(virtualParentConcept.getAllPropertiesFromConcept());
            }
            // go through all properties not propagated and remove property instances if needed
            iterator=notPropagatedProperties.iterator();
            while (iterator.hasNext()) {
                VirtualProperty virtualProperty=(VirtualProperty)iterator.next();
                Iterator propertyInstances=new HashSet(virtualProperty.getPropertyInstances()).iterator();
                while (propertyInstances.hasNext()) {
                    VirtualPropertyInstance virtualPropertyInstance=(VirtualPropertyInstance)propertyInstances.next();
                    if (virtualPropertyInstance.getSourceInstance().isParent(virtualSubConcept))
                        applyChange(new RemovePropertyInstance(m_oimodel,getCurrentEvent(),virtualPropertyInstance.getPropertyInstance()));
                }
            }
        }
    }
    /**
     * Visits an event for adding a domain to the property.
     */
    public void visit(AddPropertyDomain event) throws KAONException {
        m_stackOfChanges.push(event);
        VirtualConcept virtualConcept=m_virtualOIModel.getConcept(event.getConcept());
        VirtualProperty virtualProperty=m_virtualOIModel.getProperty(event.getProperty());
        assertEntityInOIModel(virtualConcept);
        assertEntityInOIModel(virtualProperty);
        if (!DistributedEvolutionStrategy.isReplica(m_oimodel)) {
            if (virtualProperty.isDomainConcept(virtualConcept))
                throw new KAONException("Concept '"+virtualConcept.getURI()+"' is already a domain concept of property '"+virtualProperty.getURI()+"'.");
            // if it is not allowed to have a concept and its subconcept in the domain then...
            if (m_evolutionParameters.getDomainRangeShape()==EvolutionParameters.DOMAIN_RANGE_SHAPE_DISALLOW_REPEATED_SUBCONCEPTS) {
                // ...if the concept I'm adding to the domain is a subconcept of some domain concept, this is an error
                // ...if the concept I'm adding to the domain is a subperconcept of some domain concept, remove the current domain concept
                Iterator iterator=new HashSet(virtualProperty.getDomainConcepts()).iterator();
                while (iterator.hasNext()) {
                    VirtualConcept virtualDomainConcept=(VirtualConcept)iterator.next();
                    if (virtualConcept.isSubConceptOf(virtualDomainConcept))
                        throw new KAONException("Domain of property '"+virtualProperty.getURI()+"' already contains concept '"+virtualDomainConcept.getURI()+"' that is the superconcept of concept '"+virtualConcept.getURI()+"' that is being added to the domain.");
                    if (virtualDomainConcept.isSubConceptOf(virtualConcept))
                        applyChange(new RemovePropertyDomain(m_oimodel,getCurrentEvent(),virtualProperty.getProperty(),virtualDomainConcept.getConcept()));
                }
            }
        }
        // finally apply the original change
        applyChange(event);
        m_stackOfChanges.pop();
    }
    /**
     * Visits an event for removing a domain from the property.
     */
    public void visit(RemovePropertyDomain event) throws KAONException  {
        m_stackOfChanges.push(event);
        VirtualConcept virtualConcept=m_virtualOIModel.getConcept(event.getConcept());
        VirtualProperty virtualProperty=m_virtualOIModel.getProperty(event.getProperty());
        assertEntityInOIModel(virtualConcept);
        assertEntityInOIModel(virtualProperty);
        if (!DistributedEvolutionStrategy.isReplica(m_oimodel)) {
            if (!virtualProperty.isDomainConcept(virtualConcept))
                throw new KAONException("Concept '"+virtualConcept.getURI()+"' is not in the domain of property '"+virtualProperty.getURI()+"'.");
            // adapt instances
            if (m_evolutionParameters.getInstanceConsistency()==EvolutionParameters.INSTANCE_CONSISTENCY_ENFORCE)
                removePropertyInstancesFromConcept(virtualConcept,virtualProperty);
            // finally apply the original change
            applyChange(event);
            // if the property has no more domains domain, then remove it
            if (m_evolutionParameters.getPropertyDomainEmpty()==EvolutionParameters.PROPERTY_DOMAIN_EMPTY_DISALLOW && virtualProperty.getDomainConcepts().isEmpty()) {
                if(isCurrentEventFromThisOIModel())
                    visit(new RemoveEntity(m_oimodel,getCurrentEvent(),virtualProperty.getProperty()));
            }
        }
        else
            applyChange(event);
        m_stackOfChanges.pop();
    }
    /**
     * Removes all instances of given property that are pointing from instances of given concept.
     *
     * @param virtualConcept          concept for whose instances are property instances removed
     * @param virtualProperty         property whose instances are removed
     */
    private void removePropertyInstancesFromConcept(VirtualConcept virtualConcept,VirtualProperty virtualProperty) throws KAONException {
        Iterator propertyInstances=new HashSet(virtualProperty.getPropertyInstances()).iterator();
        while (propertyInstances.hasNext()) {
            VirtualPropertyInstance virtualPropertyInstance=(VirtualPropertyInstance)propertyInstances.next();
            if (virtualPropertyInstance.getSourceInstance().isParent(virtualConcept))
                applyChange(new RemovePropertyInstance(m_oimodel,getCurrentEvent(),virtualPropertyInstance.getPropertyInstance()));
        }
    }
    /**
     * Removes all instances of given property that are pointing to instances of given concept.
     *
     * @param virtualConcept          concept for whose instances are property instances removed
     * @param virtualProperty         property whose instances are removed
     */
    private void removePropertyInstancesToConcept(VirtualConcept virtualConcept,VirtualProperty virtualProperty) throws KAONException {
        Iterator propertyInstances=new HashSet(virtualProperty.getPropertyInstances()).iterator();
        while (propertyInstances.hasNext()) {
            VirtualPropertyInstance virtualPropertyInstance=(VirtualPropertyInstance)propertyInstances.next();
            if (virtualPropertyInstance.getTargetValue() instanceof VirtualInstance) {
                VirtualInstance targetInstance=(VirtualInstance)virtualPropertyInstance.getTargetValue();
                if (targetInstance.isParent(virtualConcept))
                    applyChange(new RemovePropertyInstance(m_oimodel,getCurrentEvent(),virtualPropertyInstance.getPropertyInstance()));
            }
        }
    }
    /**
     * Resolves all instances of some property.
     *
     * @param virtualProperty               virtual property whose range is changed
     */
    protected void resolvePropertyInstances(VirtualProperty virtualProperty) throws KAONException {
        Iterator propertyInstances=new HashSet(virtualProperty.getPropertyInstances()).iterator();
        while(propertyInstances.hasNext()) {
            VirtualPropertyInstance virtualPropertyInstance=(VirtualPropertyInstance)propertyInstances.next();
                if (m_evolutionParameters.getOrphanedPropertyInstances()==EvolutionParameters.ORPHANED_PROPERTYINSTANCES_RECONNECT_TO_SUPER) {
                Iterator parentsIterator=virtualProperty.getSuperProperties().iterator();
                while (parentsIterator.hasNext()) {
                    VirtualProperty virtualParent=(VirtualProperty) parentsIterator.next();
                    boolean foundDomain=false;
                    Iterator domainsIterator=virtualParent.getAllDomainConcepts().iterator();
                    while ( (!foundDomain) && (domainsIterator.hasNext()) ) {
                        VirtualConcept domain=(VirtualConcept)domainsIterator.next();
                        if (virtualPropertyInstance.getSourceInstance().getAllParentConcepts().contains(domain))
                            foundDomain=true;
                    }
                    boolean foundRange=false;
                    if (foundDomain) {
                        if (virtualProperty.getProperty().isAttribute()) {
                            if (virtualParent.getProperty().isAttribute())
                                foundRange=true;
                        }
                        else {
                            Iterator rangesIterator=virtualParent.getAllRangeConcepts().iterator();
                            while ( (!foundRange) && (rangesIterator.hasNext()) ) {
                                VirtualConcept range=(VirtualConcept)rangesIterator.next();
                                VirtualInstance target=(VirtualInstance) virtualPropertyInstance.getTargetValue();
                                if (target.getAllParentConcepts().contains(range))
                                    foundRange=true;
                            }
                        }
                    }
                    if (foundRange) {
                        VirtualInstance source=(VirtualInstance) virtualPropertyInstance.getSourceInstance();
                        if (virtualProperty.getProperty().isAttribute()) {
                            String target=(String) virtualPropertyInstance.getTargetValue();
                            applyChange(new AddPropertyInstance(m_oimodel, getCurrentEvent(), virtualParent.getProperty(), source.getInstance(), target));
                        }
                        else {
                            VirtualInstance target=(VirtualInstance) virtualPropertyInstance.getTargetValue();
                            applyChange(new AddPropertyInstance(m_oimodel, getCurrentEvent(), virtualParent.getProperty(), source.getInstance(), target.getInstance()));
                        }
                    }
                }
            }
            //Removes all instances of the property
            applyChange(new RemovePropertyInstance(m_oimodel,getCurrentEvent(),virtualPropertyInstance.getPropertyInstance()));
        }
    }
    /**
     * Visits an event for adding a range to the property.
     */
    public void visit(AddPropertyRange event) throws KAONException {
        m_stackOfChanges.push(event);
        VirtualConcept virtualConcept=m_virtualOIModel.getConcept(event.getConcept());
        VirtualProperty virtualProperty=m_virtualOIModel.getProperty(event.getProperty());
        assertEntityInOIModel(virtualConcept);
        assertEntityInOIModel(virtualProperty);

        if (!DistributedEvolutionStrategy.isReplica(m_oimodel)) {
            // if property was attribute, then remove its attribute status
            if (virtualProperty.isAttribute())
                visit(new SetPropertyIsAttribute(m_oimodel,getCurrentEvent(),virtualProperty.getProperty(),false));
            // test if range concept is already in range
            if (virtualProperty.isRangeConcept(virtualConcept))
                throw new KAONException("Concept '"+virtualConcept.getURI()+"' is already a range concept of property '"+virtualProperty.getURI()+"'.");
            // if it is not allowed to have a concept and its subconcept in the domain then...
            if (m_evolutionParameters.getDomainRangeShape()==EvolutionParameters.DOMAIN_RANGE_SHAPE_DISALLOW_REPEATED_SUBCONCEPTS) {
                // ...if the concept I'm adding to the range is a subconcept of some range concept, this is an error
                // ...if the concept I'm adding to the range is a subperconcept of some range concept, remove the current range concept
                Iterator iterator=new HashSet(virtualProperty.getRangeConcepts()).iterator();
                while (iterator.hasNext()) {
                    VirtualConcept virtualRangeConcept=(VirtualConcept)iterator.next();
                    if (virtualConcept.isSubConceptOf(virtualRangeConcept))
                        throw new KAONException("Range of property '"+virtualProperty.getURI()+"' already contains concept '"+virtualRangeConcept.getURI()+"' that is the superconcept of concept '"+virtualConcept.getURI()+"' that is being added to the domain");
                    if (virtualRangeConcept.isSubConceptOf(virtualConcept))
                        applyChange(new RemovePropertyRange(m_oimodel,getCurrentEvent(),virtualProperty.getProperty(),virtualRangeConcept.getConcept()));
                }
            }
        }
        // finally apply the event
        applyChange(event);
        m_stackOfChanges.pop();
    }
    /**
     * Visits an event for removing a range from the property.
     */
    public void visit(RemovePropertyRange event) throws KAONException {
        m_stackOfChanges.push(event);
        VirtualConcept virtualConcept=m_virtualOIModel.getConcept(event.getConcept());
        VirtualProperty virtualProperty=m_virtualOIModel.getProperty(event.getProperty());
        assertEntityInOIModel(virtualConcept);
        assertEntityInOIModel(virtualProperty);
        if (!DistributedEvolutionStrategy.isReplica(m_oimodel)) {

            if (!virtualProperty.isRangeConcept(virtualConcept))
                throw new KAONException("Concept '"+virtualConcept.getURI()+"' is not in the range of property '"+virtualProperty.getURI()+"'.");
            // adapt instances
            if (m_evolutionParameters.getInstanceConsistency()==EvolutionParameters.INSTANCE_CONSISTENCY_ENFORCE)
                removePropertyInstancesToConcept(virtualConcept,virtualProperty);
            // finally apply the original change
            applyChange(event);
            // if the property has no more ranges, then remove it
            if (m_evolutionParameters.getPropertyRangeEmpty()==EvolutionParameters.PROPERTY_RANGE_EMPTY_DISALLOW && virtualProperty.getRangeConcepts().isEmpty()) {
                if(isCurrentEventFromThisOIModel())
                    visit(new RemoveEntity(m_oimodel,getCurrentEvent(),virtualProperty.getProperty()));
            }
        }
        else
            applyChange(event);
        m_stackOfChanges.pop();
    }
    /**
     * Visits an event for determinig whether property is attirubte.
     */
    public void visit(SetPropertyIsAttribute event) throws KAONException {
        m_stackOfChanges.push(event);
        VirtualProperty virtualProperty=m_virtualOIModel.getProperty(event.getProperty());
        assertEntityInOIModel(virtualProperty);
        if (!DistributedEvolutionStrategy.isReplica(m_oimodel)) {
            if (virtualProperty.isAttribute()==event.getIsAttribute())
                throw new KAONException("Property '"+virtualProperty.getURI()+"' already has required attribute status.");
            if (event.getIsAttribute()) {
                // remove all existing ranges
                Iterator iterator=new HashSet(virtualProperty.getRangeConcepts()).iterator();
                while (iterator.hasNext()) {
                    VirtualConcept virtualRangeConcept=(VirtualConcept)iterator.next();
                    visit(new RemovePropertyRange(m_oimodel,getCurrentEvent(),virtualProperty.getProperty(),virtualRangeConcept.getConcept()));
                }
            }
            else {
                // if turning a property from an attribute into non-attribute, then remove property instances
                if (m_evolutionParameters.getInstanceConsistency()==EvolutionParameters.INSTANCE_CONSISTENCY_ENFORCE)
                    resolvePropertyInstances(virtualProperty);
            }
        }
        // apply the original event
        applyChange(event);
        m_stackOfChanges.pop();
    }
    /**
     * Visits an event for creation of subproperties.
     */
    public void visit(AddSubProperty event) throws KAONException {
        m_stackOfChanges.push(event);
        VirtualProperty virtualSubProperty=m_virtualOIModel.getProperty(event.getSubProperty());
        VirtualProperty virtualSuperProperty=m_virtualOIModel.getProperty(event.getSuperProperty());
        assertEntityInOIModel(virtualSubProperty);
        assertEntityInOIModel(virtualSuperProperty);
        if (!DistributedEvolutionStrategy.isReplica(m_oimodel)) {
            if (virtualSubProperty.isDirectSubPropertyOf(virtualSuperProperty))
                throw new KAONException("Property '"+virtualSubProperty.getURI()+"' is already a subproperty of property '"+virtualSuperProperty.getURI()+"'.");
            if (virtualSuperProperty.isSubPropertyOf(virtualSubProperty))
                throw new KAONException("Making property '"+virtualSubProperty.getURI()+"' a subproperty of property '"+virtualSuperProperty.getURI()+"' would result in property hierarchy cycles.");
        }
        applyChange(event);
        m_stackOfChanges.pop();
    }
    /**
     * Visits an event for removal of subproperties.
     */
    public void visit(RemoveSubProperty event) throws KAONException {
        m_stackOfChanges.push(event);
        VirtualProperty virtualSubProperty=m_virtualOIModel.getProperty(event.getSubProperty());
        VirtualProperty virtualSuperProperty=m_virtualOIModel.getProperty(event.getSuperProperty());
        assertEntityInOIModel(virtualSubProperty);
        assertEntityInOIModel(virtualSuperProperty);
        if (!DistributedEvolutionStrategy.isReplica(m_oimodel)) {
            if (!virtualSubProperty.isDirectSubPropertyOf(virtualSuperProperty))
                throw new KAONException("Property '"+virtualSubProperty.getURI()+"' is not a subproperty of property '"+virtualSuperProperty.getURI()+"'.");
        }
        applyChange(event);
        m_stackOfChanges.pop();
    }
    /**
     * Visits an event for setting the inverse propertyship between properties.
     */
    public void visit(SetInverseProperties event) throws KAONException {
        m_stackOfChanges.push(event);
        VirtualProperty virtualProperty1=m_virtualOIModel.getProperty(event.getProperty1());
        VirtualProperty virtualProperty2=m_virtualOIModel.getProperty(event.getProperty2());
        assertEntityInOIModel(virtualProperty1);
        assertEntityInOIModel(virtualProperty2);
        if (!DistributedEvolutionStrategy.isReplica(m_oimodel)) {
            if (virtualProperty1.getInverseProperty()==virtualProperty2)
                throw new KAONException("Property '"+virtualProperty2.getURI()+"' is alredy an inverse property of property '"+virtualProperty1.getURI()+"'.");
            if (virtualProperty1.getInverseProperty()!=null)
                throw new KAONException("Property '"+virtualProperty1.getURI()+"' has alredy an inverse property '"+virtualProperty1.getInverseProperty().getURI()+"'.");
            if (virtualProperty2.getInverseProperty()!=null)
                throw new KAONException("Property '"+virtualProperty2.getURI()+"' has alredy an inverse property '"+virtualProperty2.getInverseProperty().getURI()+"'.");
        }
        applyChange(event);
        m_stackOfChanges.pop();
    }
    /**
     * Visits an event for removing an inverse propertyship between properties.
     */
    public void visit(SetNoInverseProperties event) throws KAONException {
        m_stackOfChanges.push(event);
        VirtualProperty virtualProperty1=m_virtualOIModel.getProperty(event.getProperty1());
        VirtualProperty virtualProperty2=m_virtualOIModel.getProperty(event.getProperty2());
        assertEntityInOIModel(virtualProperty1);
        assertEntityInOIModel(virtualProperty2);
        if (!DistributedEvolutionStrategy.isReplica(m_oimodel)) {
            if (virtualProperty1.getInverseProperty()!=virtualProperty2)
                throw new KAONException("Property '"+virtualProperty1.getURI()+"' is not inverse of property '"+virtualProperty2.getURI()+"'.");
        }
        applyChange(event);
        m_stackOfChanges.pop();
    }
    /**
     * Visits an event for setting the symmetry flag of the property.
     */
    public void visit(SetPropertySymmetric event) throws KAONException {
        m_stackOfChanges.push(event);
        VirtualProperty virtualProperty=m_virtualOIModel.getProperty(event.getProperty());
        assertEntityInOIModel(virtualProperty);
        if (!DistributedEvolutionStrategy.isReplica(m_oimodel)) {
            if (virtualProperty.isSymmetric()==event.getSymmetric())
                if (event.getSymmetric())
                    throw new KAONException("Property '"+virtualProperty.getURI()+"' is a symmetric property.");
                else
                    throw new KAONException("Property '"+virtualProperty.getURI()+"' is not a symmetric property.");
        }
        applyChange(event);
        m_stackOfChanges.pop();
    }
    /**
     * Visits an event for setting the transitivity flag of the property.
     */
    public void visit(SetPropertyTransitive event) throws KAONException {
        m_stackOfChanges.push(event);
        VirtualProperty virtualProperty=m_virtualOIModel.getProperty(event.getProperty());
        assertEntityInOIModel(virtualProperty);
        if (!DistributedEvolutionStrategy.isReplica(m_oimodel)) {
            if (virtualProperty.isTransitive()==event.getTransitive())
                if (event.getTransitive())
                    throw new KAONException("Property '"+virtualProperty.getURI()+"' is a transitive property.");
                else
                    throw new KAONException("Property '"+virtualProperty.getURI()+"' is not a transitive property.");
        }
        applyChange(event);
        m_stackOfChanges.pop();
    }
    /**
     * Visits an event for setting the minimum cardinality of a property for a concept.
     */
    public void visit(SetMinimumCardinality event) throws KAONException {
        applyChange(event);
    }
    /**
     * Visits an event for setting the maximum cardinality of a property for a concept.
     */
    public void visit(SetMaximumCardinality event) throws KAONException {
        applyChange(event);
    }
    /**
     * Visits an event for making an instance a subinstance of given concept.
     */
    public void visit(AddInstanceOf event) throws KAONException  {
        m_stackOfChanges.push(event);
        VirtualConcept virtualConcept=m_virtualOIModel.getConcept(event.getConcept());
        VirtualInstance virtualInstance=m_virtualOIModel.getInstance(event.getInstance());
        assertEntityInOIModel(virtualConcept);
        assertEntityInOIModel(virtualInstance);
        if (!DistributedEvolutionStrategy.isReplica(m_oimodel)) {
            if (virtualInstance.getParentConcepts().contains(virtualConcept))
                throw new KAONException("Instance '"+virtualInstance.getURI()+"' is already an instance of concept '"+virtualConcept.getURI()+"'.");
        }
        applyChange(event);
        m_stackOfChanges.pop();
    }
    /**
     * Visits an event for making an instance not a subinstance of given concept.
     */
    public void visit(RemoveInstanceOf event) throws KAONException {
        m_stackOfChanges.push(event);
        VirtualConcept virtualConcept=m_virtualOIModel.getConcept(event.getConcept());
        VirtualInstance virtualInstance=m_virtualOIModel.getInstance(event.getInstance());
        assertEntityInOIModel(virtualConcept);
        assertEntityInOIModel(virtualInstance);
        if (!DistributedEvolutionStrategy.isReplica(m_oimodel)) {
            if (!virtualInstance.getParentConcepts().contains(virtualConcept))
                throw new KAONException("Instance '"+virtualInstance.getURI()+"' is not an instance of concept '"+virtualConcept.getURI()+"'.");
            if (virtualConcept.equals(m_virtualOIModel.getRootConcept()))
                throw new KAONException("Each instance must always be an instance of the root concept.");
            resolvePropertyInstances(virtualConcept,virtualInstance);
        }
        applyChange(event);
        m_stackOfChanges.pop();
    }
    /**
     * Resolves relation properties when link between concept and instance is deleted.
     *
     * @param virtualConcept                concept
     * @param virtualInstance               instance
     */
    protected void resolvePropertyInstances(VirtualConcept virtualConcept,VirtualInstance virtualInstance) throws KAONException {
        if (m_evolutionParameters.getInstanceConsistency()==EvolutionParameters.INSTANCE_CONSISTENCY_ENFORCE) {
            // determine the set of remaining parent concepts
            Set allParentVirtualConcepts=new HashSet();
            Iterator iterator=virtualInstance.getParentConcepts().iterator();
            while (iterator.hasNext()) {
                VirtualConcept parentVirtualConcept=(VirtualConcept)iterator.next();
                if (!parentVirtualConcept.equals(virtualConcept)) {
                    allParentVirtualConcepts.add(parentVirtualConcept);
                    allParentVirtualConcepts.addAll(parentVirtualConcept.getAllSuperConcepts());
                }
            }
            // now determine the set of all allowed properties from and to
            Set allowedVirtualPropertiesFrom=new HashSet();
            Set allowedVirtualPropertiesTo=new HashSet();
            iterator=allParentVirtualConcepts.iterator();
            while (iterator.hasNext()) {
                VirtualConcept virtualParentConcept=(VirtualConcept)iterator.next();
                allowedVirtualPropertiesFrom.addAll(virtualParentConcept.getPropertiesFromConcept());
                allowedVirtualPropertiesTo.addAll(virtualParentConcept.getPropertiesToConcept());
            }
            iterator=new HashSet(virtualInstance.getFromPropertyInstances()).iterator();
            while (iterator.hasNext()) {
                VirtualPropertyInstance virtualPropertyInstance=(VirtualPropertyInstance)iterator.next();
                VirtualProperty virtualProperty=virtualPropertyInstance.getVirtualProperty();
                if (!allowedVirtualPropertiesFrom.contains(virtualProperty))
                    applyChange(new RemovePropertyInstance(m_oimodel,getCurrentEvent(),virtualPropertyInstance.getPropertyInstance()));
            }
            iterator=new HashSet(virtualInstance.getToPropertyInstances()).iterator();
            while (iterator.hasNext()) {
                VirtualPropertyInstance virtualPropertyInstance=(VirtualPropertyInstance)iterator.next();
                VirtualProperty virtualProperty=virtualPropertyInstance.getVirtualProperty();
                if (!allowedVirtualPropertiesTo.contains(virtualProperty))
                    applyChange(new RemovePropertyInstance(m_oimodel,getCurrentEvent(),virtualPropertyInstance.getPropertyInstance()));
            }
        }
    }
    /**
     * Visits an event for adding a property instance.
     */
    public void visit(AddPropertyInstance event) throws KAONException {
        m_stackOfChanges.push(event);
        VirtualPropertyInstance virtualPropertyInstance=m_virtualOIModel.getPropertyInstance(event.getPropertyInstance());
        assertEntityInOIModel(virtualPropertyInstance.getVirtualProperty());
        assertEntityInOIModel(virtualPropertyInstance.getVirtualSourceInstance());
        if (!DistributedEvolutionStrategy.isReplica(m_oimodel)) {
            if (virtualPropertyInstance.getTargetValue() instanceof VirtualInstance)
                assertEntityInOIModel((VirtualInstance)virtualPropertyInstance.getTargetValue());
            if (propertyInstanceExists(virtualPropertyInstance))
                throw new KAONException("PropertyInstance already exists.");
            checkPropertyInstanceDomain(virtualPropertyInstance);
            checkPropertyInstanceRange(virtualPropertyInstance);
        }
        applyChange(event);
        m_stackOfChanges.pop();
    }
    /**
     * Checks whether given virtual property instance can be created with respect to the domain constraints of the property.
     *
     * @param virtualPropertyInstance           virtual property instance being checked
     */
    protected void checkPropertyInstanceDomain(VirtualPropertyInstance virtualPropertyInstance) throws KAONException {
        VirtualProperty virtualProperty=virtualPropertyInstance.getVirtualProperty();
        VirtualInstance virtualSourceInstance=virtualPropertyInstance.getVirtualSourceInstance();
        Iterator iterator=virtualSourceInstance.getParentConcepts().iterator();
        while (iterator.hasNext()) {
            VirtualConcept virtualConcept=(VirtualConcept)iterator.next();
            if (virtualConcept.getAllPropertiesFromConcept().contains(virtualProperty))
                return;
        }
        throw new KAONException("Instance '"+virtualSourceInstance.getURI()+"' does not follow the domain constraint of property '"+virtualProperty.getURI()+"'.");
    }
    /**
     * Checks whether given virtual property instance can be created with respect to the range constraints of the property.
     *
     * @param virtualPropertyInstance           virtual property instance being checked
     */
    protected void checkPropertyInstanceRange(VirtualPropertyInstance virtualPropertyInstance) throws KAONException {
        VirtualProperty virtualProperty=virtualPropertyInstance.getVirtualProperty();
        Object target=virtualPropertyInstance.getTargetValue();
        if (target instanceof VirtualInstance) {
            if (virtualProperty.isAttribute())
                throw new KAONException("Property '"+virtualProperty.getURI()+"' is an attribute, so its value cannot contain an instance.");
            VirtualInstance virtualTargetInstance=(VirtualInstance)target;
            Iterator iterator=virtualTargetInstance.getParentConcepts().iterator();
            while (iterator.hasNext()) {
                VirtualConcept virtualConcept=(VirtualConcept)iterator.next();
                if (virtualConcept.getAllPropertiesToConcept().contains(virtualProperty))
                    return;
            }
            throw new KAONException("Instance '"+virtualTargetInstance.getURI()+"' does not follow the range constraint of property '"+virtualProperty.getURI()+"'.");
        }
        else {
            if (!virtualProperty.isAttribute())
                throw new KAONException("Property '"+virtualProperty.getURI()+"' is not an attribute, so its value cannot a literal.");
        }
    }
    /**
     * Visits an event for removing a property instance.
     */
    public void visit(RemovePropertyInstance event) throws KAONException {
        m_stackOfChanges.push(event);
        VirtualPropertyInstance virtualPropertyInstance=m_virtualOIModel.getPropertyInstance(event.getPropertyInstance());
        assertEntityInOIModel(virtualPropertyInstance.getVirtualProperty());
        assertEntityInOIModel(virtualPropertyInstance.getVirtualSourceInstance());
        if (!DistributedEvolutionStrategy.isReplica(m_oimodel)) {
            if (virtualPropertyInstance.getTargetValue() instanceof VirtualInstance)
                assertEntityInOIModel((VirtualInstance)virtualPropertyInstance.getTargetValue());
            if (!propertyInstanceExists(virtualPropertyInstance))
                throw new KAONException("PropertyInstance doesn't exist.");
        }
        applyChange(event);
        m_stackOfChanges.pop();
    }
    /**
     * Checks whether given property instance exists.
     *
     * @param virtualRropertyInstance           virtual property instance being checked
     * @return                                  <code>true</code> if property instance exists
     */
    private boolean propertyInstanceExists(VirtualPropertyInstance virtualPropertyInstance) throws KAONException {
        VirtualProperty virtualProperty=virtualPropertyInstance.getVirtualProperty();
        VirtualInstance virtualSourceInstance=virtualPropertyInstance.getVirtualSourceInstance();
        Set values=virtualSourceInstance.getFromPropertyValues(virtualProperty);
        return values.contains(virtualPropertyInstance.getTargetValue());
    }
    /**
     * Visits an event for setting the value of a property instance.
     */
    public void visit(SetPropertyInstanceValue event) throws KAONException {
        m_stackOfChanges.push(event);
        VirtualPropertyInstance virtualPropertyInstance=m_virtualOIModel.getPropertyInstance(event.getPropertyInstance());
        assertEntityInOIModel(virtualPropertyInstance.getVirtualProperty());
        assertEntityInOIModel(virtualPropertyInstance.getVirtualSourceInstance());
        if (!DistributedEvolutionStrategy.isReplica(m_oimodel)) {
            if (virtualPropertyInstance.getTargetValue() instanceof VirtualInstance)
                assertEntityInOIModel((VirtualInstance)virtualPropertyInstance.getTargetValue());
            checkPropertyInstanceDomain(virtualPropertyInstance);
            checkPropertyInstanceRange(virtualPropertyInstance);
        }
        applyChange(event);
        m_stackOfChanges.pop();
    }
    /**
     * Visits an event for changing the value of a property instance.
     */
    public void visit(ChangePropertyInstanceValue event) throws KAONException {
        m_stackOfChanges.push(event);
        VirtualPropertyInstance virtualPropertyInstance=m_virtualOIModel.getPropertyInstance(event.getPropertyInstance());
        assertEntityInOIModel(virtualPropertyInstance.getVirtualProperty());
        assertEntityInOIModel(virtualPropertyInstance.getVirtualSourceInstance());
        if (!DistributedEvolutionStrategy.isReplica(m_oimodel)) {
            if (virtualPropertyInstance.getTargetValue() instanceof VirtualInstance)
                assertEntityInOIModel((VirtualInstance)virtualPropertyInstance.getTargetValue());
            if (!propertyInstanceExists(virtualPropertyInstance))
                throw new KAONException("PropertyInstance doesn't exist.");
            VirtualPropertyInstance newVirtualPropertyInstance=(VirtualPropertyInstance)m_virtualOIModel.getPropertyInstance(event.getPropertyInstance().getProperty(),event.getPropertyInstance().getSourceInstance(),event.getNewTargetValue());
            if (newVirtualPropertyInstance.getTargetValue() instanceof VirtualInstance)
                assertEntityInOIModel((VirtualInstance)newVirtualPropertyInstance.getTargetValue());
            if (propertyInstanceExists(newVirtualPropertyInstance))
                throw new KAONException("PropertyInstance already exists.");
            checkPropertyInstanceDomain(newVirtualPropertyInstance);
            checkPropertyInstanceRange(newVirtualPropertyInstance);
        }
        applyChange(event);
        m_stackOfChanges.pop();
    }
    /**
     * Visits an event for making a model included in another model.
     */
    public void visit(AddIncludedOIModel event) {
    }
    /**
     * Visits an event for making a model not included in another model.
     */
    public void visit(RemoveIncludedOIModel event) {
    }
    /**
     * Gets the virtual entity for the entity.
     *
     * @param entity                entity
     * @return                      virtual entity for given entity
     */
    protected VirtualEntity getVirtualEntity(Entity entity) throws KAONException {
        if (entity instanceof Concept)
            return m_virtualOIModel.getConcept((Concept)entity);
        else if (entity instanceof Property)
            return m_virtualOIModel.getProperty((Property)entity);
        else if (entity instanceof Instance)
            return m_virtualOIModel.getInstance((Instance)entity);
        else if (entity instanceof LexicalEntry)
            return m_virtualOIModel.getLexicalEntry((LexicalEntry)entity);
        else
            throw new KAONException("Invalid original entity type.");
    }
    /**
     * Checks whether the entity is in OI-Model
     *
     * @param entity                entity that is checked
     * @throws KAONException        thrown if given entity is not in OI-model
     */
    protected void assertEntityInOIModel(VirtualEntity entity) throws KAONException {
        if (!entity.isInOIModel())
            throw new KAONException("Entity with URI '"+entity.getURI()+"' is not in OI model.");
    }
    /**
     * Checks whether the entity is not in OI-Model
     *
     * @param entity                entity that is checked
     * @throws KAONException        thrown if given entity is in OI-model
     */
    protected void assertEntityNotInOIModel(VirtualEntity entity) throws KAONException {
        if (entity.isInOIModel())
            throw new KAONException("Entity with URI '"+entity.getURI()+"' is in OI model.");
    }

    /**
     * Tests if current event is from this model.
     *
     * @return                          <code>true</code> if current event is generated in this model
     */
    protected boolean isCurrentEventFromThisOIModel() throws KAONException  {
        if (getCurrentEvent().getOIModel().equals(m_oimodel))
            return true;
        else
            return false;
    }

    /**
     * Returns last event from stack
     *
     * @throws KAONException        thrown if stack is empty
     */
    protected ChangeEvent getCurrentEvent() throws KAONException  {
        if (m_stackOfChanges.empty())
            throw new KAONException("There is no previous event.");
        return (ChangeEvent) m_stackOfChanges.peek();
    }
    /**
     * Returns oimodel
     *
     */
    public OIModel getOIModel() {
        return m_oimodel;
    }

    public static class ChangeVisitorHolder {
        public EvolutionChangeVisitor m_changeVisitor;
        public ChangeVisitorHolder m_next;
    }
}
