package edu.unika.aifb.kaon.defaultevolution;

import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;
import java.util.Map;
import java.util.HashMap;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.util.*;

/**
 * Strategy for handling evolution issues for dependent ontologies.
 *
 * @author Ljiljana Stojanovic (Ljiljana.Stojanovic@fzi.de)
 * @author Boris Motik (boris.motik@fzi.de)
 */
public class DependentEvolutionStrategyImpl implements DependentEvolutionStrategy {
    /** KAON connection that finds all local OI-models. */
    protected KAONConnection m_kaonConnection;
    /** Map that contains an evolution parameters for each oimodel. */
    protected Map m_evolutionParametersByOIModel;

    /**
     * Creates an instance of this class.
     *
     * @param kaonConnection            KAON connection for which this strategy provides distributed evolution
     */
    public DependentEvolutionStrategyImpl(KAONConnection kaonConnection) {
        m_kaonConnection=kaonConnection;
        m_evolutionParametersByOIModel=new HashMap();
    }
    /**
     * Computes the changes needed to keep the connection in a consistent state.
     *
     * @param changes               the starting changes
     * @return                      the extended list of changes
     * @throws KAONException        thrown if there is an error
     */
    public List computeRequestedChanges(List changes) throws KAONException {
        List sortedOIModels=OIModels.topologicallySortOIModels(m_kaonConnection.getAllOIModels());
        List requestedChanges=new LinkedList();
        Map changeVisitorHolders=new HashMap();
        EvolutionChangeVisitor.ChangeVisitorHolder lastHolder=null;
        Iterator iterator=sortedOIModels.iterator();
        while (iterator.hasNext()) {
            OIModel oimodel=(OIModel)iterator.next();
            EvolutionChangeVisitor.ChangeVisitorHolder newHolder=new EvolutionChangeVisitor.ChangeVisitorHolder();
            changeVisitorHolders.put(oimodel,newHolder);
            EvolutionParameters evolutionParameters=getEvolutionParameters(oimodel);
            newHolder.m_changeVisitor=new EvolutionChangeVisitor(evolutionParameters,oimodel,changeVisitorHolders,requestedChanges);
            if (lastHolder!=null)
                lastHolder.m_next=newHolder;
            lastHolder=newHolder;
        }
        // Important! This map will contain strong references to visitors, which will contain strong references to objects.
        // If this map is not kept in scope until events are processed, then loaded may be garbage-collected
        // before the event relating to the object is actually dispatched.
        Map loadVisitors=null;
        if ((m_kaonConnection.getCapabilities() & OIModel.CAPABILITY_SUPPORTS_OPTIMIZED_LOADING)!=0) {
            loadVisitors=new HashMap();
            preloadObjects(changeVisitorHolders,changes,loadVisitors);
        }
        iterator=changes.iterator();
        while (iterator.hasNext()) {
            ChangeEvent changeEvent=(ChangeEvent)iterator.next();
            EvolutionChangeVisitor.ChangeVisitorHolder holder=(EvolutionChangeVisitor.ChangeVisitorHolder)changeVisitorHolders.get(changeEvent.getOIModel());
            if (holder==null)
                throw new KAONException("Internal error: invalid OI-model passed in the event.");
            changeEvent.accept(holder.m_changeVisitor);
        }
        return requestedChanges;
    }
    /**
     * Preloads ggiven objects.
     *
     * @param changeVisitorHolders  the map of change visitor holders
     * @param changes               the list of changes
     * @param loadVisitors          the load visitors
     * @throws KAONException        thrown if there is an error
     */
    protected void preloadObjects(Map changeVisitorHolders,List changes,Map loadVisitors) throws KAONException {
        Iterator iterator=changes.iterator();
        while (iterator.hasNext()) {
            ChangeEvent changeEvent=(ChangeEvent)iterator.next();
            OIModel oimodel=changeEvent.getOIModel();
            EvolutionChangeVisitor.ChangeVisitorHolder holder=(EvolutionChangeVisitor.ChangeVisitorHolder)changeVisitorHolders.get(oimodel);
            if (holder==null)
                throw new KAONException("Internal error: invalid OI-model passed in the event.");
            while (holder!=null) {
                OIModel currentOIModel=holder.m_changeVisitor.getOIModel();
                if (oimodel==currentOIModel || oimodel.getAllIncludedByOIModels().contains(currentOIModel)) {
                    EvolutionLoadVisitor loadVisitor=(EvolutionLoadVisitor)loadVisitors.get(currentOIModel);
                    if (loadVisitor==null) {
                        loadVisitor=new EvolutionLoadVisitor(currentOIModel);
                        loadVisitors.put(currentOIModel,loadVisitor);
                    }
                    changeEvent.accept(loadVisitor);
                }
                holder=holder.m_next;
            }
        }
        iterator=loadVisitors.values().iterator();
        while (iterator.hasNext()) {
            EvolutionLoadVisitor loadVisitor=(EvolutionLoadVisitor)iterator.next();
            loadVisitor.performLoading();
        }
    }
    /**
     * Finds evolution parameters for a given OIModel.
     *
     * @param oimodel               oimodel
     * @return                      evolution parameters
     * @throws KAONException        thrown if the evolution strategy for the model hasn't been set
     */
    public EvolutionParameters getEvolutionParameters(OIModel oimodel) throws KAONException {
        EvolutionParameters evolutionParameters=(EvolutionParameters)m_evolutionParametersByOIModel.get(oimodel);
        if (evolutionParameters==null)
            throw new KAONException("Evolution parameters hasn't been set for OI-model '"+oimodel.getLogicalURI()+"'");
        return evolutionParameters;
    }
    /**
     * Sets the evolution parameters for given OI-model.
     *
     * @param oimodel               the OI-model
     * @param evolutionParameters   the evolution parameters for given OI-model
     */
    public void setEvolutionStrategy(OIModel oimodel,EvolutionParameters evolutionParameters) {
        m_evolutionParametersByOIModel.put(oimodel,evolutionParameters);
    }
}
