package edu.unika.aifb.kaon.api.util;

import java.util.Set;
import java.util.List;
import java.util.LinkedList;
import java.util.Map;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Collections;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.oimodel.*;

/**
 * A utility class for copying one model into another.
 *
 * @author Raphael Volz (volz@aifb.uni-karlsruhe.de)
 * @author Boris Motik (boris.motik@fzi.de)
 */
public class OIModelReplicator extends OIModelProcessor {
    /** Phase of loading concepts. */
    public static final int PHASE_LOAD_CONCEPTS=1;
    /** Phase of copying concepts. */
    public static final int PHASE_COPY_CONCEPTS=2;
    /** Phase of loading properties. */
    public static final int PHASE_LOAD_PROPERTIES=3;
    /** Phase of copying properties. */
    public static final int PHASE_COPY_PROPERTIES=4;
    /** Phase of loading instances. */
    public static final int PHASE_LOAD_INSTANCES=5;
    /** Phase of copying instances. */
    public static final int PHASE_COPY_INSTANCES=6;

    /** The map of logical to physical URI mappings. */
    protected Map m_logicalToPhysicalURIs;
    /** The source OI-model. */
    protected OIModel m_sourceOIModel;
    /** The target OI-model. */
    protected OIModel m_targetOIModel;
    /** The target KAON connection. */
    protected KAONConnection m_targetKAONConnection;
    /** The list of events. */
    protected List m_changeEvents;
    /** The map of source to target models. */
    protected Map m_sourceToTarget;

    /**
     * Creates an instance of this class.
     *
     * @param sourceOIModel                     the source OI-model
     * @param logicalToPhysicalURIs             the map of logical to physical URIs
     * @param targetKAONConnection              the target KAON connection
     */
    public OIModelReplicator(OIModel sourceOIModel,Map logicalToPhysicalURIs,KAONConnection targetKAONConnection) {
        m_sourceToTarget=new HashMap();
        m_logicalToPhysicalURIs=logicalToPhysicalURIs;
        m_sourceOIModel=sourceOIModel;
        m_targetKAONConnection=targetKAONConnection;
        m_changeEvents=new LinkedList();
    }
    /**
     * Starts the replication of the source model to the target model
     *
     * @return                                  the target model
     * @throws KAONException                    thrown if there is an error
     * @throws InterruptedException             thrown if the replicator has been interrupted
     */
    public OIModel doReplication() throws KAONException,InterruptedException {
        try {
            setUpModelMappings();
            processConcepts();
            processProperties();
            processInstances();
            return m_targetOIModel;
        }
        finally {
            m_logicalToPhysicalURIs=null;
            m_sourceOIModel=null;
            m_targetOIModel=null;
            m_targetKAONConnection=null;
            m_changeEvents=null;
        }
    }
    /**
     * Establishes the map from source to target OI-models.
     *
     * @throws KAONException                    thrown it there is an error
     * @throws InterruptedException             thrown if the replicator has been interrupted
     */
    protected void setUpModelMappings() throws KAONException,InterruptedException {
        m_targetOIModel=createTargetOIModel(m_sourceOIModel);
        m_sourceToTarget.put(m_sourceOIModel,m_targetOIModel);
        Iterator iterator=OIModels.getAllIncludedOIModels(m_sourceOIModel).iterator();
        while (iterator.hasNext()) {
            OIModel sourceOIModel=(OIModel)iterator.next();
            try {
                m_targetKAONConnection.openOIModelLogical(sourceOIModel.getLogicalURI());
            }
            catch (KAONException modelShouldBeReplicated) {
                OIModel targetOIModel=createTargetOIModel(sourceOIModel);
                m_sourceToTarget.put(sourceOIModel,targetOIModel);
            }
            checkInterrupted();
        }
        iterator=m_sourceToTarget.keySet().iterator();
        while (iterator.hasNext()) {
            OIModel sourceOIModel=(OIModel)iterator.next();
            OIModel targetOIModel=(OIModel)m_sourceToTarget.get(sourceOIModel);
            Iterator inner=sourceOIModel.getIncludedOIModels().iterator();
            while (inner.hasNext()) {
                OIModel sourceIncludedOIModel=(OIModel)inner.next();
                OIModel targetIncludedOIModel=m_targetKAONConnection.openOIModelLogical(sourceIncludedOIModel.getLogicalURI());
                if (!targetOIModel.getIncludedOIModels().contains(targetIncludedOIModel))
                    targetOIModel.applyChanges(Collections.singletonList(new AddIncludedOIModel(targetIncludedOIModel)));
                checkInterrupted();
            }
        }
    }
    /**
     * Creates a target model for given source OI-model.
     *
     * @param sourceOIModel                     the source OI-model
     * @return                                  the OI-model generated for given source OI-model
     * @throws KAONException                    thrown it there is an error
     */
    protected OIModel createTargetOIModel(OIModel sourceOIModel) throws KAONException {
        String logicalURI=sourceOIModel.getLogicalURI();
        String targetPhysicalURI=(String)m_logicalToPhysicalURIs.get(logicalURI);
        OIModel targetOIModel=m_targetKAONConnection.createOIModel(targetPhysicalURI,logicalURI);
        Map attributes=sourceOIModel.getAttributes();
        Iterator keys=attributes.keySet().iterator();
        while (keys.hasNext()) {
            String key=(String)keys.next();
            String value=(String)attributes.get(key);
            targetOIModel.setAttribute(key,value);
        }
        targetOIModel.setAttribute("OIModel.replicatedFromPhysicalURI",sourceOIModel.getPhysicalURI());
        return targetOIModel;
    }
    /**
     * Processes all concepts.
     *
     * @throws KAONException                    thrown if there is an error
     * @throws InterruptedException             thrown if the replicator has been interrupted
     */
    protected void processConcepts() throws KAONException,InterruptedException {
        Set concepts=loadConcepts();
        flushChanges();
        copyConcepts(concepts);
        flushChanges();
    }
    /**
     * Processes all properties.
     *
     * @throws KAONException                    thrown if there is an error
     * @throws InterruptedException             thrown if the replicator has been interrupted
     */
    protected void processProperties() throws KAONException,InterruptedException {
        Set properties=loadProperties();
        flushChanges();
        copyProperties(properties);
        flushChanges();
    }
    /**
     * Processes all instances.
     *
     * @throws KAONException                    thrown if there is an error
     * @throws InterruptedException             thrown if the replicator has been interrupted
     */
    protected void processInstances() throws KAONException,InterruptedException {
        Set instances=loadInstances();
        flushChanges();
        copyInstances(instances);
        flushChanges();
    }
    /**
     * Loads the concepts.
     *
     * @return                                  the set of concepts
     * @throws KAONException                    thrown if there is an error
     * @throws InterruptedException             thrown if the replicator has been interrupted
     */
    protected Set loadConcepts() throws KAONException,InterruptedException {
        Set concepts=m_sourceOIModel.getConcepts();
        processElements(concepts,m_sourceOIModel,OIModel.LOAD_CONCEPT_BASICS | OIModel.LOAD_SUPER_CONCEPTS,new ObjectProcessor() {
            public void processLoadedObjects(Set objects) throws KAONException,InterruptedException {
                Iterator iterator=objects.iterator();
                while (iterator.hasNext()) {
                    Concept sourceConcept=(Concept)iterator.next();
                    OIModel targetOIModel=(OIModel)m_sourceToTarget.get(sourceConcept.getSourceOIModel());
                    if (targetOIModel!=null) {
                        Concept targetConcept=targetOIModel.getConcept(sourceConcept.getURI());
                        applyChange(new AddEntity(targetOIModel,null,targetConcept));
                    }
                    checkInterrupted();
                }
                flushChanges();
            }
        },PHASE_LOAD_CONCEPTS);
        return concepts;
    }
    /**
     * Loads the properties.
     *
     * @return                                  the set of properties
     * @throws KAONException                    thrown if there is an error
     * @throws InterruptedException             thrown if the replicator has been interrupted
     */
    protected Set loadProperties() throws KAONException,InterruptedException {
        Set properties=m_sourceOIModel.getProperties();
        processElements(properties,m_sourceOIModel,OIModel.LOAD_PROPERTY_BASICS | OIModel.LOAD_SUPER_PROPERTIES | OIModel.LOAD_PROPERTY_DOMAINS | OIModel.LOAD_PROPERTY_RANGES,new ObjectProcessor() {
            public void processLoadedObjects(Set objects) throws KAONException,InterruptedException {
                Iterator iterator=objects.iterator();
                while (iterator.hasNext()) {
                    Property sourceProperty=(Property)iterator.next();
                    OIModel targetOIModel=(OIModel)m_sourceToTarget.get(sourceProperty.getSourceOIModel());
                    if (targetOIModel!=null) {
                        Property targetProperty=targetOIModel.getProperty(sourceProperty.getURI());
                        applyChange(new AddEntity(targetOIModel,null,targetProperty));
                        if (sourceProperty.isAttribute())
                            applyChange(new SetPropertyIsAttribute(targetOIModel,null,targetProperty,true));
                        if (sourceProperty.isSymmetric())
                            applyChange(new SetPropertySymmetric(targetOIModel,null,targetProperty,true));
                        if (sourceProperty.isTransitive())
                            applyChange(new SetPropertyTransitive(targetOIModel,null,targetProperty,true));
                    }
                    checkInterrupted();
                }
                flushChanges();
            }
        },PHASE_LOAD_PROPERTIES);
        return properties;
    }
    /**
     * Loads the instances.
     *
     * @return                                  the set of instances
     * @throws KAONException                    thrown if there is an error
     * @throws InterruptedException             thrown if the replicator has been interrupted
     */
    protected Set loadInstances() throws KAONException,InterruptedException {
        Set instances=m_sourceOIModel.getInstances();
        processElements(instances,m_sourceOIModel,OIModel.LOAD_INSTANCE_BASICS | OIModel.LOAD_INSTANCE_PARENT_CONCEPTS | OIModel.LOAD_INSTANCE_FROM_PROPERTY_VALUES,new ObjectProcessor() {
            public void processLoadedObjects(Set objects) throws KAONException,InterruptedException {
                Iterator iterator=objects.iterator();
                while (iterator.hasNext()) {
                    Instance sourceInstance=(Instance)iterator.next();
                    OIModel targetOIModel=(OIModel)m_sourceToTarget.get(sourceInstance.getSourceOIModel());
                    if (targetOIModel!=null) {
                        Instance targetInstance=targetOIModel.getInstance(sourceInstance.getURI());
                        applyChange(new AddEntity(targetOIModel,null,targetInstance));
                    }
                    checkInterrupted();
                }
                flushChanges();
            }
        },PHASE_LOAD_INSTANCES);
        return instances;
    }
    /**
     * Copies the information about the concepts.
     *
     * @param concepts                          the set of concepts
     * @throws KAONException                    thrown if there is an error
     * @throws InterruptedException             thrown if the replicator has been interrupted
     */
    protected void copyConcepts(Set concepts) throws KAONException,InterruptedException {
        processElements(concepts,null,0,new ObjectProcessor() {
            public void processLoadedObjects(Set objects) throws KAONException,InterruptedException {
                Iterator iterator=objects.iterator();
                while (iterator.hasNext()) {
                    Concept sourceConcept=(Concept)iterator.next();
                    Concept targetConcept=m_targetOIModel.getConcept(sourceConcept.getURI());
                    Iterator superConcepts=sourceConcept.getSuperConcepts().iterator();
                    while (superConcepts.hasNext()) {
                        Concept sourceSuperConcept=(Concept)superConcepts.next();
                        OIModel targetOIModel=(OIModel)m_sourceToTarget.get(sourceConcept.getSuperSubConceptOIModel(sourceSuperConcept));
                        if (targetOIModel!=null) {
                            Concept targetSuperConcept=m_targetOIModel.getConcept(sourceSuperConcept.getURI());
                            applyChange(new AddSubConcept(targetOIModel,null,targetSuperConcept,targetConcept));
                        }
                        checkInterrupted();
                    }
                    checkInterrupted();
                }
                flushChanges();
            }
        },PHASE_COPY_CONCEPTS);
    }
    /**
     * Copies the information about the properties.
     *
     * @param properties                        the set of properties
     * @throws KAONException                    thrown if there is an error
     * @throws InterruptedException             thrown if the replicator has been interrupted
     */
    protected void copyProperties(Set properties) throws KAONException,InterruptedException {
        processElements(properties,null,0,new ObjectProcessor() {
            public void processLoadedObjects(Set objects) throws KAONException,InterruptedException {
                Iterator iterator=objects.iterator();
                while (iterator.hasNext()) {
                    Property sourceProperty=(Property)iterator.next();
                    Property targetProperty=m_targetOIModel.getProperty(sourceProperty.getURI());
                    OIModel targetOIModel=(OIModel)m_sourceToTarget.get(sourceProperty.getInversePropertyOIModel());
                    if (targetOIModel!=null && targetProperty.getInverseProperty()==null) {
                        Property targetInverseProperty=m_targetOIModel.getProperty(sourceProperty.getInverseProperty().getURI());
                        applyChange(new SetInverseProperties(targetOIModel,null,targetProperty,targetInverseProperty));
                        flushChanges();
                    }
                    Iterator superProperties=sourceProperty.getSuperProperties().iterator();
                    while (superProperties.hasNext()) {
                        Property sourceSuperProperty=(Property)superProperties.next();
                        targetOIModel=(OIModel)m_sourceToTarget.get(sourceProperty.getSuperSubPropertyOIModel(sourceSuperProperty));
                        if (targetOIModel!=null) {
                            Property targetSuperProperty=m_targetOIModel.getProperty(sourceSuperProperty.getURI());
                            applyChange(new AddSubProperty(targetOIModel,null,targetSuperProperty,targetProperty));
                        }
                        checkInterrupted();
                    }
                    Iterator domainConcepts=sourceProperty.getDomainConcepts().iterator();
                    while (domainConcepts.hasNext()) {
                        Concept sourceDomainConcept=(Concept)domainConcepts.next();
                        targetOIModel=(OIModel)m_sourceToTarget.get(sourceProperty.getDomainConceptOIModel(sourceDomainConcept));
                        if (targetOIModel!=null) {
                            Concept targetDomainConcept=m_targetOIModel.getConcept(sourceDomainConcept.getURI());
                            applyChange(new AddPropertyDomain(targetOIModel,null,targetProperty,targetDomainConcept));
                            int minimumCardinality=sourceProperty.getMinimumCardinality(sourceDomainConcept);
                            if (minimumCardinality!=0)
                                applyChange(new SetMinimumCardinality(targetOIModel,null,targetProperty,targetDomainConcept,minimumCardinality));
                            int maximumCardinality=sourceProperty.getMaximumCardinality(sourceDomainConcept);
                            if (maximumCardinality!=Integer.MAX_VALUE)
                                applyChange(new SetMaximumCardinality(targetOIModel,null,targetProperty,targetDomainConcept,maximumCardinality));
                        }
                        checkInterrupted();
                    }
                    Iterator rangeConcepts=sourceProperty.getRangeConcepts().iterator();
                    while (rangeConcepts.hasNext()) {
                        Concept sourceRangeConcept=(Concept)rangeConcepts.next();
                        targetOIModel=(OIModel)m_sourceToTarget.get(sourceProperty.getRangeConceptOIModel(sourceRangeConcept));
                        if (targetOIModel!=null) {
                            Concept targetRangeConcept=m_targetOIModel.getConcept(sourceRangeConcept.getURI());
                            applyChange(new AddPropertyRange(targetOIModel,null,targetProperty,targetRangeConcept));
                        }
                        checkInterrupted();
                    }
                    checkInterrupted();
                }
                flushChanges();
            }
        },PHASE_COPY_PROPERTIES);
    }
    /**
     * Copies the information about the instances.
     *
     * @param instances                         the set of instances
     * @throws KAONException                    thrown if there is an error
     * @throws InterruptedException             thrown if the replicator has been interrupted
     */
    protected void copyInstances(Set instances) throws KAONException,InterruptedException {
        processElements(instances,null,0,new ObjectProcessor() {
            public void processLoadedObjects(Set objects) throws KAONException,InterruptedException {
                Iterator iterator=objects.iterator();
                while (iterator.hasNext()) {
                    Instance sourceInstance=(Instance)iterator.next();
                    Instance targetInstance=m_targetOIModel.getInstance(sourceInstance.getURI());
                    Iterator parentConcepts=sourceInstance.getParentConcepts().iterator();
                    while (parentConcepts.hasNext()) {
                        Concept sourceConcept=(Concept)parentConcepts.next();
                        OIModel targetOIModel=(OIModel)m_sourceToTarget.get(sourceInstance.getConceptInstanceOIModel(sourceConcept));
                        if (targetOIModel!=null) {
                            Concept targetConcept=m_targetOIModel.getConcept(sourceConcept.getURI());
                            applyChange(new AddInstanceOf(targetOIModel,null,targetConcept,targetInstance));
                        }
                        checkInterrupted();
                    }
                    Iterator propertyInstances=sourceInstance.getFromPropertyInstances().iterator();
                    while (propertyInstances.hasNext()) {
                        PropertyInstance sourcePropertyInstance=(PropertyInstance)propertyInstances.next();
                        OIModel targetOIModel=(OIModel)m_sourceToTarget.get(sourcePropertyInstance.getSourceOIModel());
                        if (targetOIModel!=null) {
                            Property targetProperty=m_targetOIModel.getProperty(sourcePropertyInstance.getProperty().getURI());
                            Instance targetSourceInstance=m_targetOIModel.getInstance(sourcePropertyInstance.getSourceInstance().getURI());
                            Object targetTargetValue;
                            if (sourcePropertyInstance.getTargetValue() instanceof Instance)
                                targetTargetValue=m_targetOIModel.getInstance(((Instance)sourcePropertyInstance.getTargetValue()).getURI());
                            else
                                targetTargetValue=sourcePropertyInstance.getTargetValue();
                            applyChange(new AddPropertyInstance(targetOIModel,null,targetProperty,targetSourceInstance,targetTargetValue));
                        }
                        checkInterrupted();
                    }
                    checkInterrupted();
                }
                flushChanges();
            }
        },PHASE_COPY_INSTANCES);
    }
    /**
     * Adds an event to be processed.
     *
     * @param changeEvent                       the event
     */
    protected void applyChange(ChangeEvent changeEvent) {
        m_changeEvents.add(changeEvent);
    }
    /**
     * Flushes the changes to the OI-model.
     *
     * @throws KAONException                    thrown if there is an error
     */
    protected void flushChanges() throws KAONException {
        if (!m_changeEvents.isEmpty()) {
            m_targetKAONConnection.applyChanges(m_changeEvents);
            m_changeEvents.clear();
        }
    }
}
