package edu.unika.aifb.kaon.api.util;

import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

/**
 * A utility class for processing OI-models in chunks.
 *
 * @author Raphael Volz (volz@aifb.uni-karlsruhe.de)
 * @author Boris Motik (boris.motik@fzi.de)
 */
public abstract class OIModelProcessor {
    /** The size of the chunks for processing. */
    protected static final int CHUNK_SIZE=600;

    /** The progress listener of the processor. */
    protected ProgressListener m_progressListener;

    /**
     * Creates an instance of this class.
     */
    public OIModelProcessor() {
        m_progressListener=NullProgressListener.INSTANCE;
    }
    /**
     * Registers a progress listener.
     *
     * @param progressListener                  the listener for processor progress
     */
    public void setProgressListener(ProgressListener progressListener) {
        m_progressListener=progressListener;
    }
    /**
     * Loads and processes elements from the given set in chunks.
     *
     * @param set                               the original set of objects
     * @param oimodel                           the OI-model that loads elements (may be <code>null</code> - loading is then not done)
     * @param loadFlag                          the flag specifying which information should be loaded
     * @param objectProcessor                   receives objects to be processed
     * @param progressPhase                     the phase of the replication
     * @throws KAONException                    thrown if there is an error
     * @throws InterruptedException             thrown if the replicator has been interrupted
     */
    protected void processElements(Set set,OIModel oimodel,int loadFlag,ObjectProcessor objectProcessor,int progressPhase) throws KAONException,InterruptedException {
        int processed=0;
        m_progressListener.processorProgress(progressPhase,processed,set.size());
        Set objectsToLoad=new HashSet();
        Iterator objects=set.iterator();
        while (objects.hasNext()) {
            Entity entity=(Entity)objects.next();
            objectsToLoad.add(entity);
            if (objectsToLoad.size()==CHUNK_SIZE) {
                if (oimodel!=null)
                    oimodel.loadObjects(objectsToLoad,loadFlag);
                objectProcessor.processLoadedObjects(objectsToLoad);
                processed+=objectsToLoad.size();
                objectsToLoad.clear();
                m_progressListener.processorProgress(progressPhase,processed,set.size());
            }
            checkInterrupted();
        }
        if (!objectsToLoad.isEmpty()) {
            if (oimodel!=null)
                oimodel.loadObjects(objectsToLoad,loadFlag);
            objectProcessor.processLoadedObjects(objectsToLoad);
            processed+=objectsToLoad.size();
            m_progressListener.processorProgress(progressPhase,processed,set.size());
        }
    }
    /**
     * Checks whether the thread has been interrupted and throws an exception.
     *
     * @throws InterruptedException             thrown if the replicator has been interrupted
     */
    protected void checkInterrupted() throws InterruptedException {
        if (Thread.interrupted())
            throw new InterruptedException();
    }

    /**
     * Processor for loaded objects.
     */
    protected interface ObjectProcessor {
        void processLoadedObjects(Set objects) throws KAONException,InterruptedException;
    }

    /**
     * The interface receiving notification about the progress of the processor.
     */
    public interface ProgressListener {
        void processorProgress(int phase,int done,int steps);
    }

    /**
     * Empty progress listener.
     */
    protected static class NullProgressListener implements ProgressListener {
        public static final ProgressListener INSTANCE=new NullProgressListener();

        public void processorProgress(int phase,int done,int steps) {
        }
    }
}
