package edu.unika.aifb.kaon.api.util;

import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;
import javax.swing.event.EventListenerList;

import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;

/**
 * Manager of events for an OI-model.
 *
 * @author Raphael Volz (volz@aifb.uni-karlsruhe.de)
 * @author Boris Motik (boris.motik@fzi.de)
 */
public class OIModelEventManager {
    /** OI-model that this manager manages. */
    protected OIModel m_oimodel;
    /** List of listener registered for the pool. */
    protected EventListenerList m_listenerList;
    /** Event suspend count. */
    protected int m_suspendCount;
    /** List of bulk events. */
    protected List m_events;
    /** Flag set if model is refreshed. */
    protected boolean m_modelRefreshed;
    /** Flag set if model is deleted. */
    protected boolean m_modelDeleted;

    /**
     * Creates an instance of this class.
     *
     * @param oimodel                   oimodel that this class will manage
     */
    public OIModelEventManager(OIModel oimodel) {
        m_oimodel=oimodel;
        m_listenerList=new EventListenerList();
        m_suspendCount=0;
        m_events=new LinkedList();
    }
    /**
     * Adds a listener to this manager.
     *
     * @param listener                  listener to be added
     */
    public void addOIModelListener(OIModelListener listener) {
        m_listenerList.add(OIModelListener.class,listener);
    }
    /**
     * Removes a listener from this manager.
     *
     * @param listener                  listener to be removed
     */
    public void removeOIModelListener(OIModelListener listener) {
        m_listenerList.remove(OIModelListener.class,listener);
    }
    /**
     * Suspends entity pool events until {@link #resumeEvents} is called.
     */
    public void suspendEvents() {
        m_suspendCount++;
    }
    /**
     * Resumes entity pool events.
     */
    public void resumeEvents() {
        if (--m_suspendCount<=0)
            fireBulkEvent();
    }
    /**
     * Fires an event about all backlogged notifications. This method doesn't observe the suspend count
     * of the pool, so use it with care.
     */
    public void fireBulkEvent() {
        if (m_modelRefreshed) {
            try {
                Object[] listeners=m_listenerList.getListenerList();
                for (int i=listeners.length-2;i>=0;i-=2)
                    if (listeners[i]==OIModelListener.class)
                        ((OIModelListener)listeners[i+1]).modelRefreshed(m_oimodel);
            }
            finally {
                m_modelRefreshed=false;
            }
        }
        if (!m_events.isEmpty())
            try {
                Object[] listeners=m_listenerList.getListenerList();
                for (int i=listeners.length-2;i>=0;i-=2)
                    if (listeners[i]==OIModelListener.class)
                        ((OIModelListener)listeners[i+1]).modelChanged(m_oimodel,m_events);
            }
            finally {
                m_events.clear();
            }
        if (m_modelDeleted) {
            try {
                Object[] listeners=m_listenerList.getListenerList();
                for (int i=listeners.length-2;i>=0;i-=2)
                    if (listeners[i]==OIModelListener.class)
                        ((OIModelListener)listeners[i+1]).modelDeleted(m_oimodel);
            }
            finally {
                m_modelDeleted=false;
            }
        }
    }
    /**
     * Notifies this manager about a change event in the pool.
     *
     * @param changeEvent               change event in the pool
     */
    public void notifyChangeEvent(ChangeEvent changeEvent) {
        m_events.add(changeEvent);
        if (m_suspendCount<=0)
            fireBulkEvent();
    }
    /**
     * Notifies the manager about a refresh event.
     */
    public void notifyRefresh() {
        m_modelRefreshed=true;
        if (m_suspendCount<=0)
            fireBulkEvent();
    }
    /**
     * Notifies the manager about a model deleted event.
     */
    public void notifyDelete() {
        m_modelDeleted=true;
        if (m_suspendCount<=0)
            fireBulkEvent();
    }
    /**
     * Returns a bookmark representing current state of logged events.
     *
     * @return                          bookmark object representing indicating the current state of logged events
     */
    public Object getBookmark() {
        return new Integer(m_events.size());
    }
    /**
     * Applies a previously taken bookmark to remove all events logged after the bookmark was taken.
     *
     * @param bookmark                  bookmark after which all events are removed
     */
    public void applyBookmark(Object bookmark) {
        int index=((Integer)bookmark).intValue();
        if (index<m_events.size()) {
            Iterator iterator=m_events.iterator();
            while (index>0) {
                iterator.next();
                index--;
            }
            while (iterator.hasNext()) {
                iterator.next();
                iterator.remove();
            }
        }
    }
}
