package edu.unika.aifb.kaon.api.util;

import java.util.List;
import java.util.ArrayList;
import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.vocabulary.*;

/**
 * A utility class that generates events for manipulation of the lexicon.
 *
 * @author Raphael Volz (volz@aifb.uni-karlsruhe.de)
 * @author Boris Motik (boris.motik@fzi.de)
 */
public class LexiconUtil {
    /**
     * Adds a lexical entry to the OI-model.
     *
     * @param lexicalEntry                  lexical entry
     * @param typeURI                       type of the lexical entry
     * @param list                          list for events
     */
    public static void addLexicalEntry(LexicalEntry lexicalEntry,String typeURI,List list) throws KAONException {
        Concept parentConcept=lexicalEntry.getOIModel().getConcept(typeURI);
        if (!parentConcept.isInOIModel())
            throw new KAONException("Concept '"+typeURI+"' is not in model.");
        list.add(new AddEntity(lexicalEntry.getOIModel(),null,lexicalEntry));
        list.add(new AddInstanceOf(lexicalEntry.getOIModel(),null,parentConcept,lexicalEntry));
    }
    /**
     * Adds a lexical entry to the OI-model.
     *
     * @param lexicalEntry                  lexical entry
     * @param typeURI                       type of the lexical entry
     * @return                              list of events
     */
    public static List addLexicalEntry(LexicalEntry lexicalEntry,String typeURI) throws KAONException {
        List list=new ArrayList(2);
        addLexicalEntry(lexicalEntry,typeURI,list);
        return list;
    }
    /**
     * Adds a reference from a lexical entry to an entity.
     *
     * @param lexicalEntry                  lexical entry
     * @param entity                        entity
     * @param list                          list for events
     */
    public static void addLexicalReference(LexicalEntry lexicalEntry,Entity entity,List list) throws KAONException {
        list.add(new AddPropertyInstance(lexicalEntry.getOIModel(),null,lexicalEntry.getOIModel().getProperty(KAONVocabularyAdaptor.INSTANCE.getReferences()),lexicalEntry,entity.getSpanningInstance()));
    }
    /**
     * Adds a reference from a lexical entry to an entity.
     *
     * @param lexicalEntry                  lexical entry
     * @param entity                        entity
     * @return                              list of events
     */
    public static List addLexicalReference(LexicalEntry lexicalEntry,Entity entity) throws KAONException {
        List list=new ArrayList(1);
        addLexicalReference(lexicalEntry,entity,list);
        return list;
    }
    /**
     * Sets the value and language of the lexical entry.
     *
     * @param lexicalEntry                  lexical entry
     * @param value                         value of the entry
     * @param languageURI                   language of the lexical entry
     * @param list                          list for events
     */
    public static void setLexicalEntryValue(LexicalEntry lexicalEntry,String value,String languageURI,List list) throws KAONException {
        list.add(new SetPropertyInstanceValue(lexicalEntry.getOIModel(),null,lexicalEntry.getOIModel().getProperty(KAONVocabularyAdaptor.INSTANCE.getValue()),lexicalEntry,value));
        list.add(new SetPropertyInstanceValue(lexicalEntry.getOIModel(),null,lexicalEntry.getOIModel().getProperty(KAONVocabularyAdaptor.INSTANCE.getInLanguage()),lexicalEntry,lexicalEntry.getOIModel().getInstance(languageURI)));
    }
    /**
    /**
     * Sets the value and language of the lexical entry.
     *
     * @param lexicalEntry                  lexical entry
     * @param value                         value of the entry
     * @param languageURI                   language of the lexical entry
     * @return                              list of events
     */
    public static List setLexicalEntryValue(LexicalEntry lexicalEntry,String value,String languageURI) throws KAONException {
        List list=new ArrayList(1);
        setLexicalEntryValue(lexicalEntry,value,languageURI,list);
        return list;
    }
    /**
     * Creates a lexical reference with given URI, sets its value in specified language and associates it with given entity.
     *
     * @param lexicalEntry                  lexical entry
     * @param typeURI                       URI of the lexical entry type
     * @param value                         value of the lexical entry
     * @param languageURI                   URI of the language
     * @param entity                        entity to which the lexical entry is associated
     * @param list                          list for events
     */
    public static void createLexicalEntry(LexicalEntry lexicalEntry,String typeURI,String value,String languageURI,Entity entity,List list) throws KAONException {
        addLexicalEntry(lexicalEntry,typeURI,list);
        addLexicalReference(lexicalEntry,entity,list);
        list.add(new AddPropertyInstance(lexicalEntry.getOIModel(),null,lexicalEntry.getOIModel().getProperty(KAONVocabularyAdaptor.INSTANCE.getValue()),lexicalEntry,value));
        list.add(new AddPropertyInstance(lexicalEntry.getOIModel(),null,lexicalEntry.getOIModel().getProperty(KAONVocabularyAdaptor.INSTANCE.getInLanguage()),lexicalEntry,lexicalEntry.getOIModel().getInstance(languageURI)));
    }
    /**
     * Creates a lexical reference with given URI, sets its value in specified language and associates it with given entity.
     *
     * @param lexicalEntry                  lexical entry
     * @param typeURI                       URI of the lexical entry type
     * @param value                         value of the lexical entry
     * @param languageURI                   URI of the language
     * @param entity                        entity to which the lexical entry is associated
     * @return                              list of events
     */
    public static List createLexicalEntry(LexicalEntry lexicalEntry,String typeURI,String value,String languageURI,Entity entity) throws KAONException {
        List list=new ArrayList(5);
        createLexicalEntry(lexicalEntry,typeURI,value,languageURI,entity,list);
        return list;
    }
    /**
     * Removes a reference from a lexical entry to an entity.
     *
     * @param lexicalEntry                  lexical entry
     * @param entity                        entity
     * @param list                          list for events
     */
    public static void removeLexicalReference(LexicalEntry lexicalEntry,Entity entity,List list) throws KAONException {
        list.add(new RemovePropertyInstance(lexicalEntry.getOIModel(),null,lexicalEntry.getOIModel().getProperty(KAONVocabularyAdaptor.INSTANCE.getReferences()),lexicalEntry,entity.getSpanningInstance()));
    }
    /**
     * Removes a reference from a lexical entry to an entity.
     *
     * @param lexicalEntry                  lexical entry
     * @param entity                        entity
     * @return                              list of events
     */
    public static List removeLexicalReference(LexicalEntry lexicalEntry,Entity entity) throws KAONException {
        List list=new ArrayList(1);
        removeLexicalReference(lexicalEntry,entity,list);
        return list;
    }
    /**
     * Sets a value of the lexical attribute.
     *
     * @param lexicalEntry                  lexical entry
     * @param attributeURI                  URI of the lexical attribute
     * @param value                         value of the lexical attribute
     * @param list                          list for events
     */
    public static void setLexicalAttribute(LexicalEntry lexicalEntry,String attributeURI,String value,List list) throws KAONException {
        list.add(new SetPropertyInstanceValue(lexicalEntry.getOIModel(),null,lexicalEntry.getOIModel().getProperty(attributeURI),lexicalEntry,value));
    }
    /**
     * Sets a value of the lexical attribute.
     *
     * @param lexicalEntry                  lexical entry
     * @param attributeURI                  URI of the lexical attribute
     * @param value                         value of the lexical attribute
     * @return                              list of events
     */
    public static List setLexicalAttribute(LexicalEntry lexicalEntry,String attributeURI,String value) throws KAONException {
        List list=new ArrayList(1);
        setLexicalAttribute(lexicalEntry,attributeURI,value,list);
        return list;
    }
    /**
     * Given a set of change events computes a set of instances whose lexical entries have changed. This
     * in turn can be used for event processing.
     *
     * @param eventList                             the list of change events
     * @throws KAONException                        thrown if there is an error
     */
    public static Set getInstancesWithChangedLexicalEntries(List eventList) throws KAONException {
        // the goal of this method is to reduce the number of calls to LexicalEntry.getReferencedEntities()
        Set updatedLexicalEntries=new HashSet();
        Set referencedUpdatedLexicalEntries=new HashSet();
        Set instancesWithChangedLexicalEntries=new HashSet();
        Iterator iterator=eventList.iterator();
        while (iterator.hasNext()) {
            ChangeEvent changeEvent=(ChangeEvent)iterator.next();
            PropertyInstance propertyInstance=null;
            if (changeEvent instanceof AddPropertyInstance)
                propertyInstance=((AddPropertyInstance)changeEvent).getPropertyInstance();
            else if (changeEvent instanceof RemovePropertyInstance)
                propertyInstance=((RemovePropertyInstance)changeEvent).getPropertyInstance();
            else if (changeEvent instanceof ChangePropertyInstanceValue)
                propertyInstance=((ChangePropertyInstanceValue)changeEvent).getPropertyInstance();
            else if (changeEvent instanceof SetPropertyInstanceValue)
                propertyInstance=((SetPropertyInstanceValue)changeEvent).getPropertyInstance();
            if (propertyInstance!=null) {
                String uri=propertyInstance.getProperty().getURI();
                if (uri.equals(KAONVocabularyAdaptor.INSTANCE.getReferences())) {
                    Object targetValue=propertyInstance.getTargetValue();
                    if (targetValue instanceof Instance)
                        instancesWithChangedLexicalEntries.add(targetValue);
                    referencedUpdatedLexicalEntries.add(propertyInstance.getSourceInstance());
                }
                else if (uri.equals(KAONVocabularyAdaptor.INSTANCE.getValue()) || uri.equals(KAONVocabularyAdaptor.INSTANCE.getInLanguage()))
                    updatedLexicalEntries.add(propertyInstance.getSourceInstance());
            }
        }
        iterator=updatedLexicalEntries.iterator();
        while (iterator.hasNext()) {
            LexicalEntry lexicalEntry=(LexicalEntry)iterator.next();
            if (!referencedUpdatedLexicalEntries.contains(lexicalEntry))
                instancesWithChangedLexicalEntries.addAll(lexicalEntry.getReferencedEntities());
        }
        return instancesWithChangedLexicalEntries;
    }
}
