package edu.unika.aifb.kaon.api.util;

import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.change.*;
import edu.unika.aifb.kaon.api.oimodel.*;

/**
 * An instance of this class is capable of reversing a sequence of changes.
 */
public class ChangeReverse implements ChangeVisitor {
    /** The list of reversed changes. */
    protected List m_reversedChanges;

    /**
     * Visits AddEntity change.
     */
    public void visit(AddEntity event) {
        addChange(new RemoveEntity(event.getOIModel(),null,event.getEntity()));
    }
    /**
     * Visits RemoveEntity change.
     */
    public void visit(RemoveEntity event) {
        addChange(new AddEntity(event.getOIModel(),null,event.getEntity()));
    }
    /**
     * Visits an event for creation of subconcepts.
     */
    public void visit(AddSubConcept event) {
        addChange(new RemoveSubConcept(event.getOIModel(),null,event.getSuperConcept(),event.getSubConcept()));
    }
    /**
     * Visits an event for removal of subconcepts.
     */
    public void visit(RemoveSubConcept event) {
        addChange(new AddSubConcept(event.getOIModel(),null,event.getSuperConcept(),event.getSubConcept()));
    }
    /**
     * Visits an event for adding a domain to the property.
     */
    public void visit(AddPropertyDomain event) {
        addChange(new RemovePropertyDomain(event.getOIModel(),null,event.getProperty(),event.getConcept()));
    }
    /**
     * Visits an event for removing a domain from the property.
     */
    public void visit(RemovePropertyDomain event) {
        addChange(new AddPropertyDomain(event.getOIModel(),null,event.getProperty(),event.getConcept()));
    }
    /**
     * Visits an event for adding a range to the property.
     */
    public void visit(AddPropertyRange event) {
        addChange(new RemovePropertyRange(event.getOIModel(),null,event.getProperty(),event.getConcept()));
    }
    /**
     * Visits an event for removing a range from the property.
     */
    public void visit(RemovePropertyRange event) {
        addChange(new AddPropertyRange(event.getOIModel(),null,event.getProperty(),event.getConcept()));
    }
    /**
     * Visits an event for determinig whether property is attirubte.
     */
    public void visit(SetPropertyIsAttribute event) {
        addChange(new SetPropertyIsAttribute(event.getOIModel(),null,event.getProperty(),event.getOldIsAttribute()));
    }
    /**
     * Visits an event for setting the minimum cardinality of a property for a concept.
     */
    public void visit(SetMinimumCardinality event) {
        addChange(new SetMinimumCardinality(event.getOIModel(),null,event.getProperty(),event.getConcept(),event.getOldCardinality()));
    }
    /**
     * Visits an event for setting the maximum cardinality of a property for a concept.
     */
    public void visit(SetMaximumCardinality event) {
        addChange(new SetMaximumCardinality(event.getOIModel(),null,event.getProperty(),event.getConcept(),event.getOldCardinality()));
    }
    /**
     * Visits an event for creation of subproperties.
     */
    public void visit(AddSubProperty event) {
        addChange(new RemoveSubProperty(event.getOIModel(),null,event.getSuperProperty(),event.getSubProperty()));
    }
    /**
     * Visits an event for removal of subproperties.
     */
    public void visit(RemoveSubProperty event) {
        addChange(new AddSubProperty(event.getOIModel(),null,event.getSuperProperty(),event.getSubProperty()));
    }
    /**
     * Visits an event for setting the inverse relationship between properties.
     */
    public void visit(SetInverseProperties event) {
        addChange(new SetNoInverseProperties(event.getOIModel(),null,event.getProperty1(),event.getProperty2()));
    }
    /**
     * Visits an event for removing an inverse relationship between properties.
     */
    public void visit(SetNoInverseProperties event) {
        addChange(new SetInverseProperties(event.getOIModel(),null,event.getProperty1(),event.getProperty2()));
    }
    /**
     * Visits an event for setting the symmetry flag of the property.
     */
    public void visit(SetPropertySymmetric event) {
        addChange(new SetPropertySymmetric(event.getOIModel(),null,event.getProperty(),event.getOldSymmetric()));
    }
    /**
     * Visits an event for setting the transitivity flag of the property.
     */
    public void visit(SetPropertyTransitive event) {
        addChange(new SetPropertyTransitive(event.getOIModel(),null,event.getProperty(),event.getOldTransitive()));
    }
    /**
     * Visits an event for making an instance a subinstance of given concept.
     */
    public void visit(AddInstanceOf event) {
        addChange(new RemoveInstanceOf(event.getOIModel(),null,event.getConcept(),event.getInstance()));
    }
    /**
     * Visits an event for making an instance not a subinstance of given concept.
     */
    public void visit(RemoveInstanceOf event) {
        addChange(new AddInstanceOf(event.getOIModel(),null,event.getConcept(),event.getInstance()));
    }
    /**
     * Visits an event for adding a property instance.
     */
    public void visit(AddPropertyInstance event) {
        addChange(new RemovePropertyInstance(event.getOIModel(),null,event.getPropertyInstance()));
    }
    /**
     * Visits an event for removing a property instance.
     */
    public void visit(RemovePropertyInstance event) {
        addChange(new AddPropertyInstance(event.getOIModel(),null,event.getPropertyInstance()));
    }
    /**
     * Visits an event for setting the value of a property instance.
     */
    public void visit(SetPropertyInstanceValue event) {
        addChange(new RemovePropertyInstance(event.getOIModel(),null,event.getPropertyInstance()));
        PropertyInstance[] propertyInstances=event.getRemovedInstances();
        for (int i=0;i<propertyInstances.length;i++)
            addChange(new AddPropertyInstance(event.getOIModel(),null,propertyInstances[i]));
    }
    /**
     * Visits an event for changing the value of a property instance.
     */
    public void visit(ChangePropertyInstanceValue event) throws KAONException {
        PropertyInstance propertyInstance=event.getPropertyInstance();
        PropertyInstance newPropertyInstance=propertyInstance.getOIModel().getPropertyInstance(propertyInstance.getProperty(),propertyInstance.getSourceInstance(),event.getNewTargetValue());
        addChange(new ChangePropertyInstanceValue(event.getOIModel(),null,newPropertyInstance,propertyInstance.getTargetValue()));
    }
    /**
     * Visits an event for making a model included in another model.
     */
    public void visit(AddIncludedOIModel event) {
        addChange(new RemoveIncludedOIModel(event.getOIModel(),null,event.getIncludedOIModel()));
    }
    /**
     * Visits an event for making a model not included in another model.
     */
    public void visit(RemoveIncludedOIModel event) {
        addChange(new AddIncludedOIModel(event.getOIModel(),null,event.getIncludedOIModel()));
    }
    /**
     * Adds the change to the list.
     *
     * @param event                         the change to add
     */
    protected void addChange(ChangeEvent event) {
        m_reversedChanges.add(0,event);
    }
    /**
     * Gets the changes reversed from the supplied list.
     *
     * @param changeList                    the list to reverse
     * @return                              the reversed list
     */
    public List getReversedChanges(List changeList) throws KAONException {
        try {
            m_reversedChanges=new LinkedList();
            Iterator iterator=changeList.iterator();
            while (iterator.hasNext()) {
                ChangeEvent changeEvent=(ChangeEvent)iterator.next();
                changeEvent.accept(this);
            }
            return m_reversedChanges;
        }
        finally {
            m_reversedChanges=null;
        }
    }
    /**
     * Reverses a set of changes.
     *
     * @param changeList                    the list to reverse
     * @return                              the reversed list
     */
    public static List reverseChanges(List changeList) throws KAONException {
        return new ChangeReverse().getReversedChanges(changeList);
    }
}
