package edu.unika.aifb.kaon.api.util;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.change.*;

/**
 * A class that can compare whether events are equal. This doesn't use equals(), since
 * equals() on events relies on equals() on OIModel objects. Rather, this method compares
 * whether events refer to the same objects.
 *
 * @author Raphael Volz (volz@aifb.uni-karlsruhe.de)
 * @author Boris Motik (boris.motik@fzi.de)
 */
public class ChangeEqualityTester implements ChangeVisitor {
    /** The first event. */
    protected ChangeEvent m_event1;
    /** The second event. */
    protected ChangeEvent m_event2;
    /** The result of the comparison. */
    protected boolean m_result;

    /**
     * Checks whether two elements are equal.
     *
     * @param event1                    the first event
     * @param event2                    the second event
     * @return                          the result
     */
    public boolean equals(ChangeEvent event1,ChangeEvent event2) throws KAONException {
        try {
            m_event1=event1;
            m_event2=event2;
            m_result=false;
            if (equals(m_event1.getOIModel(),m_event2.getOIModel()))
                m_event1.accept(this);
            return m_result;
        }
        finally {
            m_event1=null;
            m_event2=null;
        }
    }
    /**
     * Visits AddEntity change.
     */
    public void visit(AddEntity event) throws KAONException {
        if (m_event2 instanceof AddEntity) {
            AddEntity event2=(AddEntity)m_event2;
            Entity entity1=event.getEntity();
            Entity entity2=event2.getEntity();
            if ((entity1 instanceof Concept && entity2 instanceof Concept) || (entity1 instanceof Property && entity2 instanceof Property) || (entity1 instanceof Instance && entity2 instanceof Instance))
                m_result=entity1.getURI().equals(entity2.getURI());
        }
    }
    /**
     * Visits RemoveEntity change.
     */
    public void visit(RemoveEntity event) throws KAONException {
        if (m_event2 instanceof RemoveEntity) {
            RemoveEntity event2=(RemoveEntity)m_event2;
            Entity entity1=event.getEntity();
            Entity entity2=event2.getEntity();
            if ((entity1 instanceof Concept && entity2 instanceof Concept) || (entity1 instanceof Property && entity2 instanceof Property) || (entity1 instanceof Instance && entity2 instanceof Instance))
                m_result=entity1.getURI().equals(entity2.getURI());
        }
    }
    /**
     * Visits an event for creation of subconcepts.
     */
    public void visit(AddSubConcept event) throws KAONException {
        if (m_event2 instanceof AddSubConcept) {
            AddSubConcept event2=(AddSubConcept)m_event2;
            m_result=equals(event.getSuperConcept(),event2.getSuperConcept()) && equals(event.getSubConcept(),event2.getSubConcept());
        }
    }
    /**
     * Visits an event for removal of subconcepts.
     */
    public void visit(RemoveSubConcept event) throws KAONException {
        if (m_event2 instanceof RemoveSubConcept) {
            RemoveSubConcept event2=(RemoveSubConcept)m_event2;
            m_result=equals(event.getSuperConcept(),event2.getSuperConcept()) && equals(event.getSubConcept(),event2.getSubConcept());
        }
    }
    /**
     * Visits an event for adding a domain to the property.
     */
    public void visit(AddPropertyDomain event) throws KAONException {
        if (m_event2 instanceof AddPropertyDomain) {
            AddPropertyDomain event2=(AddPropertyDomain)m_event2;
            m_result=equals(event.getProperty(),event2.getProperty()) && equals(event.getConcept(),event2.getConcept());
        }
    }
    /**
     * Visits an event for removing a domain from the property.
     */
    public void visit(RemovePropertyDomain event) throws KAONException {
        if (m_event2 instanceof RemovePropertyDomain) {
            RemovePropertyDomain event2=(RemovePropertyDomain)m_event2;
            m_result=equals(event.getProperty(),event2.getProperty()) && equals(event.getConcept(),event2.getConcept());
        }
    }
    /**
     * Visits an event for adding a range to the property.
     */
    public void visit(AddPropertyRange event) throws KAONException {
        if (m_event2 instanceof AddPropertyRange) {
            AddPropertyRange event2=(AddPropertyRange)m_event2;
            m_result=equals(event.getProperty(),event2.getProperty()) && equals(event.getConcept(),event2.getConcept());
        }
    }
    /**
     * Visits an event for removing a range from the property.
     */
    public void visit(RemovePropertyRange event) throws KAONException {
        if (m_event2 instanceof RemovePropertyRange) {
            RemovePropertyRange event2=(RemovePropertyRange)m_event2;
            m_result=equals(event.getProperty(),event2.getProperty()) && equals(event.getConcept(),event2.getConcept());
        }
    }
    /**
     * Visits an event for determinig whether property is attirubte.
     */
    public void visit(SetPropertyIsAttribute event) throws KAONException {
        if (m_event2 instanceof SetPropertyIsAttribute) {
            SetPropertyIsAttribute event2=(SetPropertyIsAttribute)m_event2;
            m_result=equals(event.getProperty(),event2.getProperty()) && event.getIsAttribute()==event2.getIsAttribute();
        }
    }
    /**
     * Visits an event for setting the minimum cardinality of a property for a concept.
     */
    public void visit(SetMinimumCardinality event) throws KAONException {
        if (m_event2 instanceof SetMinimumCardinality) {
            SetMinimumCardinality event2=(SetMinimumCardinality)m_event2;
            m_result=equals(event.getProperty(),event2.getProperty()) && equals(event.getConcept(),event2.getConcept()) && event.getCardinality()==event2.getCardinality();
        }
    }
    /**
     * Visits an event for setting the maximum cardinality of a property for a concept.
     */
    public void visit(SetMaximumCardinality event) throws KAONException {
        if (m_event2 instanceof SetMaximumCardinality) {
            SetMaximumCardinality event2=(SetMaximumCardinality)m_event2;
            m_result=equals(event.getProperty(),event2.getProperty()) && equals(event.getConcept(),event2.getConcept()) && event.getCardinality()==event2.getCardinality();
        }
    }
    /**
     * Visits an event for creation of subproperties.
     */
    public void visit(AddSubProperty event) throws KAONException {
        if (m_event2 instanceof AddSubProperty) {
            AddSubProperty event2=(AddSubProperty)m_event2;
            m_result=equals(event.getSuperProperty(),event2.getSuperProperty()) && equals(event.getSubProperty(),event2.getSubProperty());
        }
    }
    /**
     * Visits an event for removal of subproperties.
     */
    public void visit(RemoveSubProperty event) throws KAONException {
        if (m_event2 instanceof RemoveSubProperty) {
            RemoveSubProperty event2=(RemoveSubProperty)m_event2;
            m_result=equals(event.getSuperProperty(),event2.getSuperProperty()) && equals(event.getSubProperty(),event2.getSubProperty());
        }
    }
    /**
     * Visits an event for setting the inverse relationship between properties.
     */
    public void visit(SetInverseProperties event) throws KAONException {
        if (m_event2 instanceof SetInverseProperties) {
            SetInverseProperties event2=(SetInverseProperties)m_event2;
            m_result=equals(event.getProperty1(),event2.getProperty1()) && equals(event.getProperty2(),event2.getProperty2());
        }
    }
    /**
     * Visits an event for removing an inverse relationship from a property.
     */
    public void visit(SetNoInverseProperties event) throws KAONException {
        if (m_event2 instanceof SetNoInverseProperties) {
            SetNoInverseProperties event2=(SetNoInverseProperties)m_event2;
            m_result=equals(event.getProperty1(),event2.getProperty1()) && equals(event.getProperty2(),event2.getProperty2());
        }
    }
    /**
     * Visits an event for setting the symmetry flag of the property.
     */
    public void visit(SetPropertySymmetric event) throws KAONException {
        if (m_event2 instanceof SetPropertySymmetric) {
            SetPropertySymmetric event2=(SetPropertySymmetric)m_event2;
            m_result=equals(event.getProperty(),event2.getProperty()) && event.getSymmetric()==event2.getSymmetric();
        }
    }
    /**
     * Visits an event for setting the transitivity flag of the property.
     */
    public void visit(SetPropertyTransitive event) throws KAONException {
        if (m_event2 instanceof SetPropertyTransitive) {
            SetPropertyTransitive event2=(SetPropertyTransitive)m_event2;
            m_result=equals(event.getProperty(),event2.getProperty()) && event.getTransitive()==event2.getTransitive();
        }
    }
    /**
     * Visits an event for making an instance a subinstance of given concept.
     */
    public void visit(AddInstanceOf event) throws KAONException {
        if (m_event2 instanceof AddInstanceOf) {
            AddInstanceOf event2=(AddInstanceOf)m_event2;
            m_result=equals(event.getConcept(),event2.getConcept()) && equals(event.getInstance(),event2.getInstance());
        }
    }
    /**
     * Visits an event for making an instance not a subinstance of given concept.
     */
    public void visit(RemoveInstanceOf event) throws KAONException {
        if (m_event2 instanceof RemoveInstanceOf) {
            RemoveInstanceOf event2=(RemoveInstanceOf)m_event2;
            m_result=equals(event.getConcept(),event2.getConcept()) && equals(event.getInstance(),event2.getInstance());
        }
    }
    /**
     * Visits an event for adding a property instance.
     */
    public void visit(AddPropertyInstance event) throws KAONException {
        if (m_event2 instanceof AddPropertyInstance) {
            AddPropertyInstance event2=(AddPropertyInstance)m_event2;
            m_result=equals(event.getPropertyInstance(),event2.getPropertyInstance());
        }
    }
    /**
     * Visits an event for removing a property instance.
     */
    public void visit(RemovePropertyInstance event) throws KAONException {
        if (m_event2 instanceof RemovePropertyInstance) {
            RemovePropertyInstance event2=(RemovePropertyInstance)m_event2;
            m_result=equals(event.getPropertyInstance(),event2.getPropertyInstance());
        }
    }
    /**
     * Visits an event for setting the value of a property instance.
     */
    public void visit(SetPropertyInstanceValue event) throws KAONException {
        if (m_event2 instanceof SetPropertyInstanceValue) {
            SetPropertyInstanceValue event2=(SetPropertyInstanceValue)m_event2;
            m_result=equals(event.getPropertyInstance(),event2.getPropertyInstance());
        }
    }
    /**
     * Visits an event for changing the value of a property instance.
     */
    public void visit(ChangePropertyInstanceValue event) throws KAONException {
        if (m_event2 instanceof ChangePropertyInstanceValue) {
            ChangePropertyInstanceValue event2=(ChangePropertyInstanceValue)m_event2;
            m_result=equals(event.getPropertyInstance(),event2.getPropertyInstance()) && valueEquals(event.getNewTargetValue(),event2.getNewTargetValue());
        }
    }
    /**
     * Visits an event for making a model included in another model.
     */
    public void visit(AddIncludedOIModel event) throws KAONException {
        if (m_event2 instanceof AddIncludedOIModel) {
            AddIncludedOIModel event2=(AddIncludedOIModel)m_event2;
            m_result=equals(event.getIncludedOIModel(),event2.getIncludedOIModel());
        }
    }
    /**
     * Visits an event for making a model not included in another model.
     */
    public void visit(RemoveIncludedOIModel event) throws KAONException {
        if (m_event2 instanceof RemoveIncludedOIModel) {
            RemoveIncludedOIModel event2=(RemoveIncludedOIModel)m_event2;
            m_result=equals(event.getIncludedOIModel(),event2.getIncludedOIModel());
        }
    }
    /**
     * Checks whether two OI-models are equal.
     *
     * @param oimodel1              the first OI-model
     * @param oimodel2              the second OI-model
     * @return                      <code>true</code> if OI-models are equal
     */
    protected boolean equals(OIModel oimodel1,OIModel oimodel2) throws KAONException {
        return oimodel1.getLogicalURI().equals(oimodel2.getLogicalURI());
    }
    /**
     * Checks whether two concepts are equal.
     *
     * @param concept1              the first concept
     * @param concept2              the second concept
     * @return                      <code>true</code> if concepts are equal
     */
    protected boolean equals(Concept concept1,Concept concept2) throws KAONException {
        return concept1.getURI().equals(concept2.getURI());
    }
    /**
     * Checks whether two properties are equal.
     *
     * @param property1             the first property
     * @param property2             the second property
     * @return                      <code>true</code> if properties are equal
     */
    protected boolean equals(Property property1,Property property2) throws KAONException {
        return property1.getURI().equals(property2.getURI());
    }
    /**
     * Checks whether two instances are equal.
     *
     * @param instance1             the first instance
     * @param instance2             the second instance
     * @return                      <code>true</code> if instances are equal
     */
    protected boolean equals(Instance instance1,Instance instance2) throws KAONException {
        return instance1.getURI().equals(instance2.getURI());
    }
    /**
     * Checks whether two values are equal.
     *
     * @param value1                the first value
     * @param value2                the second value
     * @return                      <code>true</code> if values are equal
     */
    protected boolean valueEquals(Object value1,Object value2) throws KAONException {
        if (value1 instanceof Instance && value2 instanceof Instance)
            return equals((Instance)value1,(Instance)value2);
        else if (value1==null)
            return value1==value2;
        else
            return value1.equals(value2);
    }
    /**
     * Checks whether two property instances are equal.
     *
     * @param propertyInstance1     the first property instance
     * @param propertyInstance2     the second property instance
     * @return                      <code>true</code> if property instances are equal
     */
    protected boolean equals(PropertyInstance propertyInstance1,PropertyInstance propertyInstance2) throws KAONException {
        return equals(propertyInstance1.getProperty(),propertyInstance2.getProperty()) && equals(propertyInstance1.getSourceInstance(),propertyInstance2.getSourceInstance()) && valueEquals(propertyInstance1.getTargetValue(),propertyInstance2.getTargetValue());
    }
}
