package edu.unika.aifb.kaon.api.oimodel;

import java.util.Set;

import edu.unika.aifb.kaon.api.*;

/**
 * Represents a property between two concepts. One must not confuse this class with relations such as subclassOf or
 * subPropertyOf. SubclassOf and subPropertyOf are meta-model primitives - they are used to structure ontology elements.
 * Relations are modeling primitives - they are used to create models. A property links concepts, as opposed to attributes
 * that link concepts with literals.
 *
 * @author Raphael Volz (volz@aifb.uni-karlsruhe.de)
 * @author Boris Motik (boris.motik@fzi.de)
 */
public interface Property extends Entity {
    /**
     * Returns the inverse property of this property. If this property doesn't have an inverse property, then
     * <code>null</code> is returned.
     *
     * @return                          inverse property of this property (or <code>null</code> if there is no inverse property)
     */
    Property getInverseProperty() throws KAONException;
    /**
     * Returns the OI-model in which the inverse property of this property has been declared.
     *
     * @return                          the OI-model in which the inverse property of this property has been declared
     */
    OIModel getInversePropertyOIModel() throws KAONException;
    /**
     * Returns <code>true</code> if the inverse property of this property has been specified within the OI-model as returned
     * by <code>getOIModel()</code>.
     *
     * @return                          <code>true</code> if the inverse property of this property has been specified whithin the OI-model as returned by <code>getOIModel()</code>
     */
    boolean isInversePropertyDeclaredLocally() throws KAONException;
    /**
     * Returns <code>true</code> if this property is symmetric.
     *
     * @return                          <code>true</code> if this property is symmetric
     */
    boolean isSymmetric() throws KAONException;
    /**
     * Returns <code>true</code> if this property is transitive.
     *
     * @return                          <code>true</code> if this property is transitive
     */
    boolean isTransitive() throws KAONException;
    /**
     * Return all concepts that consitute the domain of this property.
     *
     * @return                          set of concepts that consitute the domain of this property (may be empty)
     */
    Set getAllDomainConcepts() throws KAONException;
    /**
     * Return concepts that consitute the domain of this property.
     *
     * @return                          set of concepts that consitute the domain of this property (may be empty)
     */
    Set getDomainConcepts() throws KAONException;
    /**
     * Checks whether supplied concept is in the domain of this property.
     *
     * @param concept                   concept that is checked
     * @return                          <code>true</code> if supplied concept is in the domain
     */
    boolean isDomainConcept(Concept concept) throws KAONException;
    /**
     * Returns the OI-model in which the supplied concept was added as the domain to this property.
     *
     * @param concept                   concept that is checked
     * @return                          the OI-model in which supplied concept was added as the domain to this property
     */
    OIModel getDomainConceptOIModel(Concept concept) throws KAONException;
    /**
     * Returns <code>true</code> if supplied concept was added as the domain to this property in
     * the model which returned by <code>getOIModel()</code>.
     *
     * @param concept                   the concept that is cheched
     * @return                          <code>true</code> if supplied concept was added to the domain concepts wihtin the <code>getOIModel()</code> model
     */
    boolean isDomainConceptDeclaredLocally(Concept concept) throws KAONException;
    /**
     * Returns the minimum cardinality for given domain concept.
     *
     * @param concept                   concept for which cardinality is requested
     * @return                          minimum cardinality of this property for given concept
     */
    int getMinimumCardinality(Concept concept) throws KAONException;
    /**
     * Returns the maximum cardinality for given domain concept. In case of unlimited cardinality, Integer.MAX_VALUE is returned.
     *
     * @param concept                   concept for which cardinality is requested
     * @return                          maximum cardinality of this property for given concept
     */
    int getMaximumCardinality(Concept concept) throws KAONException;
    /**
     * Returns the set of all range concepts of this property. If a property is an attribute, empty set is returned.
     *
     * @return                          set of range concepts of this property, or an empty set if property is an attribute
     */
    Set getAllRangeConcepts() throws KAONException;
    /**
     * Returns the set of range concepts of this property. If a property is an attribute, empty set is returned.
     *
     * @return                          set of range concepts of this property, or an empty set if property is an attribute
     */
    Set getRangeConcepts() throws KAONException;
    /**
     * Checks whether supplied concept is in the range of this property.
     *
     * @param concept                   concept that is checked
     * @return                          <code>true</code> if supplied concept is in the range
     */
    boolean isRangeConcept(Concept concept) throws KAONException;
    /**
     * Returns the OI-model in which the supplied concept was added as the range to this property.
     *
     * @param concept                   concept that is checked
     * @return                          the OI-model in which supplied concept was added as the range to this property
     */
    OIModel getRangeConceptOIModel(Concept concept) throws KAONException;
    /**
     * Returns <code>true</code> if supplied concept was added as the range to this property in
     * the model which returned by <code>getOIModel()</code>.
     *
     * @param concept                   the concept that is cheched
     * @return                          <code>true</code> if supplied concept was added to the range concepts wihtin the <code>getOIModel()</code> model
     */
    boolean isRangeConceptDeclaredLocally(Concept concept) throws KAONException;
    /**
     * Returns whether this property is an attribute (that is, if it has literal range).
     *
     * @return                          <code>true</code> if property is a literal
     */
    boolean isAttribute() throws KAONException;
    /**
     * Returns the set of direct subproperty of this property.
     *
     * @return                          set of subproperties (may be empty)
     */
    Set getSubProperties() throws KAONException;
    /**
     * Returns the set of all subproperties of this property.
     *
     * @return                          set of all subproperties (may be empty)
     */
    Set getAllSubProperties() throws KAONException;
    /**
     * Tests if this property is a direct subproperty of given property. If properties are the same, this method returns <code>false</code>.
     *
     * @param potentialSuperProperty    property being tested
     * @return                          <code>true</code> if this property is a direct subproperty
     */
    boolean isDirectSubPropertyOf(Property potentialSuperProperty) throws KAONException;
    /**
     * Returns the OI-model containing the this property - subproperty association.
     *
     * @param superProperty             the superproperty of this property
     * @return                          the OI-model in which the super-subproperty association is declared
     */
    OIModel getSuperSubPropertyOIModel(Property superProperty) throws KAONException;
    /**
     * Returns <code>true</code> if the supplied property has been declared a subproperty of this property in this OI-model.
     *
     * @param subProperty               the subproperty of this property
     * @return                          <code>true</code> if supplied property has been declared a subproperty of this property in this OI-model
     */
    boolean isSuperSubPropertyDeclaredLocally(Property subProperty) throws KAONException;
    /**
     * Tests if this property is a subproperty of given property. If properties are the same, this method returns <code>false</code>.
     *
     * @param potentialSuperProperty    property being tested
     * @return                          <code>true</code> if this property is a subproperty
     */
    boolean isSubPropertyOf(Property potentialSuperProperty) throws KAONException;
    /**
     * Returns the set of direct superproperties of this property.
     *
     * @return                          set of superproperties (may be empty)
     */
    Set getSuperProperties() throws KAONException;
    /**
     * Returns the set of all superproperties of this property.
     *
     * @return                          set of all superproperties (may be empty)
     */
    Set getAllSuperProperties() throws KAONException;
    /**
     * Returns the set of instances of this property.
     *
     * @return                          set of <code>PropertyInstance</code> objects of this property
     */
    Set getPropertyInstances() throws KAONException;
}
