package edu.unika.aifb.kaon.api.oimodel;

import java.util.Map;
import java.util.Set;
import java.util.List;

import edu.unika.aifb.kaon.api.*;

/**
 * Represents a single connection to some KAON provider. An OI-model is unique with respect to its
 * logical URI within a connection.
 *
 * @author Raphael Volz (volz@aifb.uni-karlsruhe.de)
 * @author Boris Motik (boris.motik@fzi.de)
 */
public interface KAONConnection {
    /** Logical URI of the root OI-model. */
    final String ROOT_OIMODEL_URI="http://kaon.semanticweb.org/2001/11/kaon-root";
    /** Logical URI of the lexical OI-model. */
    final String LEXICAL_OIMODEL_URI="http://kaon.semanticweb.org/2001/11/kaon-lexical";
    /** Logical URI of the evolution OI-model. */
    final String EVOLUTION_OIMODEL_URI="http://kaon.semanticweb.org/2002/06/kaon-evolution";

    /**
     * Returns the capabilities of the model.
     *
     * @return                          the bit-mask defining the model's capaibilities
     */
    int getCapabilities() throws KAONException;
    /**
     * Returns the parameters of this connection.
     *
     * @return                                  the parameters of this connection
     */
    Map getParameters() throws KAONException;
    /**
     * Closes this connection. All connections created by this connection are also closed.
     */
    void close() throws KAONException;
    /**
     * Returns <code>true</code> if this connection is open.
     *
     * @return                                  <code>true</code> if the connection is open
     */
    boolean isOpen() throws KAONException;
    /**
     * Returns a child connection. This can be used to implement advanced features, such as connection pooling.
     *
     * @return                                  a connection that is the child of this connection
     */
    KAONConnection getConnection() throws KAONException;
    /**
     * Opens an OI-model with given physical URI.
     *
     * @param physicalURI                       the physical URI of the OI-model
     */
    OIModel openOIModelPhysical(String physicalURI) throws KAONException;
    /**
     * Opens an OI-model with given logical URI.
     *
     * @param logicalURI                        the logical URI of the OI-model
     */
    OIModel openOIModelLogical(String logicalURI) throws KAONException;
    /**
     * Creates an OI-model with given physical URI.
     *
     * @param physicalURI                       the physical URI of the OI-model
     * @param logicalURI                        the logical URI of the OI-model
     */
    OIModel createOIModel(String physicalURI,String logicalURI) throws KAONException;
    /**
     * Creates a new OI-model with supplied logical and physical URI and makes it an exact duplicate of supplied OI-model.
     *
     * @param sourceOIModel             the source OI-model
     * @param physicalURI               the physical URI of the new model
     * @param logicalURI                the logical URI of the new model
     * @return                          newly created OI-model
     */
    OIModel createDuplicate(OIModel sourceOIModel,String physicalURI,String logicalURI) throws KAONException;
    /**
     * Notifies the connection that the OI-model has been deleted.
     *
     * @param oimodel                           OI-model that is removed
     */
    void notifyOIModelDeleted(OIModel oimodel) throws KAONException;
    /**
     * Returns the set of OI-models available at the node represented by this KAON connection.
     *
     * @return                                  the set of OIModel objects represented by this KAON connection
     */
    Set getAllOIModels() throws KAONException;
    /**
     * Returns the set of logical URIs of all OI-models available at the node represented by this KAON connection.
     *
     * @return                                  the set of OIModel objects represented by this KAON connection
     */
    Set getAllOIModelLogicalURIs() throws KAONException;
    /**
     * Returns the set of all open OI-model objects.
     *
     * @return                                  the set of all open OI-model objects
     */
    Set getOpenOIModels();
    /**
     * Applies the list of changes to the models in this connection.
     *
     * @param changeList                list of changes to the models in the connection
     */
    void applyChanges(List changeList) throws KAONException;
}
