package edu.unika.aifb.kaon.api.oimodel;

import java.util.Set;
import java.util.Map;

import edu.unika.aifb.kaon.api.*;

/**
 * Represents an instance of a concept.
 *
 * @author Raphael Volz (volz@aifb.uni-karlsruhe.de)
 * @author Boris Motik (boris.motik@fzi.de)
 */
public interface Instance extends Entity {
    /**
     * Returns set of <code>Concepts</code> objects that specify which concepts is this instance instance of.
     *
     * @return                                  set of concepts that this instance is instance of (may be empty)
     */
    Set getParentConcepts() throws KAONException;
    /**
     * Returns set of all <code>Concepts</code> objects that this instance is instance of.
     *
     * @return                                  set of all concepts that this instance is instance of (may be empty)
     */
    Set getAllParentConcepts() throws KAONException;
    /**
     * Tests if given concept is a direct parent of this instance.
     *
     * @param potentialParent           concept being tested
     * @return                          <code>true</code> if given concept is a direct parent of this instance
     */
    boolean isDirectParent(Concept potentialParent) throws KAONException;
    /**
     * Returns the OI-model containing the this concept - instance association.
     *
     * @param concept                   the concept of this instance
     * @return                          the OI-model in which the concept-instance association is declared
     */
    OIModel getConceptInstanceOIModel(Concept concept) throws KAONException;
    /**
     * Returns <code>true</code> if the supplied concept has been declared a parent of this instance in this OI-model.
     *
     * @param concept                   the concept of this instance
     * @return                          <code>true</code> if supplied concept has been declared a parent of this instance in this OI-model
     */
    boolean isConceptInstanceDeclaredLocally(Concept concept) throws KAONException;
    /**
     * Tests if given concept is a parent of this concept.
     *
     * @param potentialParent           concept being tested
     * @return                          <code>true</code> if given concept is a parent of this instance
     */
    boolean isParent(Concept potentialParent) throws KAONException;
    /**
     * Returns the set of <code>PropertyInstnace</code> objects that point out from this instance.
     *
     * @return                                  set of property instances pointing from this instance (may be empty)
     */
    Set getFromPropertyInstances() throws KAONException;
    /**
     * Returns the map of sets of property values going out of the instance. The map is indexed by <code>Property</code> objects.
     *
     * @return                                  map of sets of property values
     */
    Map getFromPropertyValues() throws KAONException;
    /**
     * Returns the value of a given property for this instance. If there is no value for a property, <code>null</code> is
     * returned. If there are multiple values for a property, one is picked arbitrarily and returned. Returned value is either
     * an <code>Instance</code> or a <code>String</code>.
     *
     * @param property                          property
     * @return                                  value of the property or <code>null</code> if there is no value
     */
    Object getFromPropertyValue(Property property) throws KAONException;
    /**
     * Returns the set of values of a given property. It there are no values, an empty set is returned. Elements of the set
     * may be <code>Instance</code> or <code>String</code> objects.
     *
     * @param property                          property
     * @return                                  set of values of the property
     */
    Set getFromPropertyValues(Property property) throws KAONException;
    /**
     * Tests whether this instance points to given value using specified property.
     *
     * @param property                          property
     * @param targetValue                       the value (can be a string or an instance)
     * @return                                  <code>true</code> if this instance points to given value through given property (by including all inferred facts)
     */
    boolean pointsToValue(Property property,Object targetValue) throws KAONException;
    /**
     * Returns the values of a property for this instance, including the inferred ones.
     *
     * @param property                          property for which the value is requested
     * @return                                  set of property values for this given property
     */
    Set getAllFromPropertyValues(Property property) throws KAONException;
    /**
     * Returns the values of all properties for this instance, including the inferred ones.
     *
     * @return                                  set of all property values for this instance
     */
    Map getAllFromPropertyValues() throws KAONException;
    /**
     * Returns the set of <code>PropertyInstnace</code> objects that point to this instance.
     *
     * @return                                  set of property instances pointing to this instance (may be empty)
     */
    Set getToPropertyInstances() throws KAONException;
    /**
     * Returns the map of sets of property values pointing to the instance. The map is indexed by <code>Property</code> objects.
     *
     * @return                                  map of sets of property values
     */
    Map getToPropertyValues() throws KAONException;
    /**
     * Returns the set of <code>Instnace</code> objects that point to this instance via given property.
     *
     * @param property                          property of interest
     * @return                                  set of <code>Instance</code> objects pointing to this instance via given property
     */
    Set getToPropertyValues(Property property) throws KAONException;
    /**
     * Returns the values of a property pointing to this instance, including the inferred ones.
     *
     * @param property                          property for which the value is requested
     * @return                                  set of all property values pointing to this instance for given property
     */
    Set getAllToPropertyValues(Property property) throws KAONException;
    /**
     * Returns the values of all properties pointing to this instance, including the inferred ones.
     *
     * @return                                  set of all property values pointing to this instance
     */
    Map getAllToPropertyValues() throws KAONException;
    /**
     * Returns the spanning concept for this instance.
     *
     * @return                                  spanning concept for this instance
     */
    Concept getSpanningConcept() throws KAONException;
    /**
     * Returns the spanning property for this instance.
     *
     * @return                                  spanning property for this instance
     */
    Property getSpanningProperty() throws KAONException;
}
