package edu.unika.aifb.kaon.api.oimodel;

import java.util.Set;

import edu.unika.aifb.kaon.api.*;

/**
 * Base interface for all elements in an OI-model.
 *
 * @author Raphael Volz (volz@aifb.uni-karlsruhe.de)
 * @author Boris Motik (boris.motik@fzi.de)
 */
public interface Entity {
    /**
     * Returns the bit mask specifying which parts of the object have been loaded.
     *
     * @return                          the bit mask specifying which parts of the object have been loaded
     */
    int getLoadedState() throws KAONException;
    /**
     * Erases the cached state of the object. The next time something is requested, it will be fetched from the storage.
     */
    void unload() throws KAONException;
    /**
     * Returns the URI of this property.
     *
     * @return                          URI of this property
     */
    String getURI() throws KAONException;
    /**
     * Retruns the version of this entity. Some implementations of the API can track changes by concurrent users to the entities. This
     * method can be used to detect the changes to entities.
     *
     * @return                          the version of the entity
     */
    int getVersion() throws KAONException;
    /**
     * Returns an OI-model of this entity.
     *
     * @return                          OI-model of this entity
     */
    OIModel getOIModel() throws KAONException;
    /**
     * Tests whether this entity exists as part of its associated OI-model.
     *
     * @return                          <code>true</code> if entity is a member in the OI-model
     */
    boolean isInOIModel() throws KAONException;
    /**
     * Returns <code>true</code> if this entity has been declared in the OI-model as returned
     * by <code>getOIModel()</code> call. Note that this may return <code>false</code> if this
     * entity was declared in an included OI-model.
     *
     * @return                          <code>true</code> if this entity case declared in the OI-model as returned by <code>getOIModel()</code> call
     */
    boolean isDeclaredLocally() throws KAONException;
    /**
     * Returns the OI-model where this entity was declared originally.
     *
     * @return                          the OI-model there this entity was declated originally
     */
    OIModel getSourceOIModel() throws KAONException;
    /**
     * Accepts an entity visitor.
     *
     * @param visitor                   visitor to apply
     */
    void accept(EntityVisitor visitor) throws KAONException;
    /**
     * Returns the spanning instance of this entity.
     *
     * @return                          spanning instance of this entiry
     */
    Instance getSpanningInstance() throws KAONException;
    /**
     * Returns all lexical entries associated with this entity.
     *
     * @return                              set of lexical entries associated with this entity
     */
    Set getLexicalEntries() throws KAONException;
    /**
     * Retruns all lexical entries of specified type associated with this entity.
     *
     * @param typeURI                       URI of the lexical entry type
     * @return                              set of lexical entries of given type
     */
    Set getLexicalEntries(String typeURI) throws KAONException;
    /**
     * Retruns all lexical entries of specified type and language associated with this entity.
     *
     * @param typeURI                       URI of the lexical entry type
     * @param languageURI                   URI of the language
     * @return                              set of lexical entries of given type
     */
    Set getLexicalEntries(String typeURI,String languageURI) throws KAONException;
    /**
     * Retruns a lexical entry of specified type and in given language associated with this entity. If there are more
     * lexical entries that match the criteria, one is picked and returned. If there are no lexical entries, <code>null</code>
     * is returned.
     *
     * @param typeURI                       URI of the lexical entry type
     * @param languageURI                   URI of the language
     * @return                              lexical entries of given type and language, or <code>null</code> if no entry is found
     */
    LexicalEntry getLexicalEntry(String typeURI,String languageURI) throws KAONException;
    /**
     * Returns the value of the lexical entry of given type in given language with given attribute name. If no appropriate
     * information is found, <code>null</code> is returned.
     *
     * @param typeURI                       URI of the lexical entry type
     * @param languageURI                   URI of the language
     * @param attributeURI                  URI name of the attribute that is requested
     * @return                              value of given attribute
     */
    String getLexicalAttribute(String typeURI,String languageURI,String attributeURI) throws KAONException;
    /**
     * Returns the label in given language.
     *
     * @param languageURI                   URI of the language
     * @return                              label in given language
     */
    String getLabel(String languageURI) throws KAONException;
}
