package edu.unika.aifb.kaon.api.oimodel;

import java.util.Set;

import edu.unika.aifb.kaon.api.*;

/**
 * Represents a concept in the ontoloty.
 *
 * @author Raphael Volz (volz@aifb.uni-karlsruhe.de)
 * @author Boris Motik (boris.motik@fzi.de)
 */
public interface Concept extends Entity {
    /**
     * Returns all properties starting from this concept.
     *
     * @return                          set of properties starting from this concept (may be empty)
     */
    Set getPropertiesFromConcept() throws KAONException;
    /**
     * Returns all properties (including inherited ones) starting from this concept.
     *
     * @return                          set of all properties starting from this concept (may be empty)
     */
    Set getAllPropertiesFromConcept() throws KAONException;
    /**
     * Returns all properties pointing to this concept.
     *
     * @return                          set of properties poining to this concept (may be empty)
     */
    Set getPropertiesToConcept() throws KAONException;
    /**
     * Returns all properties (including inherited ones) pointing to this concept.
     *
     * @return                          set of all properties pointing to this concept (may be empty)
     */
    Set getAllPropertiesToConcept() throws KAONException;
    /**
     * Returns direct subconcepts of this concept.
     *
     * @return                          subconcepts of this concept
     */
    Set getSubConcepts() throws KAONException;
    /**
     * Returns all subconcepts (inherited as well) of this concept.
     *
     * @return                          subconcepts of this concept
     */
    Set getAllSubConcepts() throws KAONException;
    /**
     * Tests if this concept is a direct subconcept of given concept. If concepts are the same, this method returns <code>false</code>.
     *
     * @param potentialSuperConcept     concept being tested
     * @return                          <code>true</code> if this concept is a direct subconcept
     */
    boolean isDirectSubConceptOf(Concept potentialSuperConcept) throws KAONException;
    /**
     * Returns the OI-model containing the this concept - subconcept association.
     *
     * @param superConcept              the superconcept of this concept
     * @return                          the OI-model in which the super-subconcept association is declared
     */
    OIModel getSuperSubConceptOIModel(Concept superConcept) throws KAONException;
    /**
     * Returns <code>true</code> if the this concept has been declared a subconcept of given concept in the OI-model as returned
     * by <code>getOIModel()</code> method.
     *
     * @param superConcept              the superconcept of this concept
     * @return                          <code>true</code> if thios concept has been declared a subconcept of given concept in this OI-model
     */
    boolean isSuperSubConceptDeclaredLocally(Concept superConcept) throws KAONException;
    /**
     * Tests if this concept is a subconcept of given concept. If concepts are the same, this method returns <code>false</code>.
     *
     * @param potentialSuperConcept     concept being tested
     * @return                          <code>true</code> if given concept is a subconcept
     */
    boolean isSubConceptOf(Concept potentialSuperConcept) throws KAONException;
    /**
     * Returns direct superconcepts of this concept.
     *
     * @return                          superconcepts of this concept
     */
    Set getSuperConcepts() throws KAONException;
    /**
     * Returns all superconcepts (inherited as well) of this concept.
     *
     * @return                          superconcepts of this concept
     */
    Set getAllSuperConcepts() throws KAONException;
    /**
     * Returns the set of direct instances of this concept.
     *
     * @return                          set of instances of this concept
     */
    Set getInstances() throws KAONException;
    /**
     * Returns the set of all instances (even of subconcepts) of a given concept in the pool.
     *
     * @return                          set of all instances of this concept
     */
    Set getAllInstances() throws KAONException;
}