package edu.unika.aifb.kaon.api.change;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

/**
 * Sets a transitivity flag for a property.
 *
 * @author Raphael Volz (volz@aifb.uni-karlsruhe.de)
 * @author Boris Motik (boris.motik@fzi.de)
 */
public class SetPropertyTransitive extends ChangeEvent {
    /** Property for which cardinality is set. */
    protected Property m_property;
    /** New transitivity value. */
    protected boolean m_transitive;
    /** Old transitivity value. */
    protected boolean m_oldTransitive;

    /**
     * Creates a request to set a transitivity flag for a property.
     *
     * @param property                  property
     * @param transitive                transitivity value
     */
    public SetPropertyTransitive(Property property,boolean transitive) {
        this(null,null,property,transitive);
    }
    /**
     * Creates a request to set a transitivity flag for a property.
     *
     * @param cause                     the cause event
     * @param property                  property
     * @param transitive                transitivity value
     */
    public SetPropertyTransitive(ChangeEvent cause,Property property,boolean transitive) {
        this(null,cause,property,transitive);
    }
    /**
     * Creates a request to set a transitivity flag for a property.
     *
     * @param oimodel                   the OI-model
     * @param cause                     the cause event
     * @param property                  property
     * @param transitive                transitivity value
     */
    public SetPropertyTransitive(OIModel oimodel,ChangeEvent cause,Property property,boolean transitive) {
        super(oimodel,cause);
        m_property=property;
        m_transitive=transitive;
    }
    /**
     * Returns the property.
     *
     * @return                          property
     */
    public Property getProperty() {
        return m_property;
    }
    /**
     * Returns the transitivity flag.
     *
     * @return                          transitivity flag
     */
    public boolean getTransitive() {
        return m_transitive;
    }
    /**
     * Returns the old transitivity flag.
     *
     * @return                          old transitivity flag
     */
    public boolean getOldTransitive() {
        return m_oldTransitive;
    }
    /**
     * Sets the old transitivity.
     *
     * @param oldTransitive             old transitivity
     */
    public void setOldTransitive(boolean oldTransitive) {
        m_oldTransitive=oldTransitive;
    }
    /**
     * Accepts a visitor for this change request.
     *
     * @param visitor                   visitor of this change
     * @throws KAONException            thrown if there is an error
     */
    public void accept(ChangeVisitor visitor) throws KAONException {
        visitor.visit(this);
    }
    /**
     * Tests whether this object is equal to some other object.
     *
     * @param that                      the other object
     * @return                          <code>true</code> if objects are equal
     */
    public boolean equals(Object that) {
        if (this==that)
            return true;
        if (!(that instanceof SetPropertyTransitive))
            return false;
        SetPropertyTransitive thatEvent=(SetPropertyTransitive)that;
        if (!equals(m_oimodel,thatEvent.m_oimodel))
            return false;
        if (!equals(m_property,thatEvent.m_property))
            return false;
        if (m_transitive!=thatEvent.m_transitive)
            return false;
        return true;
    }
    /**
     * Returns the hash-code of an object.
     *
     * @return                          the hash-code of the object
     */
    public int hashCode() {
        return (hashCode(m_oimodel)*7+hashCode(m_property))*7+hashCode(m_transitive);
    }
}
