package edu.unika.aifb.kaon.api.change;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

/**
 * Sets a property instance for an instance. All previous values for a relation are removed.
 *
 * @author Raphael Volz (volz@aifb.uni-karlsruhe.de)
 * @author Boris Motik (boris.motik@fzi.de)
 */
public class SetPropertyInstanceValue extends ChangeEvent {
    /** Property instance to set. */
    protected PropertyInstance m_propertyInstance;
    /** Property instances that were removed. */
    protected PropertyInstance[] m_removedInstances;

    /**
     * Sets a property instance for an instance.
     *
     * @param property                  property
     * @param sourceInstance            source instance
     * @param targetValue               target value
     * @throws KAONException            thrown if there is an error
     */
    public SetPropertyInstanceValue(Property property,Instance sourceInstance,Object targetValue) throws KAONException {
        this(null,null,property,sourceInstance,targetValue);
    }
    /**
     * Creates an event for setting the value of the property instance.
     *
     * @param propertyInstance          property instance to set
     */
    public SetPropertyInstanceValue(PropertyInstance propertyInstance) {
        this((OIModel)null,null,propertyInstance);
    }
    /**
     * Sets a property instance for an instance.
     *
     * @param cause                     the cause event
     * @param property                  property
     * @param sourceInstance            source instance
     * @param targetValue               target value
     * @throws KAONException            thrown if there is an error
     */
    public SetPropertyInstanceValue(ChangeEvent cause,Property property,Instance sourceInstance,Object targetValue) throws KAONException {
        this(null,cause,property,sourceInstance,targetValue);
    }
    /**
     * Creates an event for setting the value of the property instance.
     *
     * @param cause                     the cause event
     * @param propertyInstance          property instance to set
     */
    public SetPropertyInstanceValue(ChangeEvent cause,PropertyInstance propertyInstance) {
        this(null,cause,propertyInstance);
    }
    /**
     * Sets a property instance for an instance.
     *
     * @param oimodel                   the OI-model
     * @param cause                     the cause event
     * @param property                  property
     * @param sourceInstance            source instance
     * @param targetValue               target value
     * @throws KAONException            thrown if there is an error
     */
    public SetPropertyInstanceValue(OIModel oimodel,ChangeEvent cause,Property property,Instance sourceInstance,Object targetValue) throws KAONException {
        this(oimodel,cause,property.getOIModel().getPropertyInstance(property,sourceInstance,targetValue));
    }
    /**
     * Creates an event for setting the value of the property instance.
     *
     * @param oimodel                   the OI-model
     * @param cause                     the cause event
     * @param propertyInstance          property instance to set
     */
    public SetPropertyInstanceValue(OIModel oimodel,ChangeEvent cause,PropertyInstance propertyInstance) {
        super(oimodel,cause);
        m_propertyInstance=propertyInstance;
    }
    /**
     * Returns the property instance being set.
     *
     * @return                          property instance
     */
    public PropertyInstance getPropertyInstance() {
        return m_propertyInstance;
    }
    /**
     * Returns the array of property instances that were removed.
     *
     * @return                          property instances that were removed
     */
    public PropertyInstance[] getRemovedInstances() {
        return m_removedInstances;
    }
    /**
     * Sets the array of property instances that were removed.
     *
     * @param removedInstances          property instances that were removed
     */
    public void setRemovedInstances(PropertyInstance[] removedInstances) {
        m_removedInstances=removedInstances;
    }
    /**
     * Accepts a visitor for this change request.
     *
     * @param visitor                   visitor of this change
     * @throws KAONException            thrown if there is an error
     */
    public void accept(ChangeVisitor visitor) throws KAONException {
        visitor.visit(this);
    }
    /**
     * Tests whether this object is equal to some other object.
     *
     * @param that                      the other object
     * @return                          <code>true</code> if objects are equal
     */
    public boolean equals(Object that) {
        if (this==that)
            return true;
        if (!(that instanceof SetPropertyInstanceValue))
            return false;
        SetPropertyInstanceValue thatEvent=(SetPropertyInstanceValue)that;
        if (!equals(m_oimodel,thatEvent.m_oimodel))
            return false;
        if (!equals(m_propertyInstance,thatEvent.m_propertyInstance))
            return false;
        return true;
    }
    /**
     * Returns the hash-code of an object.
     *
     * @return                          the hash-code of the object
     */
    public int hashCode() {
        return hashCode(m_oimodel)*7+hashCode(m_propertyInstance);
    }
}
