package edu.unika.aifb.kaon.api.change;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

/**
 * Sets a minimium cardinality of a concept for some property.
 *
 * @author Raphael Volz (volz@aifb.uni-karlsruhe.de)
 * @author Boris Motik (boris.motik@fzi.de)
 */
public class SetMinimumCardinality extends ChangeEvent {
    /** Property for which cardinality is set. */
    protected Property m_property;
    /** Concept for which cardinality is set. */
    protected Concept m_concept;
    /** Cardinality. */
    protected int m_cardinality;
    /** Old cardinality value. */
    protected int m_oldCardinality;

    /**
     * Creates a request to set a minimum cardinality for the property and concept.
     *
     * @param property                  property
     * @param concept                   concept
     * @param cardinality               cardinality to set
     */
    public SetMinimumCardinality(Property property,Concept concept,int cardinality) {
        this(null,null,property,concept,cardinality);
    }
    /**
     * Creates a request to set a minimum cardinality for the property and concept.
     *
     * @param cause                     the cause event
     * @param property                  property
     * @param concept                   concept
     * @param cardinality               cardinality to set
     */
    public SetMinimumCardinality(ChangeEvent cause,Property property,Concept concept,int cardinality) {
        this(null,cause,property,concept,cardinality);
    }
    /**
     * Creates a request to set a minimum cardinality for the property and concept.
     *
     * @param oimodel                   the OI-model
     * @param cause                     the cause event
     * @param property                  property
     * @param concept                   concept
     * @param cardinality               cardinality to set
     */
    public SetMinimumCardinality(OIModel oimodel,ChangeEvent cause,Property property,Concept concept,int cardinality) {
        super(oimodel,cause);
        m_property=property;
        m_concept=concept;
        m_cardinality=cardinality;
    }
    /**
     * Returns the property.
     *
     * @return                          property
     */
    public Property getProperty() {
        return m_property;
    }
    /**
     * Returns the concept.
     *
     * @return                          concept
     */
    public Concept getConcept() {
        return m_concept;
    }
    /**
     * Returns the cardinality.
     *
     * @return                          cardinality
     */
    public int getCardinality() {
        return m_cardinality;
    }
    /**
     * Returns the old cardinality.
     *
     * @return                          old cardinality
     */
    public int getOldCardinality() {
        return m_oldCardinality;
    }
    /**
     * Sets the old cardinality.
     *
     * @param oldCardinality            old cardinality
     */
    public void setOldCardinality(int oldCardinality) {
        m_oldCardinality=oldCardinality;
    }
    /**
     * Accepts a visitor for this change request.
     *
     * @param visitor                   visitor of this change
     * @throws KAONException            thrown if there is an error
     */
    public void accept(ChangeVisitor visitor) throws KAONException {
        visitor.visit(this);
    }
    /**
     * Tests whether this object is equal to some other object.
     *
     * @param that                      the other object
     * @return                          <code>true</code> if objects are equal
     */
    public boolean equals(Object that) {
        if (this==that)
            return true;
        if (!(that instanceof SetMinimumCardinality))
            return false;
        SetMinimumCardinality thatEvent=(SetMinimumCardinality)that;
        if (!equals(m_oimodel,thatEvent.m_oimodel))
            return false;
        if (!equals(m_property,thatEvent.m_property))
            return false;
        if (!equals(m_concept,thatEvent.m_concept))
            return false;
        if (m_cardinality!=thatEvent.m_cardinality)
            return false;
        return true;
    }
    /**
     * Returns the hash-code of an object.
     *
     * @return                          the hash-code of the object
     */
    public int hashCode() {
        return ((hashCode(m_oimodel)*7+hashCode(m_property))*7+hashCode(m_concept))*7+m_cardinality;
    }
}
