package edu.unika.aifb.kaon.api.change;

import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;
import java.util.Set;
import java.util.HashSet;
import java.util.Map;
import java.util.HashMap;
import java.util.Collections;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.util.*;
import edu.unika.aifb.kaon.evolutionlog.*;

/**
 * Strategy for handling evolution issues for distributed ontologies.
 *
 * @author Ljiljana Stojanovic (Ljiljana.Stojanovic@fzi.de)
 * @author Boris Motik (boris.motik@fzi.de)
 */
public class DistributedEvolutionStrategy {
    /** KAON connection that finds all local OI-models. */
    protected KAONConnection m_kaonConnection;
    /** The evolution logs for the KAONConnection. */
    protected KAONConnectionEvolutionLogs m_kaonConnectionEvolutionLogs;
    /** Map that contains all included oimodels for one oimodel. */
    protected Map m_allIncludedOIModels;
    /** Map that contains version number of the original oimodel. */
    protected Map m_originalVersion;
    /** Map that contains version number of the oimodel copy. */
    protected Map m_copyVersion;

    /**
     * Creates an instance of this class.
     *
     * @param kaonConnection            the KAON connection for which the evolution is performed
     */
    public DistributedEvolutionStrategy(KAONConnection kaonConnection) {
        m_kaonConnection=kaonConnection;
        m_kaonConnectionEvolutionLogs=new KAONConnectionEvolutionLogs();
        m_allIncludedOIModels=new HashMap();
        m_originalVersion=new HashMap();
        m_copyVersion=new HashMap();
    }
    /**
     * Distribution evolution for an oimodel. Returns a list of changes.
     *
     * @param oimodel                   oimodel that should be updated
     * @return                          the list of changes to be applied to node
     * @throws KAONException            thrown if there is an error
     */
    public List updateFromOriginal(OIModel oimodel) throws KAONException {
        Set set=identificationOfChangedOriginals(oimodel);
        return extractMergeDeltas(set);
    }
    /**
     * Returns the list of changes for the set of oimodels.
     *
     * @param oimodels                  oimodels that should be updated
     * @return                          the list of changes to be applied to node
     * @throws KAONException            thrown if there is an error
     */
    public List resolveRequest(Set oimodels) throws KAONException {
        // Eliminate redundant oimodels
        Set improvedOIModels=improveRequest(oimodels);
        // Test each oimodel
        List list=new LinkedList();
        Iterator iterator=improvedOIModels.iterator();
        while (iterator.hasNext()) {
            OIModel oimodel=(OIModel)iterator.next();
            list.addAll(updateFromOriginal(oimodel));
        }
        return list;
    }
    /**
     * Finds included replicated OIModels that are changed.
     *
     * @param oimodel                   oimodel that should be updated
     * @return                          the set of changed originals
     * @throws KAONException            thrown if there is an error
     */
    protected Set identificationOfChangedOriginals(OIModel oimodel) throws KAONException {
        // find first level replicas
        Set replicas=findFirtsLevelReplicas(oimodel);
        // replace replicas with including replicas
        replicas=findIncludingReplicas(replicas);
        // Check replication consistency for the originals of replicas
        Set notUpdatedSet=new HashSet();
        Iterator iterator=replicas.iterator();
        while (iterator.hasNext()) {
            OIModel currentOIModel=(OIModel)iterator.next();
            notUpdatedSet.addAll(checkReplicaConsistency(currentOIModel));
        }
        if (!notUpdatedSet.isEmpty())
            throw new DependentEvolutionKAONException(notUpdatedSet,"Included models are not updated yet!");
        // Eliminate replicas that are up-to-date
        Set result=new HashSet();
        iterator=replicas.iterator();
        while (iterator.hasNext()) {
            OIModel currentOIModel=(OIModel)iterator.next();
            if (getCopyVersion(currentOIModel)!=getOriginalVersion(currentOIModel))
                result.add(currentOIModel);
        }
        return result;
    }
    /**
     * Finds top level replicas for the current set of replicas.
     *
     * @param replicas                  set of replicas
     * @return                          the set of including replicas
     * @throws KAONException            thrown if there is an error
     */
    protected Set findIncludingReplicas(Set replicas) throws KAONException {
        Set result=new HashSet();
        Iterator iter=replicas.iterator();
        while(iter.hasNext()) {
            OIModel replica=(OIModel)iter.next();
            result.addAll(findIncludingReplicas(replica));
        }
        return result;
    }
    /**
     * Finds top level replicas for the given replica
     *
     * @param replica                   oimodel
     * @return                          the set of including replicas
     * @throws KAONException            thrown if there is an error
     */
    protected Set findIncludingReplicas(OIModel replica) throws KAONException {
        Set parentReplicas=new HashSet();
        Set parents=replica.getAllIncludedByOIModels();
        Iterator parentIter=parents.iterator();
        while(parentIter.hasNext()) {
            OIModel parent=(OIModel)parentIter.next();
            if ((isReplica(parent)) && (parent.getIncludedOIModels().contains(replica))) {
                parentReplicas.add(parent);
            }
        }
        if(parentReplicas.size()==0)
            return new HashSet(Collections.singletonList(replica));
        else {
            Set result=new HashSet();
            Iterator iter=parentReplicas.iterator();
            while(iter.hasNext()) {
                OIModel oimodel=(OIModel)iter.next();
                result.addAll(findIncludingReplicas(oimodel));
            }
            return result;
        }
    }
    /**
     * Finds first level replicas for the given oimodel.
     *
     * @param oimodel                   oimodel whose first level replicas are searched for
     * @return                          the first-level replica
     * @throws KAONException            thrown if there is an error
     */
    protected Set findFirtsLevelReplicas(OIModel oimodel) throws KAONException {
        if (isReplica(oimodel))
            return new HashSet(Collections.singleton(oimodel));
        else {
            Set set=oimodel.getIncludedOIModels();
            Set result=new HashSet();
            Iterator iterator=set.iterator();
            while (iterator.hasNext()) {
                OIModel includedOIModel=(OIModel)iterator.next();
                result.addAll(findFirtsLevelReplicas(includedOIModel));
            }
            return result;
        }
    }
    /**
     * Finds all included oimodels.
     *
     * @param oimodel                   the oimodel
     * @return                          all included oimodels
     * @throws KAONException            thrown if there is an error
     */
    protected Set getAllIncludedOIModels(OIModel oimodel) throws KAONException {
        Set set=(Set)m_allIncludedOIModels.get(oimodel);
        if (set==null) {
            set=new HashSet(oimodel.getIncludedOIModels());
            Set result=new HashSet(set);
            Iterator iterator=set.iterator();
            while (iterator.hasNext()) {
                OIModel currentOIModel=(OIModel)iterator.next();
                result.addAll(getAllIncludedOIModels(currentOIModel));
            }
            m_allIncludedOIModels.put(oimodel,result);
            return result;
        }
        else
            return set;
    }
    /**
     * Checks whether oimodel is up-to-date or not.
     *
     * @param replicaOIModel            the oimodel
     * @return                          empty set when oimodel is up-to-date, or set of out-of-date included oimodels
     * @throws KAONException            thrown if there is an error
     */
    protected Set checkReplicaConsistency(OIModel replicaOIModel) throws KAONException {
        OIModel originalOIModel=getOriginalForReplica(replicaOIModel);
        //Check whether all included oimodels are up-to-date
        Set notUpdatedSet=new HashSet();
        Set allSet=getAllIncludedOIModels(originalOIModel);
        Iterator iterator=allSet.iterator();
        while (iterator.hasNext()) {
            OIModel currentOIModel=(OIModel)iterator.next();
            if (getCopyVersion(currentOIModel)!=getOriginalVersion(currentOIModel))
                notUpdatedSet.add(currentOIModel);
        }
        return notUpdatedSet;
    }
    /**
     * Returns the original OI-model for given replica.
     *
     * @param replicaOIModel            the replica for which the original is required
     * @return                          the original for the replica
     * @throws KAONException            thrown if replica can't be opened
     * @throws KAONException            thrown if there is an error
     */
    protected OIModel getOriginalForReplica(OIModel replicaOIModel) throws KAONException {
        String originalPhysicalURI=replicaOIModel.getAttribute("OIModel.replicatedFromPhysicalURI");
        Map parameters=new HashMap();
        parameters.put(KAONManager.KAON_CONNECTION,"edu.unika.aifb.kaon.apionrdf.KAONConnectionImpl");
        KAONConnection connection=KAONManager.getKAONConnection(parameters);
        return connection.openOIModelPhysical(originalPhysicalURI);
    }
    /**
     * Returns the version number of the original oimodel.
     *
     * @param oimodel                   oimodel
     * @return                          the version of the original
     * @throws KAONException            thrown if there is an error
     */
    protected int getOriginalVersion(OIModel oimodel) throws KAONException {
        if (isReplica(oimodel)) {
            OIModel originalOIModel=getOriginalForReplica(oimodel);
            return getVersion(originalOIModel,m_originalVersion);
        }
        else
            return getVersion(oimodel,m_originalVersion);
    }
    /**
     * Returns the version number of the local oimodel.
     *
     * @param oimodel                   oimodel
     * @return                          the version of the copy
     * @throws KAONException            thrown if there is an error
     */
    protected int getCopyVersion(OIModel oimodel) throws KAONException {
        return getVersion(oimodel,m_copyVersion);
    }
    /**
     * Returns the version number of the oimodel.
     *
     * @param oimodel                   oimodel
     * @param map                       map of the oimodels.
     * @return                          the version of the OIModel
     * @throws KAONException            thrown if there is an error
     */
    protected int getVersion(OIModel oimodel,Map map) throws KAONException {
        String version=(String)map.get(oimodel);
        if (version==null) {
            version=oimodel.getAttribute("OIModel.version");
            if (version==null)
                version="0";
            map.put(oimodel,version);
        }
        return Integer.parseInt(version);
    }
    /**
     * Collects and merges changes from the evolution logs.
     *
     * @param set                       the set of OIModels
     * @return                          the list of delta change events
     * @throws KAONException            thrown if there is an error
     */
    protected List extractMergeDeltas(Set set) throws KAONException {
        List allChanges=new LinkedList();
        Iterator iterator=set.iterator();
        while (iterator.hasNext()) {
            OIModel oimodel=(OIModel)iterator.next();
            EvolutionLog evolutionLog=m_kaonConnectionEvolutionLogs.getEvolutionLog(oimodel);
            try {
                List changes=evolutionLog.readChanges(getCopyVersion(oimodel));
                allChanges=mergeDeltas(changes,allChanges);
            }
            finally {
                evolutionLog.close();
            }
        }
        return allChanges;
    }
    /**
     * Returns the merged version of two lists.
     *
     * @param list1                     first list
     * @param list2                     second list
     * @return                          the merged list
     * @throws KAONException            thrown if there is an error
     */
    protected List mergeDeltas(List list1,List list2) throws KAONException {
        ChangeEqualityTester tester=new ChangeEqualityTester();
        final List changes=new LinkedList();
        ChangeEventCopier copier=new ChangeEventCopier() {
            protected boolean preprocessEvent(ChangeEvent event) throws KAONException {
                super.preprocessEvent(event);
                m_oimodel=m_kaonConnection.openOIModelLogical(event.getOIModel().getLogicalURI());
                return m_oimodel!=null;
            }
            protected void consumeEvent(ChangeEvent changeEvent) {
                changes.add(changeEvent);
            }
        };
        List controlList=new LinkedList();
        Iterator iterator= list1.iterator();
        while(iterator.hasNext()) {
            ChangeEvent event=(ChangeEvent)iterator.next();
            if (listContains(event,list2,tester))
                controlList.add(event);
        }
        Iterator iterator1=list1.iterator();
        Iterator iterator2=list2.iterator();
        iterator=controlList.iterator();
        while (iterator.hasNext()) {
            ChangeEvent event=(ChangeEvent)iterator.next();
            boolean found=false;
            while (iterator1.hasNext() && !found) {
                ChangeEvent event1=(ChangeEvent)iterator1.next();
                if (tester.equals(event1,event))
                    found=true;
                else
                    event1.accept(copier);
            }
            found=false;
            while (iterator2.hasNext() && !found) {
                ChangeEvent event2=(ChangeEvent)iterator2.next();
                if (tester.equals(event2,event))
                    found=true;
                else
                    event2.accept(copier);
            }
            changes.add(event);
        }
        while (iterator1.hasNext()) {
            ChangeEvent event1=(ChangeEvent)iterator1.next();
            event1.accept(copier);
        }
        while (iterator2.hasNext()) {
            ChangeEvent event2=(ChangeEvent)iterator2.next();
            event2.accept(copier);
        }
        return changes;
    }
    /**
     * Checks whether an event is a list.
     *
     * @param event                     the event being tested
     * @param list                      the list being tested
     * @param tester                    the equality tester
     * @return                          <code>true</code> if the list contains an event
     * @throws KAONException            thrown if there is an error
     */
    protected boolean listContains(ChangeEvent event,List list,ChangeEqualityTester tester) throws KAONException {
        Iterator iterator=list.iterator();
        while (iterator.hasNext()) {
            ChangeEvent currentEvent=(ChangeEvent)iterator.next();
            if (tester.equals(event,currentEvent))
                return true;
        }
        return false;
    }
    /**
     * Excludes the redundant oimodels and including oimodels.
     *
     * @param oimodels                  the set of OIModels
     * @return                          the purged set of OIModels
     */
    protected Set improveRequest(Set oimodels) {
        Set set=new HashSet();
        Iterator iterator=oimodels.iterator();
        while(iterator.hasNext()) {
            OIModel oimodel=(OIModel)iterator.next();
            Iterator innerIter=oimodels.iterator();
            boolean found=false;
            while (innerIter.hasNext() && !found) {
                OIModel currentOIModel=(OIModel)innerIter.next();
                if (oimodels.contains(currentOIModel))
                    found=true;
            }
            if (!found)
                set.add(oimodel);
        }
        return set;
    }
    /**
     * Checks whether oimodel is replica or not.
     *
     * @param oimodel                   oimodel
     * @return                          <code>true</code> if the OIModel is a replica
     * @throws KAONException            thrown if there is an error
     */
    public static boolean isReplica(OIModel oimodel) throws KAONException {
        return oimodel.getAttribute("OIModel.replicatedFromPhysicalURI")==null ? false : true;
    }
    /**
     * Makes supplied OI-model an original.
     *
     * @param oimodel                   oimodel
     * @throws KAONException            thrown if there is an error
     */
    public static void makeOriginal(OIModel oimodel) throws KAONException {
        oimodel.setAttribute("OIModel.replicatedFromPhysicalURI",null);
    }
}
