package edu.unika.aifb.kaon.api.change;

import edu.unika.aifb.kaon.api.*;
import edu.unika.aifb.kaon.api.oimodel.*;

/**
 * Removes an existing property instance and adds a new property instance with given value.
 *
 * @author Raphael Volz (volz@aifb.uni-karlsruhe.de)
 * @author Boris Motik (boris.motik@fzi.de)
 */
public class ChangePropertyInstanceValue extends ChangeEvent {
    /** Existing property instance whose value is changed. */
    protected PropertyInstance m_propertyInstance;
    /** The new value of the property instance. */
    protected Object m_newTargetValue;

    /**
     * Changes a property instance value for an instance.
     *
     * @param property                  property
     * @param sourceInstance            source instance
     * @param targetValue               target value
     * @param newTargetValue            the new target value
     * @throws KAONException            thrown if there is an error
     */
    public ChangePropertyInstanceValue(Property property,Instance sourceInstance,Object targetValue,Object newTargetValue) throws KAONException {
        this(null,null,property,sourceInstance,targetValue,newTargetValue);
    }
    /**
     * Changes a property instance value for an instance.
     *
     * @param propertyInstance          property instance to set
     * @param newTargetValue            the new target value
     */
    public ChangePropertyInstanceValue(PropertyInstance propertyInstance,Object newTargetValue) {
        this((OIModel)null,null,propertyInstance,newTargetValue);
    }
    /**
     * Changes a property instance value for an instance.
     *
     * @param cause                     the cause event
     * @param property                  property
     * @param sourceInstance            source instance
     * @param targetValue               target value
     * @param newTargetValue            the new target value
     * @throws KAONException            thrown if there is an error
     */
    public ChangePropertyInstanceValue(ChangeEvent cause,Property property,Instance sourceInstance,Object targetValue,Object newTargetValue) throws KAONException {
        this(null,cause,property,sourceInstance,targetValue,newTargetValue);
    }
    /**
     * Changes a property instance value for an instance.
     *
     * @param cause                     the cause event
     * @param propertyInstance          property instance to set
     * @param newTargetValue            the new target value
     */
    public ChangePropertyInstanceValue(ChangeEvent cause,PropertyInstance propertyInstance,Object newTargetValue) {
        this(null,cause,propertyInstance,newTargetValue);
    }
    /**
     * Changes a property instance value for an instance.
     *
     * @param oimodel                   the OI-model
     * @param cause                     the cause event
     * @param property                  property
     * @param sourceInstance            source instance
     * @param targetValue               target value
     * @param newTargetValue            the new target value
     * @throws KAONException            thrown if there is an error
     */
    public ChangePropertyInstanceValue(OIModel oimodel,ChangeEvent cause,Property property,Instance sourceInstance,Object targetValue,Object newTargetValue) throws KAONException {
        this(oimodel,cause,property.getOIModel().getPropertyInstance(property,sourceInstance,targetValue),newTargetValue);
    }
    /**
     * Changes a property instance value for an instance.
     *
     * @param oimodel                   the OI-model
     * @param cause                     the cause event
     * @param propertyInstance          property instance to set
     * @param newTargetValue            the new target value
     */
    public ChangePropertyInstanceValue(OIModel oimodel,ChangeEvent cause,PropertyInstance propertyInstance,Object newTargetValue) {
        super(oimodel,cause);
        m_propertyInstance=propertyInstance;
        m_newTargetValue=newTargetValue;
    }
    /**
     * Returns the property instance being set.
     *
     * @return                          property instance
     */
    public PropertyInstance getPropertyInstance() {
        return m_propertyInstance;
    }
    /**
     * Returns the new target value of the property instance.
     *
     * @return                          the new target value of the property instance
     */
    public Object getNewTargetValue() {
        return m_newTargetValue;
    }
    /**
     * Accepts a visitor for this change request.
     *
     * @param visitor                   visitor of this change
     * @throws KAONException            thrown if there is an error
     */
    public void accept(ChangeVisitor visitor) throws KAONException {
        visitor.visit(this);
    }
    /**
     * Tests whether this object is equal to some other object.
     *
     * @param that                      the other object
     * @return                          <code>true</code> if objects are equal
     */
    public boolean equals(Object that) {
        if (this==that)
            return true;
        if (!(that instanceof ChangePropertyInstanceValue))
            return false;
        ChangePropertyInstanceValue thatEvent=(ChangePropertyInstanceValue)that;
        if (!equals(m_oimodel,thatEvent.m_oimodel))
            return false;
        if (!equals(m_propertyInstance,thatEvent.m_propertyInstance))
            return false;
        if (!equals(m_newTargetValue,thatEvent.m_newTargetValue))
            return false;
        return true;
    }
    /**
     * Returns the hash-code of an object.
     *
     * @return                          the hash-code of the object
     */
    public int hashCode() {
        return (hashCode(m_oimodel)*7+hashCode(m_propertyInstance))*7+hashCode(m_newTargetValue);
    }
}
