package edu.unika.aifb.kaon.api;

import java.util.Map;
import java.util.HashMap;
import java.util.List;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Collections;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;

import edu.unika.aifb.kaon.api.oimodel.*;
import edu.unika.aifb.kaon.api.util.*;

/**
 * A manager for factories of OI-models.
 *
 * @author Raphael Volz (volz@aifb.uni-karlsruhe.de)
 * @author Boris Motik (boris.motik@fzi.de)
 */
public class KAONManager {
    /** Parameter specifying the OI-model connection class. */
    public static final String KAON_CONNECTION="KAON_CONNECTION";
    /** Type specifier for our constructors. */
    protected static final Class[] KAON_CONNECTION_PARAMETERS=new Class[] { Map.class };
    /** The parameters for the RDF connections. */
    public static final Map s_parametersAPIOnRDF;
    static {
        Map parameters=new HashMap();
        parameters.put(KAON_CONNECTION,"edu.unika.aifb.kaon.apionrdf.KAONConnectionImpl");
        s_parametersAPIOnRDF=Collections.unmodifiableMap(parameters);
    }

    /** The map of registered physical URI resolvers. */
    protected static final Map s_physicalURIToDefaultParametersResolvers=new HashMap();
    /** The list of registered logical URI resolvers. */
    protected static final List s_logicalURIResolvers=new ArrayList();

    static {
        // register well-known ontologies
        registerWellKnownRDFOIModelEx(KAONManager.class.getResource("res/kaon-root.xml").toString());
        registerWellKnownRDFOIModelEx(KAONManager.class.getResource("res/kaon-lexical.xml").toString());
        registerWellKnownRDFOIModelEx(KAONManager.class.getResource("res/kaon-evolution.xml").toString());
        // start well-known KAON implementations
        try {
            startKAONImplementation("edu.unika.aifb.kaon.apionrdf.KAONConnectionImpl");
        }
        catch (KAONException ignored) {
        }
        try {
            startKAONImplementation("edu.unika.aifb.kaon.engineeringserver.client.RemoteEngineeringSource");
        }
        catch (KAONException ignored) {
        }
        try {
            startKAONImplementation("edu.unika.aifb.kaon.engineeringserver.client.LocalEngineeringSource");
        }
        catch (KAONException ignored) {
        }
        try {
            startKAONImplementation("edu.unika.aifb.kaon.engineeringserver.client.DirectEngineeringSource");
        }
        catch (KAONException ignored) {
        }
    }

    /**
     * Starts the KAON API implementation with given name.
     *
     * @param connectionClassName               the name of the KAON Connection class
     * @throws KAONException                    thrown if the implementation cannot be started
     */
    public static void startKAONImplementation(String connectionClassName) throws KAONException {
        try {
            Class.forName(connectionClassName);
        }
        catch (ClassNotFoundException e) {
            throw new KAONException("Cannot load KAON connection class '"+connectionClassName+"'",e);
        }
    }
    /**
     * Returns a factory to use for given parameters.
     *
     * @param parameters                        parameters specifying the class
     * @return                                  the KAON connection for given parameters
     * @throws KAONException                    thrown if parameters are incorrect
     */
    public static KAONConnection getKAONConnection(Map parameters) throws KAONException {
        String connectionClass=(String)parameters.get(KAON_CONNECTION);
        if (connectionClass==null)
            throw new KAONException("Parameters must specify the connection class through 'KAON_CONNECTION' key.");
        try {
            Class clazz=Class.forName(connectionClass);
            Constructor constructor=clazz.getConstructor(KAON_CONNECTION_PARAMETERS);
            Object object=constructor.newInstance(new Object[] { parameters });
            if (!(object instanceof KAONConnection))
                throw new KAONException("Supplied object is not a KAONConnection.");
            return (KAONConnection)object;
        }
        catch (ClassNotFoundException e) {
            throw new KAONException("Cannot load KAON connection class '"+connectionClass+"'",e);
        }
        catch (NoSuchMethodException e) {
            throw new KAONException("Cannot instantiate KAON connection class '"+connectionClass+"'",e);
        }
        catch (InvocationTargetException e) {
            throw new KAONException("Cannot instantiate KAON connection class '"+connectionClass+"'",e);
        }
        catch (InstantiationException e) {
            throw new KAONException("Cannot instantiate KAON connection class '"+connectionClass+"'",e);
        }
        catch (IllegalAccessException e) {
            throw new KAONException("Cannot instantiate KAON connection class '"+connectionClass+"'",e);
        }
    }
    /**
     * Returns the default parameters for given physical URI. This method is useful for opening models from any URI.
     * If default parameters cannot be created, an exception is thrown.
     *
     * @param physicalURI               the physical URI
     * @param contextParameters         the parameters that are used for missing elements (may be <code>null</code>)
     * @return                          the default parameters for accessing given physical URI (never <code>null</code>)
     * @throws KAONException            thrown if givan physical URI cannot be accessed
     */
    public static synchronized Map getDefaultParametersForPhysicalURI(String physicalURI,Map contextParameters) throws KAONException {
        Iterator iterator=s_physicalURIToDefaultParametersResolvers.values().iterator();
        while (iterator.hasNext()) {
            PhysicalURIToDefaultParametersResolver resolver=(PhysicalURIToDefaultParametersResolver)iterator.next();
            Map parameters=resolver.getDefaultParametersForPhysicalURI(physicalURI,contextParameters);
            if (parameters!=null)
                return parameters;
        }
        throw new KAONException("Cannot resolve physical URI '"+physicalURI+"' to default parameters.");
    }
    /**
     * Registers a resolver from physical URI to default parameters. Only one resolver of each class can be registered.
     *
     * @param resolver                  the resolver
     */
    public static synchronized void registerPhysicalURIToDefaultParametersResolver(PhysicalURIToDefaultParametersResolver resolver) {
        String className=resolver.getClass().getName();
        s_physicalURIToDefaultParametersResolvers.put(className,resolver);
    }
    /**
     * Registers a resolver from physical URI to default parameters by its class name.
     *
     * @param resolverClassName         the name of the resolver class
     * @throws KAONException            thrown if resolver cannot be registered
     */
    public static synchronized void registerPhysicalURIToDefaultParametersResolver(String resolverClassName) throws KAONException {
        try {
            Class clazz=Class.forName(resolverClassName);
            Object object=clazz.newInstance();
            if (!(object instanceof PhysicalURIToDefaultParametersResolver))
                throw new KAONException("Supplied object is not a PhysicalURIToDefaultParametersResolver");
            registerPhysicalURIToDefaultParametersResolver((PhysicalURIToDefaultParametersResolver)object);
        }
        catch (ClassNotFoundException e) {
            throw new KAONException("Cannot load resolver class '"+resolverClassName+"'",e);
        }
        catch (InstantiationException e) {
            throw new KAONException("Cannot instantiate resolver class '"+resolverClassName+"'",e);
        }
        catch (IllegalAccessException e) {
            throw new KAONException("Cannot instantiate resolver class '"+resolverClassName+"'",e);
        }
    }
    /**
     * Tries to resolve a logical URI for the connection. This method is called when a connection is unable to obtain a model
     * with given logical URI. It then tries to resolve the logical URI to parameters and the physical URI. If model cannot be
     * opened through the same connection, then model is replicated.
     *
     * @param kaonConnection            the connection
     * @param logicalURI                the logical URI
     * @return                          the OI-model (<code>null</code> if logical URI cannot be resolved)
     * @throws KAONException            thrown if there is an error
     */
    public static OIModel resolveLogicalURI(KAONConnection kaonConnection,String logicalURI) throws KAONException {
        List resolvers;
        synchronized (KAONManager.class) {
            resolvers=new ArrayList(s_logicalURIResolvers);
        }
        for (int i=0;i<resolvers.size();i++) {
            LogicalURIResolver resolver=(LogicalURIResolver)resolvers.get(i);
            LogicalURIResolver.ResultHolder result=resolver.resolveLogicalURI(logicalURI);
            if (result!=null) {
                if (canUseConnectionForParameters(kaonConnection,result.m_connectionParameters))
                    return kaonConnection.openOIModelPhysical(result.m_physicalURI);
                else {
                    // if the connection cannot be reused, then try to copy the model with a blank physical URI
                    KAONConnection sourceConnection=getKAONConnection(result.m_connectionParameters);
                    try {
                        OIModel sourceOIModel=sourceConnection.openOIModelPhysical(result.m_physicalURI);
                        Map logicalToPhysicalURIsMap=new HashMap();
                        logicalToPhysicalURIsMap.put(logicalURI,"");
                        OIModelReplicator replicator=new OIModelReplicator(sourceOIModel,logicalToPhysicalURIsMap,kaonConnection);
                        try {
                            return replicator.doReplication();
                        }
                        catch (InterruptedException error) {
                            throw new KAONException("Model replication interrupted",error);
                        }
                    }
                    finally {
                        sourceConnection.close();
                    }
                }
            }
        }
        return null;
    }
    /**
     * Registers a logical URI resolver.
     *
     * @param resolver                          the resolver
     */
    public static synchronized void registerLogicalURIResolver(LogicalURIResolver resolver) {
        s_logicalURIResolvers.add(resolver);
    }
    /**
     * Unregisters a logical URI resolver.
     *
     * @param resolver                          the resolver
     */
    public static synchronized void unregisterLogicalURIResolver(LogicalURIResolver resolver) {
        s_logicalURIResolvers.remove(resolver);
    }
    /**
     * Reegisters a well-known RDF OI-model.
     *
     * @param wellKnownOIModelPhysicalURI       the physical URI of the well-known model
     * @throws KAONException                    thrown if there is an error
     */
    public static void registerWellKnownRDFOIModel(final String wellKnownOIModelPhysicalURI) throws KAONException {
        KAONConnection connection=getKAONConnection(s_parametersAPIOnRDF);
        final String wellKnownOIModelLogicalURI;
        try {
            OIModel oimodel=connection.openOIModelPhysical(wellKnownOIModelPhysicalURI);
            wellKnownOIModelLogicalURI=oimodel.getLogicalURI();
        }
        finally {
            connection.close();
        }
        registerLogicalURIResolver(new LogicalURIResolver() {
            public ResultHolder resolveLogicalURI(String logicalURI) {
                if (wellKnownOIModelLogicalURI.equals(logicalURI)) {
                    ResultHolder result=new ResultHolder();
                    result.m_physicalURI=wellKnownOIModelPhysicalURI;
                    result.m_connectionParameters=s_parametersAPIOnRDF;
                    return result;
                }
                else
                    return null;
            }
        });
    }
    /**
     * Checks if supplied connection can be used for opening a model with given parameters.
     *
     * @param connection                        the connection that is checked if it can be reused
     * @param parameters                        the parameters that are checked
     * @return                                  <code>true</code> if the supplied connection can be reused for given parameters
     */
    public static boolean canUseConnectionForParameters(KAONConnection connection,Map parameters) {
        Map connectionParameters;
        try {
            connectionParameters=connection.getParameters();
        }
        catch (KAONException e) {
            return false;
        }
        Iterator keys=parameters.keySet().iterator();
        while (keys.hasNext()) {
            Object key=keys.next();
            if (!connectionParameters.containsKey(key))
                return false;
            Object connectionValue=connectionParameters.get(key);
            Object parametersValue=parameters.get(key);
            if (connectionValue==null) {
                if (parametersValue!=null)
                    return false;
            }
            else {
                if (!connectionValue.equals(parametersValue) && !"???".equals(parametersValue))
                    return false;
            }
        }
        return true;
    }
    /**
     * A private version of the method that ignores the exception.
     *
     * @param wellKnownOIModelPhysicalURI       the physical URI of the well-known model
     */
    protected static void registerWellKnownRDFOIModelEx(String wellKnownOIModelPhysicalURI) {
        try {
            registerWellKnownRDFOIModel(wellKnownOIModelPhysicalURI);
        }
        catch (KAONException ignored) {
        }
    }
}
