package de.fzi.wim.guibase.viewableselector;

import java.util.List;
import java.util.Comparator;
import java.util.Arrays;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.awt.Component;
import javax.swing.JComboBox;
import javax.swing.JList;
import javax.swing.ComboBoxModel;
import javax.swing.AbstractListModel;
import javax.swing.DefaultListCellRenderer;

import de.fzi.wim.guibase.appdriver.*;

/**
 * Combo box displaying the list of viewables of a given module. For each viewable the title of the viewable is displayed,
 * and vieables are sorted by title.
 */
public class ViewableSelector extends JComboBox {
    /** Module whose viewables are displayed. */
    protected Module m_module;

    /**
     * Creates a viewable selector and specifies the module whose viewables will be displayed.
     *
     * @param module                            module whose viewables are to be displayed
     */
    public ViewableSelector(Module module) {
        m_module=module;
        setModel(new ViewableSelectorModel());
        setRenderer(new ViewableSelectorRenderer());
    }
    /**
     * Destroys this object. Must be called when this object goes out of scope, or memory leaks will occur.
     */
    public void dispose() {
        ((ViewableSelectorModel)getModel()).dispose();
    }
    /**
     * Returns the currently selected viewable.
     *
     * @return                                  the currently selected viewable
     */
    public Viewable getSelectedViewable() {
        return (Viewable)getSelectedItem();
    }
    /**
     * Returns the index of the currently selected viewable.
     *
     * @return                                  the index of the currently selected viewable
     */
    public int getSelectedViewableIndex() {
        return m_module.getAppDriver().getViewables().indexOf(getSelectedViewable());
    }
    /**
     * Selects the currently open viewable with given index.
     *
     * @param index                             the index of the viewable
     */
    public void setSelectedViewableIndex(int index) {
        List viewables=m_module.getAppDriver().getViewables();
        if (index<viewables.size()) {
            Viewable viewable=(Viewable)viewables.get(index);
            setSelectedItem(viewable);
        }
    }

    /**
     * The model for the viewable selector combo box.
     */
    protected class ViewableSelectorModel extends AbstractListModel implements ComboBoxModel,PropertyChangeListener {
        /** Array of available viewables. */
        protected Viewable[] m_viewables;
        /** Currently selected viewable. */
        protected Viewable m_currentViewable;

        /**
         * Creates an instance of this class and attaches it to the given module.
         */
        public ViewableSelectorModel() {
            m_module.getAppDriver().addPropertyChangeListener(this);
            updateViewablesList();
            if (m_viewables.length!=0)
                m_currentViewable=m_viewables[0];
        }
        /**
         * Destroys this object.
         */
        public void dispose() {
            m_module.getAppDriver().removePropertyChangeListener(this);
        }
        /**
         * Returns the object with given index.
         *
         * @param index                         the index
         * @return                              object with given index
         */
        public Object getElementAt(int index) {
            return m_viewables[index];
        }
        /**
         * Returns the number of objects in the model.
         *
         * @return                              the number of objects in the model
         */
        public int getSize() {
            return m_viewables.length;
        }
        /**
         * Returns the currently selected object.
         *
         * @return                              the currently selected object
         */
        public Object getSelectedItem() {
            return m_currentViewable;
        }
        /**
         * Sets the new object.
         *
         * @param anItem                        new current value
         */
        public void setSelectedItem(Object anItem) {
            if ((m_currentViewable==null && anItem!=null) || (m_currentViewable!= null && !m_currentViewable.equals(anItem))) {
	            m_currentViewable=null;
                for (int i=0;i<m_viewables.length;i++)
                    if (m_viewables[i].equals(anItem)) {
                        m_currentViewable=(Viewable)anItem;
                        break;
                    }
        	    fireContentsChanged(this,-1,-1);
            }
        }
        /**
         * Called when appdriver's property is changed.
         *
         * @param event                         the event object
         */
        public void propertyChange(PropertyChangeEvent event) {
            if (AppDriver.VIEWABLES_PROPERTY.equals(event.getPropertyName()))
                updateViewablesList();
        }
        /**
         * Updates the list of available viewables.
         */
        protected void updateViewablesList() {
            List viewables=m_module.getAppDriver().getViewables(m_module);
            if (shouldUpdateModel(viewables)) {
                m_viewables=new Viewable[viewables.size()];
                viewables.toArray(m_viewables);
                Arrays.sort(m_viewables,ViewableComparator.INSTANCE);
                boolean selectionIsInNewList=false;
                for (int i=0;i<m_viewables.length;i++)
                    if (m_viewables[i].equals(m_currentViewable)) {
                        selectionIsInNewList=true;
                        break;
                    }
                if (!selectionIsInNewList || m_currentViewable==null)
                    if (m_viewables.length==0)
                        m_currentViewable=null;
                    else
                        m_currentViewable=m_viewables[0];
        	    fireContentsChanged(this,-1,-1);
            }
        }
        /**
         * Checks whether should update the model.
         *
         * @param viewables                     the set of viewables
         * @return                              <code>true</code> if the model should be updated
         */
        protected boolean shouldUpdateModel(List viewables) {
            if (m_viewables==null || m_viewables.length!=viewables.size())
                return true;
            for (int i=0;i<m_viewables.length;i++)
                if (!viewables.contains(m_viewables[i]))
                    return true;
            return false;
        }
    }

    /**
     * The renderer class for this combo box that writes out the titles of the viewables.
     */
    protected static class ViewableSelectorRenderer extends DefaultListCellRenderer {
        public Component getListCellRendererComponent(JList list,Object value,int index,boolean isSelected,boolean cellHasFocus) {
            String text=null;
            if (value instanceof Viewable)
                text=((Viewable)value).getViewableAnchor().getTitle();
            return super.getListCellRendererComponent(list,text,index,isSelected,cellHasFocus);
        }
    }

    /**
     * Comparator for sorting viewables by their title.
     */
    protected static class ViewableComparator implements Comparator {
        public static final Comparator INSTANCE=new ViewableComparator();

        public int compare(Object o1,Object o2) {
            Viewable v1=(Viewable)o1;
            Viewable v2=(Viewable)o2;
            return v1.getViewableAnchor().getTitle().compareToIgnoreCase(v2.getViewableAnchor().getTitle());
        }
    }
}