package de.fzi.wim.guibase.util;

import java.util.List;
import java.util.LinkedList;

/**
 * Implements a worker thread pool that can be used to execute tasks in the backgound. Thread pool can be initialized with
 * particular number of threads and can be given an upper limit on the number of threads in the pool.
 */
public class ThreadPool {
    /** Constant specifying unlimited maximum number of threas in the pool. */
    public static final int UNLIMITED=Integer.MAX_VALUE;

    /** Number of maximum threads in the pool. */
    protected int m_maximumThreadNumber;
    /** Current number of threads in the pool. */
    protected int m_currentThreadNumber;
    /** List of available threads in the pool. */
    protected List m_availableThreads;
    /** Thread group that all threads in this pool belong to. */
    protected ThreadGroup m_threadGroup;

    /**
     * Creates and initializes this thread pool.
     *
     * @param initialThreadNumber                   initial number of threads in the pool
     * @param maximumThreadNumber                   maximum number of threads in the pool
     */
    public ThreadPool(int initialThreadNumber,int maximumThreadNumber) {
        m_threadGroup=new ThreadGroup("ThreadPool");
        m_threadGroup.setDaemon(true);
        m_maximumThreadNumber=maximumThreadNumber;
        m_currentThreadNumber=0;
        m_availableThreads=new LinkedList();
        for (int i=0;i<initialThreadNumber;i++)
            createThread();
    }
    /**
     * Executes a task on one of the worker threads in the pool. If there are no available threads, this method will wait
     * until threads are available.
     *
     * @param runnable                              the task to execute
     * @throws InterruptedException                 thrown if the calling thread has been interrupted
     */
    public void executeTask(Runnable runnable) throws InterruptedException {
        WorkerThread thread;
        synchronized (this) {
            if (m_availableThreads.size()==0)
                createThread();
            while (m_availableThreads.size()==0)
                wait();
            thread=(WorkerThread)m_availableThreads.get(0);
            m_availableThreads.remove(0);
        }
        thread.executeTask(runnable);
    }
    /**
     * Creates a thread if possible and puts it in the thread pool. Makes sure that total number of threads doesn't exceed
     * maximum number of threads.
     */
    private void createThread() {
        if (m_currentThreadNumber<m_maximumThreadNumber) {
            WorkerThread thread=new WorkerThread(m_threadGroup,"ThreadPool-"+m_currentThreadNumber);
            m_availableThreads.add(thread);
            m_currentThreadNumber++;
        }
    }
    /**
     * Interrupts all threads in the thread pool.
     */
    public synchronized void interrupt() {
        m_threadGroup.interrupt();
        m_availableThreads=new LinkedList();
    }
    /**
     * Returns the current number of active tasks.
     *
     * @return                                      the number of active tasks
     */
    public synchronized int numberOfActiveTasks() {
        return m_currentThreadNumber-m_availableThreads.size();
    }

    /**
     * One worker thread in teh pool.
     */
    protected class WorkerThread extends Thread {
        /** Task that this thread is currently executing. */
        protected Runnable m_task;

        /**
         * Creates a worker thread.
         *
         * @param threadGroup                       group to which this thread must belong to
         * @param threadName                        the name of the thread
         */
        public WorkerThread(ThreadGroup threadGroup,String threadName) {
            super(threadGroup,threadName);
            setDaemon(true);
            start();
        }
        /**
         * Executes a task on this worker thread.
         *
         * @param task                              task to be executed
         */
        public synchronized void executeTask(Runnable task) {
            m_task=task;
            notify();
        }
        /**
         * Main method of every thread that is in charge of executing tasks.
         */
        public void run() {
            while (true) {
                Runnable task=null;
                synchronized (this) {
                    while (m_task==null)
                        try {
                            wait();
                        }
                        catch (InterruptedException interrupted) {
                            return;
                        }
                    task=m_task;
                    m_task=null;
                }
                // execute a task
                try {
                    task.run();
                }
                catch (ThreadDeath error) {
                    throw error;
                }
                catch (Throwable error) {
                    error.printStackTrace();
                }
                synchronized (ThreadPool.this) {
                    if (isInterrupted())
                        return;
                    m_availableThreads.add(0,this);
                    ThreadPool.this.notifyAll();
                }
            }
        }
    }
}