package de.fzi.wim.guibase.util;

import java.io.File;
import java.util.Map;
import java.util.HashMap;
import java.util.Iterator;
import javax.swing.filechooser.FileFilter;

import de.fzi.wim.guibase.localization.LocalizationManager;

/**
 * A filter implementation that takes several extensions and combines them.
 */
public class ExtensionFileFilter extends FileFilter {
    protected int m_filterID;
    protected Map m_extensions;
    protected String m_description;
    protected String m_fullDescription;
    protected String m_defaultExtension;

    /**
     * Creates a file filter from the given string array and description.
     *
     * @param localizationManager           the localization manager
     * @param descriptionID                 the ID of the description text
     * @param filters                       array of extensions
     */
    public ExtensionFileFilter(LocalizationManager localizationManager,String descriptionID,String[] filters) {
        this(0,localizationManager,descriptionID,filters);
    }
    /**
     * Creates a file filter from the given string array and description.
     *
     * @param filterID                      the ID of the filter
     * @param localizationManager           the localization manager
     * @param descriptionID                 the ID of the description text
     * @param filters                       array of extensions
     */
    public ExtensionFileFilter(int filterID,LocalizationManager localizationManager,String descriptionID,String[] filters) {
        m_filterID=filterID;
        m_extensions=new HashMap();
        m_description=localizationManager.getPhrase(descriptionID);
        if (filters!=null)
            for (int i=0;i<filters.length;i++)
                addExtension(filters[i]);
    }
    /**
     * Return true if this file should be shown in the directory pane,
     * false if it shouldn't. Files that begin with "." are ignored.
     *
     * @param file                          the file
     * @return                              <code>true</code> if supplied file is accepted by this filter
     */
    public boolean accept(File file) {
        if (file!=null) {
            if (file.isDirectory())
                return true;
            String extension=getExtension(file);
            if (extension!=null && m_extensions.containsKey(extension))
                return true;
        }
        return false;
    }
    /**
     * Return the extension portion of the file's name.
     *
     * @param file                          the file
     * @return                              the extension of the file
     */
    public String getExtension(File file) {
        if (file!=null) {
            String fileName=file.getName();
            int i=fileName.lastIndexOf('.');
            if (i>0 && i<fileName.length()-1)
                return fileName.substring(i+1).toLowerCase();
        }
        return null;
    }
    /**
     * Adds a filetype "dot" extension to filter against.
     *
     * @param extension                     the extension being added
     */
    public void addExtension(String extension) {
        extension=extension.toLowerCase();
        m_extensions.put(extension,this);
        m_fullDescription=null;
        if (m_defaultExtension==null)
            m_defaultExtension=extension;
    }
    /**
     * Returns the human readable description of this filter.
     * For example: "JPEG and GIF Image Files (*.jpg, *.gif)"
     *
     * @return                              the human readable description
     */
    public String getDescription() {
        if (m_fullDescription==null) {
            m_fullDescription=m_description==null ? "(" : m_description+" (";
            Iterator extensions=m_extensions.keySet().iterator();
            if (extensions.hasNext()) {
                m_fullDescription+="."+(String)extensions.next();
                while (extensions.hasNext())
                    m_fullDescription+=", ."+(String)extensions.next();
            }
            m_fullDescription+=")";
        }
        return m_fullDescription;
    }
    /**
     * Returns the default extension.
     *
     * @return                              the default extension
     */
    protected String getDefaultExtension() {
        return m_defaultExtension;
    }
    /**
     * Changes the file by adding the default extension.
     *
     * @param file                          the file
     * @return                              the changed file
     */
    public File mangleFileName(File file) {
        if (m_defaultExtension!=null) {
            String fileName=file.toString();
            int dotPosition=fileName.lastIndexOf('.');
            int separatorPosition=fileName.lastIndexOf(File.separatorChar);
            if ((dotPosition==-1 || separatorPosition>dotPosition) && !fileName.endsWith(m_defaultExtension))
                file=new File(fileName+"."+m_defaultExtension);
        }
        return file;
    }
    /**
     * Returns the ID of this filter.
     *
     * @return                              the ID of this filter
     */
    public int getFilterID() {
        return m_filterID;
    }
}
