package de.fzi.wim.guibase.treetable;

import java.awt.Component;
import java.awt.Rectangle;
import java.awt.Graphics;
import java.awt.Point;
import javax.swing.JComponent;
import javax.swing.JTable;
import javax.swing.Icon;
import javax.swing.table.TableCellRenderer;
import javax.swing.UIManager;
import javax.swing.CellRendererPane;
import javax.swing.tree.TreePath;

/**
 * Renderer for the column in the table designated as tree. This class will render the expansion handles, use {@link TreeNodeRenderer}
 * from tree table to render the tree node and then use standard renderer to render the value of the node.
 */
public class TreeTableCellRenderer extends JComponent implements TableCellRenderer {
    /** The distance of the handle control from the tree node. */
    protected static final int HANDLE_DISTANCE=3;

    /** Utility rectangle that us reused in many circumstances. */
    protected Rectangle m_utilRect=new Rectangle();
    /** The component used to paint recent tree node. */
    protected Component m_treeNodeComponent;
    /** The componend used to paint the cell with the value. */
    protected Component m_cellComponent;
    /** Indentation of the recent node. */
    protected int m_treeNodeX;
    /** Icon used to paint expanded handles. */
    protected Icon m_expandedIcon;
    /** Icon used to paint collapsed handles. */
    protected Icon m_collapsedIcon;
    /** Actual icon used to paint handles. */
    protected Icon m_paintIcon;
    /** Cell renderer pane used for paining cells. */
    protected CellRendererPane m_cellRendererPane;

    /**
     * Creates and initializes an instance of this class from L&F.
     */
    public TreeTableCellRenderer() {
        m_cellRendererPane=new CellRendererPane();
        setExpandedIcon(UIManager.getIcon("Tree.expandedIcon"));
        setCollapsedIcon(UIManager.getIcon("Tree.collapsedIcon"));
    }
    /**
     * Tests whether a point lies in the handle control.
     *
     * @param treeTable                             table for which the test is perofrmed
     * @param point                                 point that is tested
     * @return                                      <code>true</code> if point lies in the handle control
     */
    public boolean isInHandleControl(JTreeTable treeTable,Point point) {
        int rowIndex=treeTable.rowAtPoint(point);
        if (rowIndex==-1)
            return false;
        int columnIndex=treeTable.columnAtPoint(point);
        if (columnIndex==-1)
            return false;
        Rectangle cellBounds=treeTable.getCellRect(rowIndex,columnIndex,false);
        getHandleControlRect(treeTable,rowIndex,m_utilRect);
        return m_utilRect.contains(point.x-cellBounds.x,point.y-cellBounds.y);
    }
    /**
     * Returns the icon used to paint expanded handles.
     *
     * @return                                      icon used to paint expanded handles
     */
    public Icon getExpandedIcon() {
        return m_expandedIcon;
    }
    /**
     * Sets the icon used to paint expanded handles.
     *
     * @param icon                                  icon used to paint expanded handles
     */
    public void setExpandedIcon(Icon icon) {
        m_expandedIcon=icon;
    }
    /**
     * Returns the icon used to paint collapsed handles.
     *
     * @return                                      icon used to paint collapsed handles
     */
    public Icon getCollapsedIcon() {
        return m_collapsedIcon;
    }
    /**
     * Sets the icon used to paint collapsed handles.
     *
     * @param icon                                  icon used to paint collapsed handles
     */
    public void setCollapsedIcon(Icon icon) {
        m_collapsedIcon=icon;
    }
    /**
     * Returns the component used to paint a cell in the table.
     *
     * @param table                                 table for which this renrerer is used
     * @param value                                 value being painted
     * @param isSelected                            <code>true</code> if the value is selected
     * @param hasFocus                              <code>true</code> if value has focus
     * @param row                                   row of the cell to be painted
     * @param column                                column of the cell to be painted
     * @return                                      the component used to paint a cell
     */
    public Component getTableCellRendererComponent(JTable table,Object value,boolean isSelected,boolean hasFocus,int row,int column) {
        JTreeTable treeTable=(JTreeTable)table;
        TableCellRenderer tableCellrenderer=treeTable.getPlainCellRenderer(row,column);
        m_cellComponent=tableCellrenderer.getTableCellRendererComponent(table,value,isSelected,hasFocus,row,column);
        if (m_cellComponent instanceof JComponent) {
            JComponent component=(JComponent)m_cellComponent;
            setBorder(component.getBorder());
            component.setBorder(null);
            setToolTipText(component.getToolTipText());
        }
        else {
            setBorder(null);
            setToolTipText(null);
        }
        setBackground(m_cellComponent.getBackground());
        TreeNodeRenderer treeNodeRenderer=treeTable.getTreeNodeRenderer();
        if (treeNodeRenderer!=null)
            m_treeNodeComponent=treeNodeRenderer.getTableCellRendererComponent(treeTable,isSelected,hasFocus,row,column);
        else
            m_treeNodeComponent=null;
        m_treeNodeX=getTreeNodeX(treeTable,row);
        boolean isRootNode=(row==0 && treeTable.isRootVisible());
        m_paintIcon=null;
        if (!isRootNode || treeTable.getShowsRootHandles()) {
            TreeTableModel model=treeTable.getTreeTableModel();
            TreePath path=treeTable.getPathForRow(row);
            if (!model.isLeaf(path.getLastPathComponent()))
                if (treeTable.isExpanded(path))
                    m_paintIcon=m_expandedIcon;
                else
                    m_paintIcon=m_collapsedIcon;
        }
        return this;
    }
    /**
     * Returns the horizontal coordinate of the node in the table.
     *
     * @param treeTable                             tree table for which information is requested
     * @param row                                   row of the node for which indent is requested
     * @return                                      indent of the node in the tree
     */
    public int getTreeNodeX(JTreeTable treeTable,int row) {
        int indent=treeTable.getTreeNodeIndent(row);
        if (treeTable.getShowsRootHandles() && treeTable.isRootVisible())
            indent+=m_expandedIcon.getIconWidth();
        return indent;
    }
    /**
     * Returns the rectangle of the handle control.
     *
     * @param handleIcon                            icon used to paint the handle
     * @param treeNodeX                             indentation level of the node
     * @param rowHeight                             height of the row
     * @param rectangle                             place to put information into
     * @return                                      rectangle of the handle control
     */
    protected Rectangle getHandleControlRect(Icon handleIcon,int treeNodeX,int rowHeight,Rectangle rectangle) {
        int iconWidth=handleIcon.getIconWidth();
        int iconHeight=handleIcon.getIconHeight();
        rectangle.setBounds(treeNodeX-iconWidth-HANDLE_DISTANCE,(rowHeight-iconHeight)/2,iconWidth,iconHeight);
        return rectangle;
    }
    /**
     * Returns the rectangle of the handle control.
     *
     * @param treeTable                             tree table for which information is requested
     * @param row                                   row for which the handle rectangle is requested
     * @param rectangle                             place to put information into
     * @return                                      rectangle of the handle control
     */
    public Rectangle getHandleControlRect(JTreeTable treeTable,int row,Rectangle rectangle) {
        int treeNodeX=getTreeNodeX(treeTable,row);
        return getHandleControlRect(m_expandedIcon,treeNodeX,treeTable.getRowHeight(row),rectangle);
    }
    /**
     * Paints the tree table cell node. This method combines the handle icon, tree node renderer and standard cell renderer
     * to paint the cell.
     *
     * @param g                                     where to paint the node to
     */
    public void paintComponent(Graphics g) {
        super.paintComponent(g);
        getBounds(m_utilRect);
        int cellWidth=m_utilRect.width;
        int cellHeight=m_utilRect.height;
        g.setColor(getBackground());
        g.fillRect(0,0,m_utilRect.width,m_utilRect.height);
        if (m_paintIcon!=null) {
            getHandleControlRect(m_paintIcon,m_treeNodeX,cellHeight,m_utilRect);
            m_paintIcon.paintIcon(this,g,m_utilRect.x,m_utilRect.y);
        }
        int treeNodeWidth=0;
        if (m_treeNodeComponent!=null) {
            treeNodeWidth=m_treeNodeComponent.getPreferredSize().width;
            m_treeNodeComponent.setBounds(m_treeNodeX,0,treeNodeWidth,cellHeight);
            m_cellRendererPane.paintComponent(g,m_treeNodeComponent,this,m_treeNodeX,0,treeNodeWidth,cellHeight,true);
        }
        m_cellRendererPane.paintComponent(g,m_cellComponent,this,m_treeNodeX+treeNodeWidth,0,cellWidth-m_treeNodeX-treeNodeWidth,cellHeight,true);
    }
}
