package de.fzi.wim.guibase.treetable;

import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.event.EventListenerList;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.event.TreeExpansionEvent;
import javax.swing.event.TreeExpansionListener;
import javax.swing.tree.TreeSelectionModel;
import javax.swing.tree.TreePath;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeEvent;

/**
 * This class implements a <code>javax.swing.ListSelectionModel</code> wrapper around a <code>javax.swing.tree.TreeSelectionModel</code>.
 * This wrapper is used as the list selection model in the tree table.
 */
class TreeSelectionModelAdapter implements ListSelectionModel,TreeSelectionListener {
    /** Constant representing the minimum selection index. */
    protected static final int MIN=-1;
    /** Constant representing the maximum selection index. */
    protected static final int MAX=Integer.MAX_VALUE;

    /** List of registered event listeners. */
    protected EventListenerList m_listenerList=new EventListenerList();
    /** Wrapped tree selection model. */
    protected TreeSelectionModel m_treeSelectionModel;
    /** Tree mapper used to transform tree into table format. */
    protected TreeMapper m_treeMapper;
    /** Set to <code>true</code> if value of the model is currently being adjusted. */
    protected boolean m_valueIsAdjusting;
    /** Index of the anchor selection. */
    protected int m_anchorSelectionIndex;
    /** Index of the lead selection. */
    protected int m_leadSelectionIndex;
    /** Minimum index changed while value is adjusting. */
    protected int m_minChangedIndex;
    /** Maximum index changed while value is adjusting. */
    protected int m_maxChangedIndex;

    /**
     * Creates and initializes an instance of this class.
     *
     * @param treeTable                     tree table that this model is being used in
     * @param treeMapper                    tree mapper used to transform the tree into table
     */
    public TreeSelectionModelAdapter(JTreeTable treeTable,TreeMapper treeMapper) {
        m_treeMapper=treeMapper;
        m_treeSelectionModel=treeTable.getTreeSelectionModel();
        m_valueIsAdjusting=false;
        m_anchorSelectionIndex=-1;
        m_leadSelectionIndex=-1;
        m_minChangedIndex=MAX;
        m_maxChangedIndex=MIN;
        treeTable.addPropertyChangeListener(createPropertyChangeListener());
        treeTable.addTreeExpansionListener(createTreeExpansionListener());
    }
    /**
     * Sets the wrapped tree seleciton model.
     *
     * @param treeSelectionModel            new wrapped tree selection model
     */
    public void setTreeSelectionModel(TreeSelectionModel treeSelectionModel) {
        if (m_treeSelectionModel!=null)
            m_treeSelectionModel.removeTreeSelectionListener(this);
        m_treeSelectionModel=treeSelectionModel;
        if (m_treeSelectionModel!=null)
            m_treeSelectionModel.addTreeSelectionListener(this);
    }
    /**
     * Called then selection in the tree is changed.
     * This method is an implementation side-effect, since this class implements <code>TreeSelectionListener</code> interface.
     *
     * @param e                             object containing information about tree selection change
     */
    public void valueChanged(TreeSelectionEvent e) {
        TreePath[] paths=e.getPaths();
        int minIndex=MAX;
        int maxIndex=MIN;
        for (int i=0;i<paths.length;i++) {
            int rowForPath=m_treeMapper.getRowForPath(paths[i]);
            minIndex=Math.min(minIndex,rowForPath);
            maxIndex=Math.max(maxIndex,rowForPath);
        }
        fireValueChanged(minIndex,maxIndex,getValueIsAdjusting());
    }
    /**
     * Updates anchor and lead selection index, as well as the minimum and maximum changed indices.
     *
     * @param anchor                        anchor of the new selection
     * @param lead                          lead of the selection
     */
    protected void updateChangedIndices(int anchor,int lead) {
        m_anchorSelectionIndex=anchor;
        m_leadSelectionIndex=lead;
        if (getValueIsAdjusting()) {
            m_minChangedIndex=Math.min(m_minChangedIndex,anchor);
            m_minChangedIndex=Math.min(m_minChangedIndex,lead);
            m_maxChangedIndex=Math.max(m_maxChangedIndex,anchor);
            m_maxChangedIndex=Math.max(m_maxChangedIndex,lead);
        }
    }
    /**
     * Returns anchor selection index or -1 if no anchor has even been set.
     *
     * @return                              anchor selection index or -1
     */
    public int getAnchorSelectionIndex() {
        return m_anchorSelectionIndex;
    }
    /**
     * Returns lead selection index or -1 if no lead has even been set.
     *
     * @return                              lead selection index or -1
     */
    public int getLeadSelectionIndex() {
        return m_leadSelectionIndex;
    }
    /**
     * Returns the maximum index of the selection or -1 if nothing is selected.
     *
     * @return                              maximum index of the selection or -1
     */
    public int getMaxSelectionIndex() {
        return m_treeSelectionModel.getMaxSelectionRow();
    }
    /**
     * Returns the minimum index of the selection or -1 if nothing is selected.
     *
     * @return                              minimum index of the selection or -1
     */
    public int getMinSelectionIndex() {
        return m_treeSelectionModel.getMaxSelectionRow();
    }
    /**
     * Returns the selection mode of this object.
     *
     * @return                              selection mode of this object
     */
    public int getSelectionMode() {
        switch (m_treeSelectionModel.getSelectionMode()) {
        case TreeSelectionModel.CONTIGUOUS_TREE_SELECTION:
            return SINGLE_INTERVAL_SELECTION;
        case TreeSelectionModel.DISCONTIGUOUS_TREE_SELECTION:
            return MULTIPLE_INTERVAL_SELECTION;
        case TreeSelectionModel.SINGLE_TREE_SELECTION:
            return SINGLE_SELECTION;
        }
        throw new IllegalStateException("Illegal selection mode value.");
    }
    /**
     * Returns <code>true</code> if the selection of this model is being adjusted. In that way the user can know that notifications
     * about changes in the selection are coming in the series.
     *
     * @return                              <code>true</code> if selection of this model is adjusting
     */
    public boolean getValueIsAdjusting() {
        return m_valueIsAdjusting;
    }
    /**
     * Returns the paths displayed in the tree table for given interval of indices. Indices do not have to be of the (low, heigh) form.
     *
     * @param index0                        first index of the interval
     * @param index1                        second index of the interval
     * @return                              array of paths in the specified interval
     */
    protected TreePath[] getPathsForInterval(int index0,int index1) {
        int minIndex=Math.min(index0,index1);
        int maxIndex=Math.max(index0,index1);
        TreePath[] paths=new TreePath[maxIndex-minIndex+1];
        for (int row=minIndex;row<=maxIndex;row++)
            paths[row-minIndex]=m_treeMapper.getPathForRow(row);
        return paths;
    }
    /**
     * Checks whether row with given index is selected.
     *
     * @param index                         index of the row that needs to be checked
     * @return                              <code>true</code> if row with given index has been selected
     */
    public boolean isSelectedIndex(int index) {
        return m_treeSelectionModel.isRowSelected(index);
    }
    /**
     * Checks if selection of the model is empty.
     *
     * @return                              <code>true</code> if selection of the model is empty
     */
    public boolean isSelectionEmpty() {
        return m_treeSelectionModel.isSelectionEmpty();
    }
    /**
     * Called to notify the selection model that inteval of indices has been inserted. Implemented as no-op.
     *
     * @param index                         index at which interval has been inserted
     * @param length                        length of the inserted interval
     * @param before                        <code>true</code> if interval has been inserted before <code>index</code>
     */
    public void insertIndexInterval(int index,int length,boolean before) {
    }
    /**
     * Called to notify the selection model that inteval of indices has been removed. Implemented as no-op.
     *
     * @param index0                        first removed index
     * @param index1                        second removed index
     */
    public void removeIndexInterval(int index0,int index1) {
    }
    /**
     * Sets the anchor selection index.
     *
     * @param index                         new ancor selection index
     */
    public void setAnchorSelectionIndex(int index) {
        m_anchorSelectionIndex=index;
    }
    /**
     * Sets the lead selection index. All indices between anchor and lead are either selected on unselected (depending on the selection
     * state of anchor index).
     *
     * @param index                         index of the lead selection
     */
    public void setLeadSelectionIndex(int index) {
        if (m_anchorSelectionIndex==-1 || index==-1)
            return;
        int oldAnchorIndex=m_anchorSelectionIndex;
        if (m_leadSelectionIndex==-1)
            m_leadSelectionIndex=index;
        if (getSelectionMode()==SINGLE_SELECTION)
            m_anchorSelectionIndex=index;
        if (isSelectedIndex(oldAnchorIndex)) {
            removeSelectionIntervalInternal(oldAnchorIndex,m_leadSelectionIndex);
            addSelectionIntervalInternal(m_anchorSelectionIndex,index);
        }
        else {
            addSelectionIntervalInternal(oldAnchorIndex,m_leadSelectionIndex);
            removeSelectionIntervalInternal(m_anchorSelectionIndex,index);
        }
        int oldLeadSelectionIndex=m_leadSelectionIndex;
        m_leadSelectionIndex=index;
        fireValueChanged(min(oldAnchorIndex,m_anchorSelectionIndex,oldLeadSelectionIndex,m_leadSelectionIndex),max(oldAnchorIndex,m_anchorSelectionIndex,oldLeadSelectionIndex,m_leadSelectionIndex),getValueIsAdjusting());
    }
    /**
     * Utility method to compute the minimum of four numbers.
     */
    protected int min(int m1,int m2,int m3,int m4) {
        return Math.min(m1,Math.min(m2,Math.min(m3,m4)));
    }
    /**
     * Utility method to compute the maximum of four numbers.
     */
    protected int max(int m1,int m2,int m3,int m4) {
        return Math.max(m1,Math.max(m2,Math.max(m3,m4)));
    }
    /**
     * Clears the selection.
     */
    public void clearSelection() {
        m_treeSelectionModel.clearSelection();
        updateChangedIndices(MIN,MAX);
        fireValueChanged(MIN,MAX,getValueIsAdjusting());
    }
    /**
     * Adds an interval to the selection.
     *
     * @param index0                        first added index
     * @param index1                        second added index
     */
    public void addSelectionInterval(int index0,int index1) {
        if (index0==-1 || index1==-1)
            return;
        updateChangedIndices(index0,index1);
        addSelectionIntervalInternal(index0,index1);
        fireValueChanged(index0,index1,getValueIsAdjusting());
    }
    /**
     * Adds an interval to the selection without firing off any notifications.
     *
     * @param index0                        first added index
     * @param index1                        second added index
     */
    protected void addSelectionIntervalInternal(int index0,int index1) {
        // optimization - do not create an array of paths if there is only one row
        if (index0==index1)
            m_treeSelectionModel.addSelectionPath(m_treeMapper.getPathForRow(index0));
        else
            m_treeSelectionModel.addSelectionPaths(getPathsForInterval(index0,index1));
    }
    /**
     * Removes an interval from the selection.
     *
     * @param index0                        first removed index
     * @param index1                        second removed index
     */
    public void removeSelectionInterval(int index0,int index1) {
        if (index0==-1 || index1==-1)
            return;
        updateChangedIndices(index0,index1);
        removeSelectionIntervalInternal(index0,index1);
        fireValueChanged(index0,index1,getValueIsAdjusting());
    }
    /**
     * Removes an interval from the selection without firing off any notifications.
     *
     * @param index0                        first removed index
     * @param index1                        second removed index
     */
    protected void removeSelectionIntervalInternal(int index0,int index1) {
        // optimization - do not create an array of paths if there is only one row
        if (index0==index1)
            m_treeSelectionModel.removeSelectionPath(m_treeMapper.getPathForRow(index0));
        else
            m_treeSelectionModel.removeSelectionPaths(getPathsForInterval(index0,index1));
    }
    /**
     * Sets the selection interval to given range of indices.
     *
     * @param index0                        first index of the selection range
     * @param index1                        second index of the selection range
     */
    public void setSelectionInterval(int index0,int index1) {
        if (index0==-1 || index1==-1)
            return;
        updateChangedIndices(m_treeSelectionModel.getMinSelectionRow(),m_treeSelectionModel.getMaxSelectionRow());
        updateChangedIndices(index0,index1);
        int minChanged=Math.min(m_treeSelectionModel.getMinSelectionRow(),Math.min(index0,index1));
        int maxChanged=Math.max(m_treeSelectionModel.getMaxSelectionRow(),Math.max(index0,index1));
        setSelectionIntervalInternal(index0,index1);
        fireValueChanged(minChanged,maxChanged,getValueIsAdjusting());
    }
    /**
     * Sets the selection interval to given range of indices without firing any notifications.
     *
     * @param index0                        first index of the selection range
     * @param index1                        second index of the selection range
     */
    protected void setSelectionIntervalInternal(int index0,int index1) {
        // optimization - do not create an array of paths if there is only one row
        if (index0==index1)
            m_treeSelectionModel.setSelectionPath(m_treeMapper.getPathForRow(index0));
        else
            m_treeSelectionModel.setSelectionPaths(getPathsForInterval(index0,index1));
    }
    /**
     * Sets the selection mode. Can be one of: <code>SINGLE_INTERVAL_SELECTION</code>, <code>MULTIPLE_INTERVAL_SELECTION</code>
     * or <code>SINGLE_SELECTION</code>.
     *
     * @param selectionMode                 new selection mode
     */
    public void setSelectionMode(int selectionMode) {
        switch (selectionMode) {
        case SINGLE_INTERVAL_SELECTION:
            m_treeSelectionModel.setSelectionMode(TreeSelectionModel.CONTIGUOUS_TREE_SELECTION);
            break;
        case MULTIPLE_INTERVAL_SELECTION:
            m_treeSelectionModel.setSelectionMode(TreeSelectionModel.DISCONTIGUOUS_TREE_SELECTION);
            break;
        case SINGLE_SELECTION:
            m_treeSelectionModel.setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);
            break;
        default:
            throw new IllegalArgumentException("Invalid mode parameter.");
        }
    }
    /**
     * Sets whether the selection of this model is being adjusted. If value is adjusting, the user may know that notifications
     * about changes in the selection are coming in the series. When this property is changed, an event is generated to inform
     * listeners that selection is not chaning any more.
     *
     * @param valueIsAdjusting              <code>true</code> if selection of this model is adjusting
     */
    public void setValueIsAdjusting(boolean valueIsAdjusting) {
        if (m_valueIsAdjusting!=valueIsAdjusting) {
            m_valueIsAdjusting=valueIsAdjusting;
            int oldMinChangedIndex=m_minChangedIndex;
            m_minChangedIndex=MAX;
            m_maxChangedIndex=MIN;
            if (oldMinChangedIndex!=MAX)
                fireValueChanged(m_minChangedIndex,m_maxChangedIndex,m_valueIsAdjusting);
        }
    }
    /**
     * Adds a list selection listener.
     *
     * @param listener                      listener to be added
     */
    public void addListSelectionListener(ListSelectionListener listener) {
        m_listenerList.add(ListSelectionListener.class,listener);
    }
    /**
     * Removes a list selection listener.
     *
     * @param listener                      listener to be removed
     */
    public void removeListSelectionListener(ListSelectionListener listener) {
        m_listenerList.remove(ListSelectionListener.class,listener);
    }
    /**
     * Fires a notification that selection has been changed.
     *
     * @param index0                        first index of the changed selection
     * @param index1                        second index of the changed selection
     * @param isAdjusting                   specifies whether this notification is just one in the series of notifications
     */
    protected void fireValueChanged(int index0,int index1,boolean isAdjusting) {
        Object[] listeners=m_listenerList.getListenerList();
        ListSelectionEvent event=null;
        for (int i=listeners.length-2;i>=0;i-=2)
            if (listeners[i]==ListSelectionListener.class) {
                if (event==null)
                    event=new ListSelectionEvent(this,index0,index1,isAdjusting);
                ((ListSelectionListener)listeners[i+1]).valueChanged(event);
            }
    }
    /**
     * Creates a property change listener that monitors properties on the associated tree table.
     *
     * @return                              property change listener
     */
    protected PropertyChangeListener createPropertyChangeListener() {
        return new PropertyChangeHandler();
    }
    /**
     * Creates a tree expansion listener that monitors expansion events on the associated tree table.
     *
     * @return                              tree expansion listener
     */
    protected TreeExpansionListener createTreeExpansionListener() {
        return new TreeExpansionHandler();
    }

    /**
     * Property change listener that monitors property changes on associated tree table.
     */
    protected class PropertyChangeHandler implements PropertyChangeListener {
        /**
         * Called whenever a property of the tree table is changed.
         *
         * @param event                     contains information about property change
         */
        public void propertyChange(PropertyChangeEvent event) {
            String changeName=event.getPropertyName();
            if (JTreeTable.TREE_SELECTION_MODEL_PROPERTY.equals(changeName))
                setTreeSelectionModel((TreeSelectionModel)event.getNewValue());
        }
    }

    /**
     * Tree expansion listener that fires notifications  that tree table selection model has been changed whenever tree is
     * expanded or collapsed.
     */
    protected class TreeExpansionHandler implements TreeExpansionListener {
        /**
         * Invoked when tree is expanded. This method fires a notification that list selection model has been changed.
         *
         * @param event                     contains information about expansion event
         */
        public void treeCollapsed(TreeExpansionEvent event) {
            fireValueChanged(MIN,MAX,getValueIsAdjusting());
        }
        /**
         * Invoked when tree is collapsed. This method fires a notification that list selection model has been changed.
         *
         * @param event                     contains information about collapsing event
         */
        public void treeExpanded(TreeExpansionEvent event) {
            fireValueChanged(MIN,MAX,getValueIsAdjusting());
        }
    }
}
