package de.fzi.wim.guibase.treetable;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Enumeration;
import java.awt.Rectangle;
import java.awt.Point;
import javax.swing.table.AbstractTableModel;
import javax.swing.event.TreeModelListener;
import javax.swing.event.TreeModelEvent;
import javax.swing.tree.AbstractLayoutCache;
import javax.swing.tree.FixedHeightLayoutCache;
import javax.swing.tree.VariableHeightLayoutCache;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

/**
 * Provides mapping from tree to a table and maintains tree state. A single instance of this class is associated with each
 * {@link JTreeTable} and is used to:
 * <ul>
 *  <li>wrap tree model into a table model,
 *  <li>provide geometry information about the tree,
 *  <li>maintain expanded paths of the tree.
 * <li>
 * To provide information about geometry of individual paths, this class uses an instance implementing {@link TreePathMeasurer}
 * interface. All information about tree layout is kept in an instalce of <code>javax.swing.tree.AbstractLayoutCache</code> class.
 * Appropriate concrete subclass is created depending if tree table has variable or fixed row height.
 *
 * @see JTreeTable
 * @see TreePathMeasurer
 */
class TreeMapper extends AbstractTableModel {
    /** Instance of tree table that this objects is used for. */
    protected JTreeTable m_treeTable;
    /** Used for manaing layout information. */
    protected AbstractLayoutCache m_layoutCache;
    /** Utility rectangle that is reused over and over. */
    protected Rectangle m_utilRect=new Rectangle();
    /** Listener attached to currently selected {@link TreeTableModel}. */
    protected TreeModelListener m_treeModelListener;
    /** Tree table model that this object wraps. */
    protected TreeTableModel m_treeTableModel;

    /**
     * Creates an instance of this class and attaches it to appropriate tree table.
     *
     * @param treeTable                 tree table that this object will be used in
     */
    public TreeMapper(JTreeTable treeTable) {
        m_treeModelListener=createTreeModelListener();
        m_treeTable=treeTable;
        m_treeTable.addPropertyChangeListener(createPropertyChangeListener());
        synchronizeTreeMapperWithTreeTable();
    }
    /**
     * Synchronizes all properties of this object with attached tree table.
     */
    protected void synchronizeTreeMapperWithTreeTable() {
        setModel(m_treeTable.getTreeTableModel());
        setSelectionModel(m_treeTable.getTreeSelectionModel());
        setTreePathMeasurer(m_treeTable.getTreePathMeasurer());
        updateLayoutCache();
    }
    /**
     * Sets tree table model.
     *
     * @model                           tree table model that this object should wrap
     */
    public void setModel(TreeTableModel model) {
        if (m_treeTableModel!=null)
            m_treeTableModel.removeTreeModelListener(m_treeModelListener);
        m_treeTableModel=model;
        if (m_treeTableModel!=null)
            m_treeTableModel.addTreeModelListener(m_treeModelListener);
        updateLayoutCache();
        fireTableStructureChanged();
    }
    /**
     * Called when the UI of associated tree table is changed.
     */
    public void updateUI() {
           m_layoutCache.invalidateSizes();
    }
    /**
     * Sets tree selection model.
     *
     * @param selectionModel            tree selection model that tree table is using
     */
    public void setSelectionModel(TreeSelectionModel selectionModel) {
        if (m_layoutCache!=null) {
            m_layoutCache.setSelectionModel(selectionModel);
            m_layoutCache.invalidateSizes();
        }
    }
    /**
     * Sets tree path measurer.
     *
     * @param treePathMeasurer          tree path measurer that will prodive information about geometry of a tree path
     */
    public void setTreePathMeasurer(TreePathMeasurer treePathMeasurer) {
        if (m_layoutCache!=null) {
            m_layoutCache.setNodeDimensions(new NodeDimensionsAdapter(m_treeTable,treePathMeasurer));
            m_layoutCache.invalidateSizes();
        }
    }
    /**
     * Updates the state of the layout cache.
     */
    protected void updateLayoutCache() {
        if (m_treeTable.isLargeModel() && m_treeTable.getRowHeight()>0)
            m_layoutCache=new FixedHeightLayoutCache();
        else
            m_layoutCache=new VariableHeightLayoutCache();
        m_layoutCache.setModel(m_treeTable.getTreeTableModel());
        m_layoutCache.setRootVisible(m_treeTable.isRootVisible());
        setRowHeight(m_treeTable.getRowHeight());
        setSelectionModel(m_treeTable.getTreeSelectionModel());
        setTreePathMeasurer(m_treeTable.getTreePathMeasurer());
        m_layoutCache.invalidateSizes();
    }
    /**
     * Returns current number of rows in the tree table.
     *
     * @return                          current number of rows in the table
     */
    public int getRowCount() {
        return m_layoutCache.getRowCount();
    }
    /**
     * Returns the index of the row containing supplied path. Returns -1 if path is not visible.
     *
     * @param path                      path whose index is requested
     * @return                          index of given path or -1 if path is not visible
     */
    public int getRowForPath(TreePath path) {
        return m_layoutCache.getRowForPath(path);
    }
    /**
     * Returns the path displayed in given row. If goven row is not visible, returns <code>null</code>.
     *
     * @param row                       row whose path is requested
     * @return                          path displayed in given row or <code>null</code> if row is not visible
     */
    public TreePath getPathForRow(int row) {
        return m_layoutCache.getPathForRow(row);
    }
    /**
     * Returns the height of supplied path.
     *
     * @param path                      path whose height is required
     * @return                          height of given path
     */
    public int getPathHeight(TreePath path) {
        return m_layoutCache.getBounds(path,m_utilRect).height;
    }
    /**
     * Returns the bounds of the given path.
     *
     * @param path                      path whose bounds are required
     * @param placeIn                   place where to put bounds (may be <code>null</code>)
     * @return                          bounds of the path
     */
    public Rectangle getPathBounds(TreePath path,Rectangle placeIn) {
        return m_layoutCache.getBounds(path,placeIn);
    }
    /**
     * Returns the indent of the last object of the path in the tree
     *
     * @param path                      path for which the indent of the last object is requested
     * @return                          indent of the last object in the path
     */
    public int getTreeNodeIndent(TreePath path) {
        return m_layoutCache.getBounds(path,m_utilRect).x;
    }
    /**
     * Sets the height of rows in the tree table. If >0, this height is used for all rows. If &lt;=0, {@link TreePathMeasurer}
     * will be used to determine the height of each path.
     *
     * @param rowHeight                 height of row or &lt;=0 to use <code>TreePathMeasurer</code>
     * @see TreePathMeasurer
     */
    public void setRowHeight(int rowHeight) {
        m_layoutCache.setRowHeight(rowHeight);
    }
    /**
     * Returns the path closest to supplied location. Returns <code>null</code> only if nothing is visible.
     *
     * @param point                     location for which a path is requested
     * @return                          tree path closest to given location
     */
    public TreePath getPathClosestTo(Point point) {
        return m_layoutCache.getPathClosestTo(point.x,point.y);
    }
    /**
     * Returns the path at given point. Returns <code>null</code> if no path is at given point.
     *
     * @param point                     location for which a path is requested
     * @return                          tree path at given point (or <code>null</code> if there is no such path)
     */
    public TreePath pathAtPoint(Point point) {
        TreePath path=m_layoutCache.getPathClosestTo(point.x,point.y);
        if (path!=null) {
            m_layoutCache.getBounds(path,m_utilRect);
            if (m_utilRect.y<=point.y && point.y<m_utilRect.y+m_utilRect.height)
                return path;
        }
        return null;
    }
    /**
     * Determines whether root of the tree is visible.
     *
     * @param rootVisible               <code>true</code> if root node should be visible in the tree table
     */
    public void setRootVisible(boolean rootVisible) {
        m_layoutCache.setRootVisible(rootVisible);
    }
    /**
     * Creates property change listener that will monitor changes of properties in the attached tree table.
     *
     * @return                          property change listener that will monitor the table
     */
    protected PropertyChangeListener createPropertyChangeListener() {
        return new PropertyChangeHandler();
    }
    /**
     * Creates tree model listener that will monitor events on wrapped tree table model.
     *
     * @return                          tree model listener that will monitor tree table model
     */
    protected TreeModelListener createTreeModelListener() {
        return new TreeModelHandler();
    }
    /**
     * Returns an <code>Enumeration</code> of <code>TreePath</code> objects that are expanded under supplied parent.
     *
     * @param parent                    parent under which all expanded paths are selected
     * @return                          <code>Enumeration</code> of <code>TreePath</code> objects that are expanded under given root
     */
    public Enumeration getExpandedDescendants(TreePath parent) {
        return m_layoutCache.getVisiblePathsFrom(parent);
    }
    /**
     * Checks whether given path is expanded.
     *
     * @param path                      path whose state is checked
     * @return                          <code>true</code> is path is expanded
     */
    public boolean isExpanded(TreePath path) {
        return m_layoutCache.isExpanded(path);
    }
    /**
     * Checks whether given path is collapsed.
     *
     * @param path                      path whose state is checked
     * @return                          <code>true</code> is path is collapsed
     */
    public boolean isCollapsed(TreePath path) {
        return !isExpanded(path);
    }
    /**
     * Expands supplied path. Also scrolls expanded path into view, and if {@link JTreeTable.getScrollsOnExpand} is
     * <code>true</code>, it will scroll expanded children into view as well.
     *
     * @param path                      path to be expanded
     */
    public void expandPath(TreePath path) {
        m_layoutCache.setExpandedState(path,true);
        fireTableDataChanged();
        int row=getRowForPath(path);
        if (m_treeTable.getScrollsOnExpand())
            ensureRowsAreVisible(row,row+m_layoutCache.getVisibleChildCount(path));
        else
            ensureRowsAreVisible(row,row);
    }
    /**
     * Makes sure that part of the table that displays supplied rows is visible in the scroll pane.
     *
     * @param beginRow                  start row that must be made visible
     * @param endRow                    end row that must be made visible
     */
    protected void ensureRowsAreVisible(int beginRow,int endRow) {
        if (beginRow==endRow) {
            Rectangle scrollBounds=m_treeTable.getRowBounds(beginRow);
            m_treeTable.scrollRectToVisible(scrollBounds);
        }
        else {
            Rectangle newVisibleRect=m_treeTable.getRowBounds(beginRow);
            Rectangle visibleRect=m_treeTable.getVisibleRect();
            newVisibleRect.x=visibleRect.x;
            newVisibleRect.width=visibleRect.width;
            for (int counter=beginRow+1;counter<=endRow;counter++) {
                Rectangle rowRect=m_treeTable.getRowBounds(counter);
                newVisibleRect.height+=rowRect.y+rowRect.height-(newVisibleRect.y+newVisibleRect.height);
                if (newVisibleRect.height>visibleRect.height)
                    break;
            }
            m_treeTable.scrollRectToVisible(newVisibleRect);
        }
    }
    /**
     * Collapses supplied tree path.
     *
     * @param path                      path to be collapsed
     */
    public void collapsePath(TreePath path) {
        m_layoutCache.setExpandedState(path,false);
        fireTableDataChanged();
    }
    /**
     * Returns number of columns in wrapped tree table model. This method is part of the <code>TableModel</code> interface.
     *
     * @return                          number of columns
     */
    public int getColumnCount() {
        return m_treeTableModel.getColumnCount();
    }
    /**
     * Returns the name of given column in wrapped tree table model. This method is part of the <code>TableModel</code> interface.
     *
     * @param column                    column whose name is required
     * @return                          name of given column
     */
    public String getColumnName(int column) {
        return m_treeTableModel.getColumnName(column);
    }
    /**
     * Returns the class of given column in wrapped tree table model. This method is part of the <code>TableModel</code> interface.
     *
     * @param column                    column whose class is required
     * @return                          class of given column
     */
    public Class getColumnClass(int column) {
        return m_treeTableModel.getColumnClass(column);
    }
    /**
     * Returns object displayed in given row.
     *
     * @param row                       row for which node is requested
     * @return                          object displayed in supplied row
     */
    protected Object nodeForRow(int row) {
        return getPathForRow(row).getLastPathComponent();
    }
    /**
     * Returns the value at given coordinates in wrapped tree table model. This method is part of the <code>TableModel</code> interface.
     *
     * @param row                       row at which value is requested
     * @param column                    column at which value is requested
     * @return                          value at given row and column
     */
    public Object getValueAt(int row,int column) {
        return m_treeTableModel.getValueAt(nodeForRow(row),column);
    }
    /**
     * Returns if the cell at given coordinates is editable in wrapped tree table model. This method is part of the <code>TableModel</code> interface.
     *
     * @param row                       row at which cell is tested
     * @param column                    column at which cell is tested
     * @return                          <code>true</code> if cell at given row and column is editable
     */
    public boolean isCellEditable(int row,int column) {
        return m_treeTableModel.isCellEditable(nodeForRow(row),column);
    }
    /**
     * Sets the value at given coordinates in wrapped tree table model. This method is part of the <code>TableModel</code> interface.
     *
     * @param value                     value to be set
     * @param row                       row at which value is to be set
     * @param column                    column at which value is to be set
     */
    public void setValueAt(Object value,int row,int column) {
        m_treeTableModel.setValueAt(value,nodeForRow(row),column);
    }

    /**
     * Monitors changes of various properties in attached tree table and makes sure that enclosing {@link TreeMapper} instance
     * is kept in sync with the tree table.
     */
    protected class PropertyChangeHandler implements PropertyChangeListener {
        /**
         * Invoked when a property of the attached tree table is changed.
         *
         * @param event                 event supplying information about the change
         */
        public void propertyChange(PropertyChangeEvent event) {
            if (event.getSource()!=m_treeTable)
                return;
            String changeName=event.getPropertyName();
            if (changeName.equals(JTreeTable.TREE_TABLE_MODEL_PROPERTY))
                setModel((TreeTableModel)event.getNewValue());
            else if (changeName.equals(JTreeTable.ROOT_VISIBLE_PROPERTY))
                setRootVisible(((Boolean)event.getNewValue()).booleanValue());
            else if (changeName.equals("rowHeight"))
                setRowHeight(((Integer)event.getNewValue()).intValue());
            else if (changeName.equals(JTreeTable.LARGE_MODEL_PROPERTY))
                updateLayoutCache();
            else if (changeName.equals(JTreeTable.TREE_SELECTION_MODEL_PROPERTY))
                setSelectionModel((TreeSelectionModel)event.getNewValue());
            else if (changeName.equals(JTreeTable.TREE_PATH_MEASURER_PROPERTY))
                setTreePathMeasurer((TreePathMeasurer)event.getNewValue());
        }
    }

    /**
     * An instance of this class is used by <code>javax.swing.tree.AbstractTreeLayout</code> to measure individual nodes.
     * This class forwards the call to {@link TreePathMeasurer} to measure a tree node.
     *
     * @see TreePathMeasurer
     */
    protected static class NodeDimensionsAdapter extends AbstractLayoutCache.NodeDimensions {
        /** Measurer that will be used to measure tree paths. */
        protected TreePathMeasurer m_treePathMeasurer;
        /** Tree table for which nodes are measured. */
        protected JTreeTable m_treeTable;
        /** Utility class that will receive dimensions of the node from <code>TreePathMeasurer</code>. */
        protected TreePathMeasurer.TreePathDimensions m_utilDimensions=new TreePathMeasurer.TreePathDimensions();

        /**
         * Creates and initializes an instance of this class.
         *
         * @param treeTable                 tree table for which nodes are measured
         * @param treePathMeasurer          tree path measurer that will provide geometry information
         */
        public NodeDimensionsAdapter(JTreeTable treeTable,TreePathMeasurer treePathMeasurer) {
            m_treeTable=treeTable;
            m_treePathMeasurer=treePathMeasurer;
        }
        /**
         * Called by <code>AbstractLayoutCache</code> to compute dimensions of given node.
         *
         * @param value                     value of the node
         * @param row                       row of the node
         * @param depth                     zero-based depth of the node in the tree
         * @param expanded                  set to <code>true</code> if node is expanded
         * @param size                      rectangle to be filled in with node size
         * @return                          rectandgle with node size
         */
        public Rectangle getNodeDimensions(Object value,int row,int depth,boolean expanded,Rectangle size) {
            m_treePathMeasurer.measureTreePath(m_treeTable,value,row,depth,m_utilDimensions);
            if (size==null)
                size=new Rectangle();
            size.setBounds(m_utilDimensions.m_indentation,0,5,m_utilDimensions.m_height);
            return size;
        }
    }

    /**
     * Monitors events on the wrapped tree model and converts them to appropriate table model events.
     */
    protected class TreeModelHandler implements TreeModelListener {
        /**
         * Called when tree nodes are changed.
         *
         * @param e                         event supplying change information
         */
        public void treeNodesChanged(TreeModelEvent e) {
            m_layoutCache.treeNodesChanged(e);
            fireTableDataChanged();
        }
        /**
         * Called when tree nodes are inserted.
         *
         * @param e                         event supplying change information
         */
        public void treeNodesInserted(TreeModelEvent e) {
            m_layoutCache.treeNodesInserted(e);
            fireTableDataChanged();
        }
        /**
         * Called when tree nodes are removed.
         *
         * @param e                         event supplying change information
         */
        public void treeNodesRemoved(TreeModelEvent e) {
            m_layoutCache.treeNodesRemoved(e);
            unselectNonExistantNodes();
            fireTableDataChanged();
        }
        /**
         * Called when tree structure under some node is changed.
         *
         * @param e                         event supplying change information
         */
        public void treeStructureChanged(TreeModelEvent e) {
            int[] childIndices=e.getChildIndices();
            if (childIndices!=null && childIndices.length>0 && childIndices[0]==-2)
                collapsePath(e.getTreePath());
            m_layoutCache.treeStructureChanged(e);
            unselectNonExistantNodes();
            fireTableDataChanged();
        }
        /**
         * Unselects nodes that do not exist in the tree model any more.
         */
        protected void unselectNonExistantNodes() {
            TreeSelectionModel selectionModel=m_treeTable.getTreeSelectionModel();
            TreePath[] paths=selectionModel.getSelectionPaths();
            if (paths==null)
                return;
            for (int i=0;i<paths.length;i++) {
                Object[] path=paths[i].getPath();
                if (!checkPathInTree(path,0,m_treeTableModel.getRoot()))
                    selectionModel.removeSelectionPath(paths[i]);
            }
        }
        /**
         * Determines whether supplied path is in tree.
         *
         * @param path                      path to be checked
         * @param objectIndex               index of the object in path
         * @param root                      root of the subtree
         * @return                          <code>true</code> if path is in tree
         */
        protected boolean checkPathInTree(Object[] path,int objectIndex,Object root) {
            if (path[objectIndex]!=root)
                return false;
            int nextIndex=objectIndex+1;
            if (nextIndex==path.length)
                return true;
            if (m_treeTableModel.getIndexOfChild(root,path[nextIndex])==-1)
                return false;
            return checkPathInTree(path,nextIndex,path[nextIndex]);
        }
    }
}

