package de.fzi.wim.guibase.treetable;

import java.util.EventObject;
import java.util.Enumeration;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Graphics;
import java.awt.event.MouseEvent;
import javax.swing.tree.TreeSelectionModel;
import javax.swing.tree.DefaultTreeSelectionModel;
import javax.swing.tree.TreePath;
import javax.swing.event.TreeExpansionListener;
import javax.swing.event.TreeExpansionEvent;
import javax.swing.event.TreeWillExpandListener;
import javax.swing.event.TableModelEvent;
import javax.swing.tree.ExpandVetoException;
import javax.swing.JComboBox;
import javax.swing.ListSelectionModel;
import javax.swing.SwingConstants;
import javax.swing.DefaultListSelectionModel;
import javax.swing.table.TableModel;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableColumnModel;
import javax.swing.table.DefaultTableModel;
import javax.swing.plaf.UIResource;

import de.fzi.wim.guibase.tables.SmartTable;

/**
 * Implements a tree table control. The interface of this class is similar to the interface of <code>javax.swing.JTable</code>
 * class, with some methods that shouldn't be called and some methods that have semantices that might not be expected:
 * <ul>
 *  <li><code>setModel()</code> - cannot be called; use {@link #setTreeTableModel}
 *  <li><code>getModel()</code> - will return a table model that wraps supplied tree table model; to get tree table model use {@link #getTreeTableModel}
 *  <li><code>setSelectionModel()</code> - cannot be called; use {@link #setTreeSelectionModel}
 *  <li><code>getSelectionModel()</code> - will return a wrapper model that converts a tree selection model into a list selection model; to get tree selection model use {@link #getTreeSelectionModel()}
 *  <li><code>setRowHeight(int,int)</code> - cannot be called; to change row height install a {@link TreePathMeasurer}
 * </ul>
 * This class also offers a right interface to access the tree part of the table.
 * <p>Model used for this control must be an instance imlpementing {@link TreeTableModel}. Selecction in the control is maintained
 * using an instance of <code>javax.swing.tree.TreeSelecionModel</code>. Hence, this control views the rows of the table as a tree,
 * and columns as usual for any table.
 * <p><code>treeColumn</code> property must be set to the index of the column model whose column will be used to display the tree.
 * All other columns are displayed normally.
 * <p>An instance implementing {@link TreePathMeasurer} interface can be installed into this control that will determine the size and
 * indentation level of individual paths.
 * <p>An instance implementing {@link TreeNodeRenderer} interface can be installed into this control that will be used to paint the
 * tree node. This object will be used for drawing of node icons only. Usual table renderer will be used to paint actual text. It is
 * possible to set node renderer to <code>null</code> - then no icon will be drawn.
 *
 * @see TreeTableModel
 * @see TreeNodeRenderer
 * @see TreePathMeasurer
 */
public class JTreeTable extends SmartTable {
    /** A dummy instance of table model that is set into the table in constructor. */
    protected static final TableModel DUMMY_TABLE_MODEL=new DefaultTableModel();
    /** A dummy instance of list selection model that is set into the table in constructor. */
    protected static final ListSelectionModel DUMMY_SELECTION_MODEL=new DefaultListSelectionModel();

    /** Identifier for <code>treeTableModel</code> property. */
    public final static String TREE_TABLE_MODEL_PROPERTY="treeTableModel";
    /** Identifier for <code>rootVisible</code> property. */
    public final static String ROOT_VISIBLE_PROPERTY="rootVisible";
    /** Identifier for <code>largeModel</code> property. */
    public final static String LARGE_MODEL_PROPERTY="largeModel";
    /** Identifier for <code>treeSelectionModel</code> property. */
    public final static String TREE_SELECTION_MODEL_PROPERTY="treeSelectionModel";
    /** Identifier for <code>treeColumn</code> property. */
    public final static String TREE_COLUMN_PROPERTY="treeColumn";
    /** Identifier for <code>treeNodeRenderer</code> property. */
    public final static String TREE_NODE_RENDERER_PROPERTY="treeNodeRenderer";
    /** Identifier for <code>treePathMeasurer</code> property. */
    public final static String TREE_PATH_MEASURER_PROPERTY="treePathMeasurer";
    /** Identifier for <code>showsRootHandles</code> property. */
    public final static String SHOWS_ROOT_HANDLES_PROPERTY="showsRootHandles";
    /** Identifier for <code>scrollsOnExpand</code> property. */
    public final static String SCROLLS_ON_EXPAND_PROPERTY="scrollsOnExpand";

    /** Tree mapper used to transform the tree into a table. */
    protected TreeMapper m_treeMapper;
    /** Tree table model used by this control. */
    protected TreeTableModel m_treeTableModel;
    /** Adapter that converts a tree selection model into list selection model. */
    protected TreeSelectionModelAdapter m_selectionModelAdapter;
    /** Adapter that converts a tree selection model into list selection model. */
    protected boolean m_largeModel;
    /** Specifies whether the root node of the tree is visible. */
    protected boolean m_rootVisible;
    /** Specifies whether the tree will show handles at the root. */
    protected boolean m_showsRootHandles;
    /** Specifies whether children of expanded nodes will be scrolled into view upon expansion. */
    protected boolean m_scrollsOnExpand;
    /** Tree selection model used by this control. */
    protected TreeSelectionModel m_treeSelectionModel;
    /** Index of the column model that is used to paint the tree. */
    protected int m_treeColumn;
    /** Tree node renderer used to paint the tree node. */
    protected TreeNodeRenderer m_treeNodeRenderer;
    /** Table cell renderer that adapts all renreres to <code>JTable</code> interface. */
    protected TreeTableCellRenderer m_treeTableCellRenderer;
    /** Used for measuring a path in the tree. */
    protected TreePathMeasurer m_treePathMeasurer;

    /**
     * Creates and initializes an instance of this class and attaches it to the tree table model.
     *
     * @param treeTableModel                    model that tree table will use
     */
    public JTreeTable(TreeTableModel treeTableModel) {
        this(treeTableModel,null,null);
    }
    /**
     * Creates and initializes an instance of this class and attaches it to the tree table model and table column model.
     *
     * @param treeTableModel                    model that tree table will use
     * @param columnModel                       column model that table will use
     */
    public JTreeTable(TreeTableModel treeTableModel,TableColumnModel columnModel) {
        this(treeTableModel,columnModel,null);
    }
    /**
     * Creates and initializes an instance of this class and attaches it to the tree table model and table column model.
     *
     * @param treeTableModel                    model that tree table will use
     * @param columnModel                       column model that table will use
     * @param treeSelectionModel                tree selection model that tree table will use
     */
    public JTreeTable(TreeTableModel treeTableModel,TableColumnModel columnModel,TreeSelectionModel treeSelectionModel) {
        super(DUMMY_TABLE_MODEL,columnModel,DUMMY_SELECTION_MODEL);
        m_scrollsOnExpand=true;
        m_largeModel=false;
        m_rootVisible=false;
        m_treeColumn=0;
        if (columnModel==null)
            autoCreateColumnsFromModel=true;
        this.rowHeight=20;
        m_treeTableModel=treeTableModel;
        if (treeSelectionModel==null)
            treeSelectionModel=createDefaultTreeSelectionModel();
        m_treeSelectionModel=treeSelectionModel;
        m_treePathMeasurer=createDefaultTreePathMeasurer();
        m_treeNodeRenderer=createDefaultTreeNodeRenderer();
        m_treeTableCellRenderer=createTreeTableCellRenderer();
        m_treeMapper=createTreeMapper();
        m_selectionModelAdapter=new TreeSelectionModelAdapter(this,m_treeMapper);
        super.setModel(m_treeMapper);
        super.setSelectionModel(m_selectionModelAdapter);
    }
    /**
     * Called when look-and-feel is updated. This method will update tree node renderer and tree table cell renderer to match
     * to the new UI.
     */
    public void updateUI() {
        super.updateUI();
        if (m_treeNodeRenderer==null || m_treeNodeRenderer instanceof UIResource)
            m_treeNodeRenderer=createDefaultTreeNodeRenderer();
        m_treeTableCellRenderer=createTreeTableCellRenderer();
        if (m_treeMapper!=null)
            m_treeMapper.updateUI();
    }
    /**
     * Creates default tree selectuion model. Can be overriden in subclass to create different default tree selection model.
     *
     * @return                                  default tree selection model
     */
    protected TreeSelectionModel createDefaultTreeSelectionModel() {
        return new DefaultTreeSelectionModel();
    }
    /**
     * Creates default tree selection model. Can be overriden in subclass to create different default tree selection model.
     *
     * @return                                  default tree selection model
     */
    protected TreePathMeasurer createDefaultTreePathMeasurer() {
        return new DefaultTreePathMeasurer();
    }
    /**
     * Creates default tree node renderer. Can be overriden in subclass to create different default tree node renderer.
     *
     * @return                                  default tree node renderer
     */
    protected TreeNodeRenderer createDefaultTreeNodeRenderer() {
        return new DefaultTreeNodeRendererResource();
    }
    /**
     * Creates tree table cell renderer that tree table will use. Can be overriden in subclass to create different tree table cell renderer.
     *
     * @return                                  tree table cell renderer
     */
    protected TreeTableCellRenderer createTreeTableCellRenderer() {
        return new TreeTableCellRenderer();
    }
    /**
     * Creates tree mapper that tree table will use. Can be overriden in subclass to create different tree mapper.
     *
     * @return                                  tree mapper
     */
    protected TreeMapper createTreeMapper() {
        return new TreeMapper(this);
    }
    /**
     * Sets the tree table model that this tree table should use.
     *
     * @param treeTableModel                    tree table model that this table should use
     */
    public void setTreeTableModel(TreeTableModel treeTableModel) {
        TreeTableModel oldModel=m_treeTableModel;
        m_treeTableModel=treeTableModel;
        firePropertyChange(TREE_TABLE_MODEL_PROPERTY,oldModel,m_treeTableModel);
    }
    /**
     * Returns the tree table model that this table is using.
     *
     * @return                                  tree table model that this table is using
     */
    public TreeTableModel getTreeTableModel() {
        return m_treeTableModel;
    }
    /**
     * Notifies tree table that the model is large. Tree table will try to adjust its operation to take this into account.
     * Default for this property is <code>false</code>.
     *
     * @param largeModel                        <code>true</code> if the model of the table is large
     */
    public void setLargeModel(boolean largeModel) {
        m_largeModel=largeModel;
        boolean oldValue=m_largeModel;
        m_largeModel=largeModel;
        firePropertyChange(LARGE_MODEL_PROPERTY,oldValue,m_largeModel);
    }
    /**
     * Returns whether the model of this tree table is large. Default for this property is <code>false</code>.
     *
     * @return                                  <code>true</code> if the model of the table is large
     */
    public boolean isLargeModel() {
        return m_largeModel;
    }
    /**
     * Sets the tree selection model that this control should use.
     *
     * @param treeSelectionModel                tree selection model that this table should use
     */
    public void setTreeSelectionModel(TreeSelectionModel treeSelectionModel) {
        TreeSelectionModel oldValue=m_treeSelectionModel;
        m_treeSelectionModel=treeSelectionModel;
        firePropertyChange(TREE_SELECTION_MODEL_PROPERTY,oldValue,m_treeSelectionModel);
    }
    /**
     * Returns the tree selection model that this control is using.
     *
     * @return                                  tree selection model that this table is using
     */
    public TreeSelectionModel getTreeSelectionModel() {
        return m_treeSelectionModel;
    }
    /**
     * Overrides the <code>JTable</code> implementation to disallow setting of row height. This method should not be called.
     * To have rows of different heights install a {@link TreePathMeasurer}.
     *
     * @param row                                       the row
     * @param rowHeight                                 the row height
     * @see TreePathMeasurer
     * @exception UnsupportedOperationException         always thrown
     */
    public void setRowHeight(int row,int rowHeight) {
        throw new UnsupportedOperationException("Call JTreeTable.setRowHeight(int,int) is not allowed - use TreePathMeasurer to determine row size.");
    }
    /**
     * Sets the height of all rows in the table. If this value is >0, then all rows will have the same height, regardles if a tree
     * path measurer has been installed. If this value is &lt;=0, then installed {@link TreePathMeasurer} will determine the height
     * of each row.
     *
     * @param rowHeight                         height of all rows in the tree table
     * @see TreePathMeasurer
     */
    public void setRowHeight(int rowHeight) {
        int old=this.rowHeight;
        this.rowHeight=rowHeight;
        resizeAndRepaint();
        firePropertyChange("rowHeight",old,rowHeight);
    }
    /**
     * Returns whether all rows have the same (fixed) row height. Row height is fixed if <code>rowHeight</code> property has
     * the value >0.
     *
     * @return                                  <code>true</code> if all rows have the same row height
     */
    public boolean isFixedRowHeight() {
        return this.rowHeight>0;
    }
    /**
     * Returns whether root node is visible.
     *
     * @return                                  <code>true</code> if root node is visible
     */
    public boolean isRootVisible() {
        return m_rootVisible;
    }
    /**
     * Sets whether root node should be visible.
     *
     * @param rootVisible                       <code>true</code> if root node should be visible
     */
    public void setRootVisible(boolean rootVisible) {
        boolean oldValue=m_rootVisible;
        m_rootVisible=rootVisible;
        firePropertyChange(ROOT_VISIBLE_PROPERTY,oldValue,m_rootVisible);
    }
    /**
     * Fixes the table interface to return the height or supplied row. if <code>rowHeight</code> property is >0, then this
     * value will be returned. If <code>rowHeight</code> is &lt;=0, then installed {@link TreePathMeasurer} will be used.
     *
     * @param row                               index of the row for which height is requested
     * @return                                  height of supplied row
     * @see TreePathMeasurer
     */
    public int getRowHeight(int row) {
        return getPathHeight(getPathForRow(row));
    }
    /**
     * Returns the height or supplied tree path. if <code>rowHeight</code> property is >0, then this
     * value will be returned. If <code>rowHeight</code> is &lt;=0, then installed {@link TreePathMeasurer} will be used.
     *
     * @param path                              path for which height is requested
     * @return                                  height of supplied row
     * @see TreePathMeasurer
     */
    public int getPathHeight(TreePath path) {
        if (m_treeMapper==null)
            return getRowHeight();
        return m_treeMapper.getPathHeight(path);
    }
    /**
     * Returns the index of the row at given point.
     *
     * @param point                             point at which a tree path should be looked up
     * @return                                  index of the row at given point or -1 if there is no such row
     */
    public int rowAtPoint(Point point) {
        TreePath path=pathAtPoint(point);
        if (path==null)
            return -1;
        else
            return getRowForPath(path);
    }
    /**
     * Returns the tree path as given point. If there is no path at this point, <code>null</code> will be returned.
     *
     * @param point                             point at which a tree path should be looked up
     * @return                                  tree path at given point, or <code>null</code> if there is no such path
     */
    public TreePath pathAtPoint(Point point) {
        return m_treeMapper.pathAtPoint(point);
    }
    /**
     * Returns the path closest to given point. It returns <code>null</code> if no three path is shown in the tree table.
     *
     * @param point                             point at which closest tree path should be looked up
     * @return                                  tree path at given point, or <code>null</code> if no tree path is shown in the tree table
     */
    public TreePath getPathClosestTo(Point point) {
        return m_treeMapper.getPathClosestTo(point);
    }
    /**
     * Returns the row closest to given point. It returns <code>-1</code> if no row is shown in the tree table.
     *
     * @param point                             point at which closest row should be looked up
     * @return                                  row at given point, or <code>-1</code> if no row is shown in the tree table
     */
    public int getRowClosestTo(Point point) {
        TreePath path=getPathClosestTo(point);
        if (path==null)
            return -1;
        return getRowForPath(path);
    }
    /**
     * Overrides table implementation to return tree table cell renderer to tree columns. This method is called by Swing when
     * tree table needs to be painted.
     *
     * @param row                               row for which renderer is requested
     * @param column                            column for which renderer is requested
     * @return                                  table cell renderer for supplied row and column
     */
    public TableCellRenderer getCellRenderer(int row,int column) {
        if (convertColumnIndexToModel(column)==m_treeColumn)
            return m_treeTableCellRenderer;
        return getPlainCellRenderer(row,column);
    }
    /**
     * Returns the plain (original) cell renderer that has been registered for given column. Even for column makred as tree,
     * this method will never return tree table cell renderer, but will always return original renderer.
     *
     * @param row                               row for which renderer is requested
     * @param column                            column for which renderer is requested
     * @return                                  plain (original) table cell renderer for supplied row and column
     */
    public TableCellRenderer getPlainCellRenderer(int row,int column) {
        return super.getCellRenderer(row,column);
    }
    /**
     * Overrides the default Swing implementation to make sure that selection is not cleared when table data is updated.
     *
     * @param e                                 contains information about change
     */
    public void tableChanged(TableModelEvent e) {
        if (e==null || e.getFirstRow()==TableModelEvent.HEADER_ROW || e.getType()!=TableModelEvent.UPDATE) {
            super.tableChanged(e);
            resizeAndRepaint();
            return;
        }
        int end=e.getLastRow();
        if (end==Integer.MAX_VALUE) {
            resizeAndRepaint();
            return;
        }
        int modelColumn=e.getColumn();
        int start=e.getFirstRow();
        Rectangle dirtyRegion;
        if (modelColumn==TableModelEvent.ALL_COLUMNS) {
            dirtyRegion=getCellRect(start,0,false);
            dirtyRegion.x=0;
            dirtyRegion.width=getColumnModel().getTotalColumnWidth();
        }
        else {
            int column=convertColumnIndexToView(modelColumn);
            dirtyRegion=getCellRect(start,column,false);
        }
        Rectangle endCellRect=getCellRect(end,0,false);
        dirtyRegion.height=endCellRect.y+endCellRect.height-dirtyRegion.y;
        repaint(dirtyRegion.x,dirtyRegion.y,dirtyRegion.width,dirtyRegion.height);
    }
    /**
     * Overridden to disallow setting table model. Tree table control uses an internal table model that adapts tree table model
     * to a table layout. Changing this model is not possible.
     *
     * @param tableModel                                the table model
     * @exception UnsupportedOperationException         always thrown
     */
    public void setModel(TableModel tableModel) {
        if (tableModel==DUMMY_TABLE_MODEL) {
            super.setModel(tableModel);
            return;
        }
        throw new UnsupportedOperationException("Call JTreeTable.setModel(TableModel) is not allowed - use JTreeTable.setTreeTableModel() instead.");
    }
    /**
     * Overridden to disallow setting list selection model. Tree table control uses an internal list selection model that adapts
     * tree selection model to list selection model. Changing this model is not possible.
     *
     * @param listSelectionModel                        the list selection model
     * @exception UnsupportedOperationException         always thrown
     */
    public void setSelectionModel(ListSelectionModel listSelectionModel) {
        if (listSelectionModel==DUMMY_SELECTION_MODEL) {
            super.setSelectionModel(listSelectionModel);
            return;
        }
        throw new UnsupportedOperationException("Call JTreeTable.setSelectionModel(ListSelectionModel) is not allowed - use JTreeTable.setTreeSelectionModel() instead.");
    }
    /**
     * Returns the rectangle of the cell at given row and column.
     *
     * @param row                               row of the cell
     * @param column                            column of the cell
     * @param includeSpacing                    if <code>false</code>, return the true cell bounds - computed by subtracting the intercell spacing
     *                                          from the height and widths of the column and row models
     * @return                                  effective cell rectangle at given position
     */
    public Rectangle getCellRect(int row,int column,boolean includeSpacing) {
        Rectangle result=getCellRect(getPathForRow(row),column,includeSpacing);
        if (row>=getRowCount())
            result.height=getHeight();
        return result;
    }
    /**
     * Returns the rectangle of the cell at given path and column.
     *
     * @param path                              path to the cell
     * @param column                            column of the cell
     * @param includeSpacing                    if <code>false</code>, return the true cell bounds - computed by subtracting the intercell spacing
     *                                          from the height and widths of the column and row models
     * @return                                  cell rectangle at given position
     */
    public Rectangle getCellRect(TreePath path,int column,boolean includeSpacing) {
        Rectangle rectangle=new Rectangle();
        boolean valid=true;
        if (path!=null)
            m_treeMapper.getPathBounds(path,rectangle);
        else
            valid=false;
        rectangle.x=rectangle.width=0;
        if (column<0)
            valid=false;
        else if (column>=getColumnCount()) {
            rectangle.x=getWidth();
            valid=false;
        }
        else {
            TableColumnModel columnModel=getColumnModel();
            for (int i=0;i<column;i++)
                rectangle.x+=columnModel.getColumn(i).getWidth();
            rectangle.width=columnModel.getColumn(column).getWidth();
        }
        if (valid && !includeSpacing) {
            int rm=getRowMargin();
            int cm=getColumnModel().getColumnMargin();
            rectangle.setBounds(rectangle.x+cm/2,rectangle.y+rm/2,rectangle.width-cm,rectangle.height-rm);
        }
        return rectangle;
    }
    /**
     * Returns the effective rectangle of the cell at given path and column. Effiective recnagle differs from the usual cell rectangle
     * for tree column. There effective rectangle will include the rectangle used for displaying cell data only (without the area user
     * to paint node icon or the expansion handle control.
     *
     * @param path                              path to the cell
     * @param column                            column of the cell
     * @param includeSpacing                    if <code>false</code>, return the true cell bounds - computed by subtracting the intercell spacing
     *                                          from the height and widths of the column and row models
     * @return                                  effective cell rectangle at given position
     */
    public Rectangle getEffectiveCellRect(TreePath path,int column,boolean includeSpacing) {
        return getEffectiveCellRect(getRowForPath(path),column,includeSpacing);
    }
    /**
     * Returns the effective rectangle of the cell at given row and column. Effiective recnagle differs from the usual cell rectangle
     * for tree column. There effective rectangle will include the rectangle used for displaying cell data only (without the area user
     * to paint node icon or the expansion handle control.
     *
     * @param row                               row of the cell
     * @param column                            column of the cell
     * @param includeSpacing                    if <code>false</code>, return the true cell bounds - computed by subtracting the intercell spacing
     *                                          from the height and widths of the column and row models
     * @return                                  effective cell rectangle at given position
     */
    public Rectangle getEffectiveCellRect(int row,int column,boolean includeSpacing) {
        Rectangle result=getCellRect(row,column,includeSpacing);
        if (convertColumnIndexToModel(column)==m_treeColumn) {
            int indent=m_treeTableCellRenderer.getTreeNodeX(this,row);
            result.x+=indent;
            result.width-=indent;
            if (m_treeNodeRenderer!=null)
                m_treeNodeRenderer.adjustEffectiveRect(this,row,column,result);
        }
        return result;
    }
    /**
     * Returns the rectangle of the expansion handle control for given row.
     *
     * @param path                              the path to the row
     * @param rectangle                         rectangle to place data into
     * @return                                  rectangle of the handle control
     */
    public Rectangle getHandleControlRect(TreePath path,Rectangle rectangle) {
        return getHandleControlRect(getRowForPath(path),rectangle);
    }
    /**
     * Returns the rectangle of the expansion handle control for given path.
     *
     * @param row                               the row index
     * @param rectangle                         rectangle to place data into
     * @return                                  rectangle of the handle control
     */
    public Rectangle getHandleControlRect(int row,Rectangle rectangle) {
        Rectangle cellBounds=getCellRect(row,convertColumnIndexToView(m_treeColumn),false);
        m_treeTableCellRenderer.getHandleControlRect(this,row,rectangle);
        rectangle.x+=cellBounds.x;
        return rectangle;
    }
    /**
     * Returns the bounds of specifed row.
     *
     * @param row                               row index
     * @return                                  bounds of specified row
     */
    public Rectangle getRowBounds(int row) {
        Rectangle rect=getCellRect(row,0,true);
        rect.x=0;
        rect.width=getWidth();
        return rect;
    }
    /**
     * Returns the bounds of the row displaying last element of specifed path.
     *
     * @param path                              path for whose last element bounds are required
     * @return                                  bounds of specified path
     */
    public Rectangle getPathBounds(TreePath path) {
        return getRowBounds(getRowForPath(path));
    }
    /**
     * HACK: This method is overriden to spoof the UI to always paint any cell, regardless if table is currently editing or not.
     * By default, when painting the cell that is being edited, UI stretches the editor component across the entire cell and doesn't
     * paint the cell. This is not correct in case when cell in the tree column is being edited - then, it the editor component
     * needs to be stretched to the effective cell rectangle and the actual cell needs to be rendered (so that the handles and item
     * icon are rendered). This 'cheating' is done by temporary setting <code>editingRow</code> to <code>-1</code> during the
     * call to the super-class version of this method.
     *
     * @param g                                 where to paint the component
     */
    public void paint(Graphics g) {
        if (getEditorComponent()!=null && isEditing()) {
            getEditorComponent().setBounds(getEffectiveCellRect(getEditingRow(),getEditingColumn(),false));
            getEditorComponent().validate();
        }
        int currentEditingRow=getEditingRow();
        this.editingRow=-1;
        super.paint(g);
        this.editingRow=currentEditingRow;
    }
    /**
     * Overriddes the default implementation to do several things:
     * <ul>
     *  <li>check for clicks in the handle control and expands or collapses the tree if needed
     *  <li>makes sure that only clicks in effective cell rectangle start editing of the tree
     *  <li>set the size of the editor to effective cell rect
     *  <li>request focus to cell editor
     * </ul>
     * It would be wrong to install a mouse listener to handle expansion clicks, since the mouse event would always be passed
     * from the UI to this method first. Therefore, each click in the cell to expand the tree would also start editing.
     * By handling expansion events here we are sure that editing and expansion do not happen as the consequence of the same
     * mouse click.
     *
     * @param row                               row at which editing is requested
     * @param column                            column at which editing is requested
     * @param e                                 event object used to trigger editing
     * @return                                  <code>true</code> if editing should be started
     */
    public boolean editCellAt(int row,int column,EventObject e) {
        Rectangle effectiveCellRect=getEffectiveCellRect(row,column,false);
        if (convertColumnIndexToModel(column)==m_treeColumn && (e instanceof MouseEvent)) {
            MouseEvent mouseEvent=(MouseEvent)e;
            if (checkExpansionEvent(row,column,mouseEvent))
                return false;
            if (!effectiveCellRect.contains(mouseEvent.getPoint()))
                return false;
        }
        if (cellEditor!=null && !cellEditor.stopCellEditing())
            return false;
        if (row<0 || row>=getRowCount() || column<0 || column>=getColumnCount())
            return false;
        if (!isCellEditable(row,column))
            return false;
        TableCellEditor editor=getCellEditor(row,column);
        if (editor!=null && editor.isCellEditable(e)) {
            editorComp=prepareEditor(editor,row,column);
            if (editorComp==null) {
                removeEditor();
                return false;
            }
            editorComp.setBounds(effectiveCellRect);
            add(editorComp);
            editorComp.validate();
            if (editorComp instanceof JComboBox && ((JComboBox)editorComp).getEditor()!=null)
                ((JComboBox)editorComp).getEditor().getEditorComponent().requestFocus();
            else
                editorComp.requestFocus();
            setCellEditor(editor);
            setEditingRow(row);
            setEditingColumn(column);
            editor.addCellEditorListener(this);
            return true;
        }
        return false;
    }
    /**
     * Edits the cell at specified path and column.
     *
     * @param path                              path at which editing is requested
     * @param column                            column at which editing is requested
     * @param e                                 event object used to trigger editing
     * @return                                  <code>true</code> if editing should be started
     */
    public boolean editCellAt(TreePath path,int column,EventObject e) {
        return editCellAt(getRowForPath(path),column,e);
    }
    /**
     * Edits the cell at specified path and column.
     *
     * @param path                              path at which editing is requested
     * @param column                            column at which editing is requested
     * @return                                  <code>true</code> if editing should be started
     */
    public boolean editCellAt(TreePath path,int column) {
        return editCellAt(path,column,null);
    }
    /**
     * Returns the path that is currently being edited, or <code>null</code> if table is not editing a cell currently.
     *
     * @return                                  path being edited or <code>null</code> if table is not editing currently
     */
    public TreePath getEditingPath() {
        if (getEditingRow()==-1)
            return null;
        return getPathForRow(getEditingRow());
    }
    /**
     * Makes tree table stop editing a cell.
     *
     * @return                                  <code>true</code> if edited value was validated successfully by the cell editor and editing was actually stopped
     */
    public boolean stopEditing() {
        if (!isEditing() || getCellEditor()==null)
            return false;
        return getCellEditor().stopCellEditing();
    }
    /**
     * Makes tree table cancel editing a cell. Cell retains its original value.
     */
    public void cancelEditing() {
        if (isEditing() && getCellEditor()!=null)
            getCellEditor().cancelCellEditing();
    }
    /**
     * Returns the index of the row displaying last component of specified tree path. Will return <code>-1</code> if path is not visible.
     *
     * @param path                              path of the row being edited
     * @return                                  index of the row displaying specifed path (or <code>-1</code> if path is not visible)
     */
    public int getRowForPath(TreePath path) {
        return m_treeMapper.getRowForPath(path);
    }
    /**
     * Returns the path whose last component is displayed in specified row. Will return <code>null</code> if specified row doesn't contain a path.
     *
     * @param row                               row for which a path is requested
     * @return                                  path displayed in specified row
     */
    public TreePath getPathForRow(int row) {
        return m_treeMapper.getPathForRow(row);
    }
    /**
     * Returns the index of the column (in column model coordinates) that is used to paint the tree. Default value is 0.
     *
     * @return                                  index of the tree column
     */
    public int getTreeColumn() {
        return m_treeColumn;
    }
    /**
     * Sets the index of the column (in column model coordinates) that is used to paint the tree. Default value is 0.
     *
     * @param treeColumn                        index of the tree column
     */
    public void setTreeColumn(int treeColumn) {
        int oldTreeColumn=m_treeColumn;
        m_treeColumn=treeColumn;
        firePropertyChange(TREE_COLUMN_PROPERTY,oldTreeColumn,m_treeColumn);
        repaint();
    }
    /**
     * Returns current tree node renderer. Can be <code>null</code>, in that case tree node is not painted.
     *
     * @return                                  current tree node renderer
     */
    public TreeNodeRenderer getTreeNodeRenderer() {
        return m_treeNodeRenderer;
    }
    /**
     * Sets the tree node renderer used to paint the tree node. Can be <code>null</code>, in that case tree node is not painted.
     *
     * @param treeNodeRenderer                  new tree node renderer
     */
    public void setTreeNodeRenderer(TreeNodeRenderer treeNodeRenderer) {
        TreeNodeRenderer oldTreeNodeRenderer=m_treeNodeRenderer;
        m_treeNodeRenderer=treeNodeRenderer;
        firePropertyChange(TREE_NODE_RENDERER_PROPERTY,oldTreeNodeRenderer,m_treeNodeRenderer);
    }
    /**
     * Returns current tree path measurer used to determine the side of tree path. Tree path measurer will be used only if
     * <code>rowSize</code> property has value &lt;=0.
     *
     * @return                                  tree path measurer used to measure paths
     */
    public TreePathMeasurer getTreePathMeasurer() {
        return m_treePathMeasurer;
    }
    /**
     * Sets tree path measurer used to determine the side of tree path. Tree path measurer will be used only if
     * <code>rowSize</code> property has value &lt;=0.
     *
     * @param treePathMeasurer                  tree path measurer used to measure paths
     */
    public void setTreePathMeasurer(TreePathMeasurer treePathMeasurer) {
        if (treePathMeasurer==null)
            throw new IllegalArgumentException("treePathMeasurer property cannot have the value of null.");
        TreePathMeasurer oldTreePathMeasurer=m_treePathMeasurer;
        m_treePathMeasurer=treePathMeasurer;
        firePropertyChange(TREE_PATH_MEASURER_PROPERTY,oldTreePathMeasurer,m_treePathMeasurer);
    }
    /**
     * Returns the number of pixels that the tree node at given path will be indented.
     *
     * @param path                              path for which indent is requsted
     * @return                                  number of pixels that tree node for given path should be indented
     */
    public int getTreeNodeIndent(TreePath path) {
        return m_treeMapper.getTreeNodeIndent(path);
    }
    /**
     * Returns the number of pixels that the tree node at given row will be indented.
     *
     * @param row                               row for which indent is requsted
     * @return                                  number of pixels that tree node for given row should be indented
     */
    public int getTreeNodeIndent(int row) {
        return getTreeNodeIndent(getPathForRow(row));
    }
    /**
     * Checks whether given tree path is expanded.
     *
     * @param path                              path to be checked
     * @return                                  <code>true</code> if path is expanded
     */
    public boolean isExpanded(TreePath path) {
        return m_treeMapper.isExpanded(path);
    }
    /**
     * Checks whether given row is expanded.
     *
     * @param row                               row to be checked
     * @return                                  <code>true</code> if row is expanded
     */
    public boolean isExpanded(int row) {
        return isExpanded(getPathForRow(row));
    }
    /**
     * Checks whether given tree path is collapsed.
     *
     * @param path                              path to be checked
     * @return                                  <code>true</code> if path is collapsed
     */
    public boolean isCollapsed(TreePath path) {
        return m_treeMapper.isCollapsed(path);
    }
    /**
     * Checks whether given row is collapsed.
     *
     * @param row                               row to be checked
     * @return                                  <code>true</code> if row is collapsed
     */
    public boolean isCollapsed(int row) {
        return isCollapsed(getPathForRow(row));
    }
    /**
     * Checks whteher given path is visible (that is, all its parents have been expanded).
     *
     * @param path                              path to be checked
     * @return                                  <code>true</code> if path is visible
     */
    public boolean isVisible(TreePath path) {
        if(path==null)
            return false;
        TreePath parentPath=path.getParentPath();
        if (parentPath!=null)
            return isExpanded(parentPath);
        return true;
    }
    /**
     * Checks whteher given row is visible (that is, all its parents have been expanded).
     *
     * @param row                               row to be checked
     * @return                                  <code>true</code> if row is visible
     */
    public boolean isVisible(int row) {
        return isVisible(getPathForRow(row));
    }
    /**
     * Makes sure that supplied path is visible.
     *
     * @param path                              path that should be made visible
     */
    public void makeVisible(TreePath path) {
        if (path==null)
            return;
        TreePath parentPath=path.getParentPath();
        if (parentPath!=null) {
            expandPath(parentPath);
            makeVisible(parentPath);
        }
    }
    /**
     * Makes sure that supplied row is visible.
     *
     * @param row                               row that should be made visible
     */
    public void makeVisible(int row) {
        makeVisible(getPathForRow(row));
    }
    /**
     * Expands specified path. If path is expanded this method does nothing.
     *
     * @param path                              path to be expanded
     */
    public void expandPath(TreePath path) {
        if (!isCollapsed(path))
            return;
        if (isEditing() && !getCellEditor().stopCellEditing())
            return;
        try {
            fireTreeWillExpand(path);
        }
        catch (ExpandVetoException vetoed) {
            return;
        }
        m_treeMapper.expandPath(path);
        fireTreeExpanded(path);
    }
    /**
     * Expands specified row. If row is expanded this method does nothing.
     *
     * @param row                               row to be expanded
     */
    public void expandRow(int row) {
        expandPath(getPathForRow(row));
    }
    /**
     * Collapses specified path. If path is not expanded this method does nothing.
     *
     * @param path                              path to be collapsed
     */
    public void collapsePath(TreePath path) {
        if (!isExpanded(path))
            return;
        if (isEditing() && !getCellEditor().stopCellEditing())
            return;
        try {
            fireTreeWillCollapse(path);
        }
        catch (ExpandVetoException vetoed) {
            return;
        }
        m_treeMapper.collapsePath(path);
        fireTreeCollapsed(path);
    }
    /**
     * Collapses specified path. If path is not expanded this method does nothing.
     *
     * @param row                               row index to be collapsed
     */
    public void collapseRow(int row) {
        collapsePath(getPathForRow(row));
    }
    /**
     * Checks if the mouse event in specified row and column is an event that should trigger expaning or collapsing of the tree.
     * If so, this method will expand or collapse the tree and return <code>true</code>.
     *
     * @param row                               row in which event occured
     * @param column                            column in which event occured
     * @param mouseEvent                        event that occurend in speicfied row and column that needs to be checked
     * @return                                  <code>true</code> if supplied event is an event that should expand or collapse the tree
     */
    protected boolean checkExpansionEvent(int row,int column,MouseEvent mouseEvent) {
        if (m_treeTableCellRenderer.isInHandleControl(this,mouseEvent.getPoint())) {
            TreePath path=getPathForRow(row);
            if (isExpanded(path))
                collapsePath(path);
            else
                expandPath(path);
            return true;
        }
        return false;
    }
    /**
     * Returns an <code>Enumeration</code> of <code>TreePath</code> objects that idenfity paths expanded under given node.
     *
     * @param parent                            identifies node under which all expanded descendants are returned
     * @return                                  <code>Enumeration</code> of <code>TreePath</code> object that are expanded under given node
     */
    public Enumeration getExpandedDescendants(TreePath parent) {
        return m_treeMapper.getExpandedDescendants(parent);
    }
    /**
     * Determines whether tree should show expansion handles for the root node.
     *
     * @param newValue                          if <code>true</code>, root node will show expansion handles
     */
    public void setShowsRootHandles(boolean newValue) {
        boolean oldValue=m_showsRootHandles;
        m_showsRootHandles=newValue;
        repaint();
        firePropertyChange(SHOWS_ROOT_HANDLES_PROPERTY,oldValue,m_showsRootHandles);
    }
    /**
     * Returns whether tree should show expansion handles for the root node.
     *
     * @return                                  if <code>true</code>, root node will show expansion handles
     */
    public boolean getShowsRootHandles() {
        return m_showsRootHandles;
    }
    /**
     * Determines whether tree should scroll as many as possible chidlren into view when some item has been expanded.
     *
     * @param newValue                          if <code>true</code>, when some item is expanded, tree will scroll as many children into view
     */
    public void setScrollsOnExpand(boolean newValue) {
        boolean oldValue=m_scrollsOnExpand;
        m_scrollsOnExpand=newValue;
        firePropertyChange(SCROLLS_ON_EXPAND_PROPERTY,oldValue,m_scrollsOnExpand);
    }
    /**
     * Returns whether tree should scroll as many as possible chidlren into view when some item has been expanded.
     *
     * @return                                  if <code>true</code>, when some item is expanded, tree will scroll as many children into view
     */
    public boolean getScrollsOnExpand() {
        return m_scrollsOnExpand;
    }
    /**
     * Updates the selection models of the table, depending on the state of the two flags: <code>toggle</code> and <code>extend</code>.
     * <p>This implementation uses the following conventions:
     * <ul>
     *  <li><code>toggle</code>: <em>false</em>, <code>extend</code>: <em>false</em>.
     *      Clear the previous selection and ensure the new cell is selected.
     *  <li><code>toggle</code>: <em>false</em>, <code>extend</code>: <em>true</em>.
     *      Extend the previous selection to include the specified cell.
     *  <li><code>toggle</code>: <em>true</em>, <code>extend</code>: <em>false</em>.
     *      If the specified cell is selected, deselect it. If it is not selected, select it.
     *  <li><code>toggle</code>: <em>true</em>, <code>extend</code>: <em>true</em>.
     *      Leave the selection state as it is, but move the anchor index to the specified location.
     * </ul>
     *
     * @param path                              affects the selection at that path
     * @param columnIndex                       affects the selection at that column
     * @param toggle                            see description above
     * @param extend                            if <code>true</code>, extend the current selection
     *
     */
    public void changeSelection(TreePath path,int columnIndex,boolean toggle,boolean extend) {
        changeSelection(getRowForPath(path),columnIndex,toggle,extend);
    }
    /**
     * Cehcks whether cell at specified path and column is selected.
     *
     * @param path                              path at which cell needs to be checked
     * @param column                            column at which cell needs to be checked
     * @return                                  <code>true</code> if the cell at given path and column is selected
     */
    public boolean isCellSelected(TreePath path,int column) {
        return isCellSelected(getRowForPath(path),column);
    }
    /**
     * Adds the path to row selection.
     *
     * @param path                              path to be selected
     */
    public void addRowSelectionPath(TreePath path) {
        getTreeSelectionModel().addSelectionPath(path);
    }
    /**
     * Adds paths to row selection.
     *
     * @param paths                             paths to be selected
     */
    public void addRowSelectionPaths(TreePath[] paths) {
        getTreeSelectionModel().addSelectionPaths(paths);
    }
    /**
     * Removes the path from row selection.
     *
     * @param path                              path to be unselected
     */
    public void removeRowSelectionPath(TreePath path) {
        getTreeSelectionModel().removeSelectionPath(path);
    }
    /**
     * Removes paths from row selection.
     *
     * @param paths                             paths to be unselected
     */
    public void removeRowSelectionPaths(TreePath[] paths) {
        getTreeSelectionModel().removeSelectionPaths(paths);
    }
    /**
     * Changes row selection to contain only specified path - all other paths are unselected.
     *
     * @param path                              path to be selected
     */
    public void setRowSelectionPath(TreePath path) {
        getTreeSelectionModel().setSelectionPath(path);
    }
    /**
     * Changes row selection to contain only specified paths - all other paths are unselected.
     *
     * @param paths                             paths to be selected
     */
    public void setRowSelectionPaths(TreePath[] paths) {
        getTreeSelectionModel().setSelectionPaths(paths);
    }
    /**
     * Returns the last component of selected path. Can be <code>null</code> if nothing is selected. This is useful only if only
     * one path is selected.
     *
     * @return                                  last component of selected path or code>null</code> if nothing is selected
     */
    public Object getLastSelectedPathComponent() {
        TreePath selectedPath=getTreeSelectionModel().getSelectionPath();
        if (selectedPath!=null)
            return selectedPath.getLastPathComponent();
        return null;
    }
    /**
     * Returns the first path in the selection. This is useful if there if only one item currently selected. Can be <code>null</code>
     * if nothing is selected.
     *
     * @return                                  first path in selection or <code>null</code> if nothing is selected
     */
    public TreePath getSelectionPath() {
        return getTreeSelectionModel().getSelectionPath();
    }
    /**
     * Returns the paths in the selection. It will return <code>null</code> or an empty array if nothing is selected.
     *
     * @return                                  selected paths (or <code>null</code> or an empty array if nothing is selected)
     */
    public TreePath[] getSelectionPaths() {
        return getTreeSelectionModel().getSelectionPaths();
    }
    /**
     * Returns the number of paths in the selection.
     *
     * @return                                  the number of paths in the selection
     */
    public int getSelectionCount() {
        return getTreeSelectionModel().getSelectionCount();
    }
    /**
     * Returns the minimum index of the selected row (or <code>-1</code> if nothing is selected).
     *
     * @return                                  the minimum index of the selected row (or <code>-1</code> if nothing is selected)
     */
    public int getMinSelectionRow() {
        return getTreeSelectionModel().getMinSelectionRow();
    }
    /**
     * Returns the maximum index of the selected row (or <code>-1</code> if nothing is selected).
     *
     * @return                                  the maximum index of the selected row (or <code>-1</code> if nothing is selected)
     */
    public int getMaxSelectionRow() {
        return getTreeSelectionModel().getMaxSelectionRow();
    }
    /**
     * Checks whether supplied path is selected.
     *
     * @param path                              path to be checked
     * @return                                  <code>true</code> if path is currently selected
     */
    public boolean isPathSelected(TreePath path) {
        return getTreeSelectionModel().isPathSelected(path);
    }
    /**
     * Checks whether supplied row is selected.
     *
     * @param row                               row to be checked
     * @return                                  <code>true</code> if row is currently selected
     */
    public boolean isRowSelected(int row) {
        return getTreeSelectionModel().isRowSelected(row);
    }
    /**
     * Adds a listener that is notified whenever tree node is expanded.
     *
     * @param listener                          listener to be added
     */
    public void addTreeExpansionListener(TreeExpansionListener listener) {
        listenerList.add(TreeExpansionListener.class,listener);
    }
    /**
     * Removes a listener that is notified whenever tree node is expanded.
     *
     * @param listener                          listener to be removed
     */
    public void removeTreeExpansionListener(TreeExpansionListener listener) {
        listenerList.remove(TreeExpansionListener.class,listener);
    }
    /**
     * Fires a notification that given tree path has been expanded.
     *
     * @param path                              path for which notification is generated
     */
    protected void fireTreeExpanded(TreePath path) {
        Object[] listeners=listenerList.getListenerList();
        TreeExpansionEvent event=null;
        for (int i=listeners.length-2;i>=0;i-=2)
            if (listeners[i]==TreeExpansionListener.class) {
                if (event==null)
                    event=new TreeExpansionEvent(this,path);
                ((TreeExpansionListener)listeners[i+1]).treeExpanded(event);
            }
    }
    /**
     * Fires a notification that given tree path has been collapsed.
     *
     * @param path                              path for which notification is generated
     */
    protected void fireTreeCollapsed(TreePath path) {
        Object[] listeners=listenerList.getListenerList();
        TreeExpansionEvent event=null;
        for (int i=listeners.length-2;i>=0;i-=2)
            if (listeners[i]==TreeExpansionListener.class) {
                if (event==null)
                    event=new TreeExpansionEvent(this,path);
                ((TreeExpansionListener)listeners[i+1]).treeCollapsed(event);
            }
    }
    /**
     * Adds a listener that is notified whenever tree node is about to be expanded. This listener can prevent expansion to occur.
     *
     * @param listener                          listener to be added
     */
    public void addTreeWillExpandListener(TreeWillExpandListener listener) {
        listenerList.add(TreeWillExpandListener.class,listener);
    }
    /**
     * Removes a listener that is notified whenever tree node is about to be expanded.
     *
     * @param listener                          listener to be removed
     */
    public void removeTreeWillExpandListener(TreeWillExpandListener listener) {
        listenerList.remove(TreeWillExpandListener.class,listener);
    }
    /**
     * Fires a notification that given tree path is about to be expanded. Listener can throw <code>javax.swing.event.ExpandVetoException</code>
     * to prevent expanding to occur.
     *
     * @param path                              path for which notification is generated
     * @exception ExpandVetoException           thrown by a listener to prevent expanding to happen
     */
    protected void fireTreeWillExpand(TreePath path) throws ExpandVetoException {
        Object[] listeners=listenerList.getListenerList();
        TreeExpansionEvent event=null;
        for (int i=listeners.length-2;i>=0;i-=2)
            if (listeners[i]==TreeWillExpandListener.class) {
                if (event==null)
                    event=new TreeExpansionEvent(this,path);
                ((TreeWillExpandListener)listeners[i+1]).treeWillExpand(event);
            }
    }
    /**
     * Fires a notification that given tree path is about to be collapsed. Listener can throw <code>javax.swing.event.ExpandVetoException</code>
     * to prevent collapsing to occur.
     *
     * @param path                              path for which notification is generated
     * @exception ExpandVetoException           thrown by a listener to prevent collapsing to happen
     */
    protected void fireTreeWillCollapse(TreePath path) throws ExpandVetoException {
        Object[] listeners=listenerList.getListenerList();
        TreeExpansionEvent event=null;
        for (int i=listeners.length-2;i>=0;i-=2)
            if (listeners[i]==TreeWillExpandListener.class) {
                if (event==null)
                    event=new TreeExpansionEvent(this,path);
                ((TreeWillExpandListener)listeners[i+1]).treeWillCollapse(event);
            }
    }
    /**
     * Scrolls supplied path into view.
     *
     * @param path                              path to be scrolled into view
     */
    public void scrollPathToVisible(TreePath path) {
        makeVisible(path);
        Rectangle bounds=getPathBounds(path);
        if (bounds!=null)
            scrollRectToVisible(bounds);
    }
    /**
     * Scrolls supplied row into view.
     *
     * @param row                               row to be scrolled into view
     */
    public void scrollRowToVisible(int row) {
        scrollPathToVisible(getPathForRow(row));
    }
    /**
     * Returns the value of the node at given path in supplied column.
     *
     * @param path                              path for which value should be retrieved
     * @param column                            column in vhich value should be retrieved
     * @return                                  value at given path in given column
     */
    public Object getValueAt(TreePath path,int column) {
        return getValueAt(getRowForPath(path),column);
    }
    /**
     * Sets the value of the node at given path in supplied column.
     *
     * @param aValue                            value to be set at given path in given column
     * @param path                              path for which value should be set
     * @param column                            column in vhich value should be set
     */
    public void setValueAt(Object aValue,TreePath path,int column) {
        setValueAt(aValue,getRowForPath(path),column);
    }

    /**
     * Tree node renderer that implements <code>javax.swing.plaf.UIResource</code>. This renderer is used in the tree table
     * by default. It is marked as the UIResource, so if UI is changed, table will create new tree node renderer instance.
     */
    protected static class DefaultTreeNodeRendererResource extends DefaultTreeNodeRenderer implements UIResource {
    }
    /**
     * Returns the scroll increment (in pixels) that completely exposes one new row or column (depending on the orientation).
     *
     * @param visibleRect                       view area visible within the viewport
     * @param orientation                       either <code>SwingConstants.VERTICAL</code> or <code>SwingConstants.HORIZONTAL</code>
     * @param direction                         less than zero to scroll up/left, greater than zero for down/right
     * @return                                  "unit" increment for scrolling in the specified direction
     */
    public int getScrollableUnitIncrement(Rectangle visibleRect,int orientation,int direction) {
        if (orientation==SwingConstants.HORIZONTAL)
            return super.getScrollableUnitIncrement(visibleRect,orientation,direction);
        int increment=0;
        if (direction<0) {
            int visibleRow=getRowClosestTo(new Point(visibleRect.x,visibleRect.y));
            if (visibleRow!=-1) {
                Rectangle newTopRowBounds=getRowBounds(visibleRow);
                if (newTopRowBounds.y>=visibleRect.y && visibleRow>0)
                    newTopRowBounds=getRowBounds(visibleRow-1);
                increment=visibleRect.y-newTopRowBounds.y;
            }
        }
        if (direction>0) {
            int visibleRow=getRowClosestTo(new Point(visibleRect.x,visibleRect.y+visibleRect.height));
            if (visibleRow!=-1) {
                int visibleBottom=visibleRect.y+visibleRect.height;
                Rectangle newBottomRowBounds=getRowBounds(visibleRow);
                if (newBottomRowBounds.y+newBottomRowBounds.height<=visibleBottom && visibleRow<getRowCount()-1)
                    newBottomRowBounds=getRowBounds(visibleRow+1);
                increment=newBottomRowBounds.y+newBottomRowBounds.height-visibleBottom;
            }
        }
        if (increment<0)
            increment=0;
        return increment;
    }
    /**
     * Returns <code>visibleRect.height</code> or <code>visibleRect.width</code>, depending on this table's orientation.
     *
     * @param visibleRect                       view area visible within the viewport
     * @param orientation                       either <code>SwingConstants.VERTICAL</code> or <code>SwingConstants.HORIZONTAL</code>
     * @param direction                         less than zero to scroll up/left, greater than zero for down/right
     * @return                                  <code>visibleRect.height</code> or <code>visibleRect.width</code> per the orientation
     */
    public int getScrollableBlockIncrement(Rectangle visibleRect,int orientation,int direction) {
        if (orientation==SwingConstants.HORIZONTAL)
            return super.getScrollableBlockIncrement(visibleRect,orientation,direction);
        int increment=0;
        if (direction<0) {
            int visibleRow=getRowClosestTo(new Point(visibleRect.x,visibleRect.y-visibleRect.height));
            if (visibleRow!=-1) {
                Rectangle newTopRowBounds=getRowBounds(visibleRow);
                increment=visibleRect.y-newTopRowBounds.y;
            }
        }
        if (direction>0) {
            int visibleRow=getRowClosestTo(new Point(visibleRect.x,visibleRect.y+2*visibleRect.height));
            if (visibleRow!=-1) {
                int visibleBottom=visibleRect.y+visibleRect.height;
                Rectangle newBottomRowBounds=getRowBounds(visibleRow);
                increment=newBottomRowBounds.y+newBottomRowBounds.height-visibleBottom;
            }
        }
        if (increment<0)
            increment=0;
        return increment;
    }
}
