package de.fzi.wim.guibase.treetable;

import java.awt.Component;
import java.awt.Rectangle;
import javax.swing.JLabel;
import javax.swing.Icon;
import javax.swing.UIManager;

/**
 * Default implementation of {@link TreeNodeRenderer} interface that mimics the way tree nodes are rendered in trees. This class
 * should not render actual value - only possibly the icon of a node.
 */
public class DefaultTreeNodeRenderer extends JLabel implements TreeNodeRenderer {
    /** Icon displayed when the node is open. */
    protected Icon m_openIcon;
    /** Icon displayed when node is closed. */
    protected Icon m_closedIcon;
    /** Icon displayed when node is leaf. */
    protected Icon m_leafIcon;

    /**
     * Creates and initializes an instance of this class with defaults in the current L&F.
     */
    public DefaultTreeNodeRenderer() {
        setHorizontalAlignment(JLabel.LEFT);
        setLeafIcon(UIManager.getIcon("Tree.leafIcon"));
        setClosedIcon(UIManager.getIcon("Tree.closedIcon"));
        setOpenIcon(UIManager.getIcon("Tree.openIcon"));
    }
    /**
     * Returns the icon for leaf nodes.
     *
     * @return                              icon for leaf nodes
     */
    public Icon getLeafIcon() {
        return m_leafIcon;
    }
    /**
     * Sets the icon for leaf nodes.
     *
     * @param icon                          icon for leaf nodes
     */
    public void setLeafIcon(Icon icon) {
        m_leafIcon=icon;
    }
    /**
     * Returns the icon for closed nodes.
     *
     * @return                              icon for closed nodes
     */
    public Icon getClosedIcon() {
        return m_closedIcon;
    }
    /**
     * Returns the icon for closed nodes.
     *
     * @param icon                          icon for closed nodes
     */
    public void setClosedIcon(Icon icon) {
        m_closedIcon=icon;
    }
    /**
     * Returns the icon for open nodes.
     *
     * @return                              icon for open nodes
     */
    public Icon getOpenIcon() {
        return m_openIcon;
    }
    /**
     * Returns the icon for open nodes.
     *
     * @param icon                          icon for open nodes
     */
    public void setOpenIcon(Icon icon) {
        m_openIcon=icon;
    }
    /**
     * Returns the component used to paint given cell.
     *
     * @param treeTable                         tree table in which node is to be rendered
     * @param isSelected                        <code>true</code> if node is currently selected
     * @param hasFocus                          <code>true</code> if node has focus
     * @param row                               row of the node
     * @param column                            column of the node
     * @return                                  component used to paint the node of the tree
     */
    public Component getTableCellRendererComponent(JTreeTable treeTable,boolean isSelected,boolean hasFocus,int row,int column) {
        if (isSelected) {
            setForeground(treeTable.getSelectionForeground());
            setBackground(treeTable.getSelectionBackground());
        }
        else {
            setForeground(treeTable.getForeground());
            setBackground(treeTable.getBackground());
        }
        if (hasFocus && treeTable.isCellEditable(row,column)) {
            setForeground(UIManager.getColor("Table.focusCellForeground"));
            setBackground(UIManager.getColor("Table.focusCellBackground"));
        }
        setIcon(getItemIcon(treeTable,row));
        return this;
    }
    /**
     * Returns the icon used for the item in given row.
     *
     * @param treeTable                         tree table in which row is displayed
     * @param row                               row for which icon should be returned
     * @return                                  icon used for the item in given row
     */
    protected Icon getItemIcon(JTreeTable treeTable,int row) {
        TreeTableModel model=treeTable.getTreeTableModel();
        Object node=treeTable.getPathForRow(row).getLastPathComponent();
        if (model.isLeaf(node))
            return m_leafIcon;
        else if (treeTable.isExpanded(row))
            return m_openIcon;
        else
            return m_closedIcon;
    }
    /**
     * Overridden as no-op to prevent validation events to propagate in the Swing containment hierarchy.
     */
    public void validate() {
    }
    /**
     * Overridden as no-op to prevent validation events to propagate in the Swing containment hierarchy.
     */
    public void revalidate() {
    }
    /**
     * Overridden as no-op to prevent repaint events being posted.
     */
    public void repaint(long tm,int x,int y,int width,int height) {
    }
    /**
     * Overridden as no-op to prevent repaint events being posted.
     */
    public void repaint(Rectangle r) {
    }
    /**
     * Overridden as no-op to prevent property change notifications to be fired
     */
    public void firePropertyChange(String propertyName,boolean oldValue,boolean newValue) {
    }
    /**
     * Adjusts the effective cell rectange to exclude the icon for the node.
     *
     * @param treeTable                         tree table in which node is to be rendered
     * @param row                               row of the node
     * @param column                            column of the node
     * @param rectangle                         rectangle to be adjusted
     */
    public void adjustEffectiveRect(JTreeTable treeTable,int row,int column,Rectangle rectangle) {
        Icon icon=getItemIcon(treeTable,row);
        int iconWidth=icon==null ? 0 : icon.getIconWidth();
        rectangle.x+=iconWidth;
        rectangle.width-=iconWidth;
    }
}
