package de.fzi.wim.guibase.toolbars;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import javax.swing.Action;
import javax.swing.AbstractButton;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JToolBar;
import javax.swing.JToggleButton;
import javax.swing.JComponent;
import org.w3c.dom.Node;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import de.fzi.wim.guibase.localization.LocalizationManager;
import de.fzi.wim.guibase.actions.SmartActionAggregate;
import de.fzi.wim.guibase.actions.SmartActionManager;
import de.fzi.wim.guibase.actions.SmartAction;
import de.fzi.wim.guibase.actions.SmartActionMap;

/**
 * Utility class that can be used to build menus. Structure of menus is defined by an XML structure.
 * Instance of this class must be given a localization manager and an action map of all actions known to the application.
 * <p>Toolbars can be initialized from an XML defintion. This definition has the folliwing structure:
 * <pre>
 * <toolbar>
 *  <button actionID="some.action.id"/>
 *  <separator/>
 *  ...
 * </toolbar>
 * </pre>
 */
public class LocalizedToolBarBuilder {
    /** Localization manager that is used to build menu items. */
    protected LocalizationManager m_localizationManager;
    /** Map of all actions (mapped by their action ID). */
    protected SmartActionMap m_actions;
    /** The manager. */
    protected SmartActionManager m_smartActionManager;

    /**
     * Creates an instalce of this class and initializes it with a localization manager and an action map.
     *
     * @param localizationManager               source of the localization information
     * @param smartActionManager                the smart action manager
     * @param actions                           map of actions (mapped by their action ID)
     */
    public LocalizedToolBarBuilder(LocalizationManager localizationManager,SmartActionManager smartActionManager,SmartActionMap actions) {
        m_localizationManager=localizationManager;
        m_smartActionManager=smartActionManager;
        m_actions=actions;
    }
    /**
     * Creates a toolbar from given definition.
     *
     * @param orientation                       orientation of this toolbar
     * @param definition                        definition of toolbar in XML
     * @return                                  the toolbar
     */
    public JToolBar createToolBar(int orientation,Element definition) {
        if (!"toolbar".equalsIgnoreCase(definition.getNodeName()))
            throw new IllegalArgumentException("Root element of toolbar definition must be <toolbar>.");
        String toolBarID=definition.getAttribute("key");
        JToolBar toolBar=new JToolBar(m_localizationManager.getPhrase(toolBarID),orientation);
        loadComponent(definition.getChildNodes(),toolBar,null,"",null);
        return toolBar;
    }
    /**
     * Loads the component from the definition.
     *
     * @param nodeList                          the list of nodes
     * @param component                         component where the elements are added
     * @param inheritedContexts                 the names of inherited contexts
     * @param inheritedType                     the inherited component type
     * @param inheritedHideWhenDisabled         the inderited 'hide when disabled' flag
     */
    protected void loadComponent(NodeList nodeList,JComponent component,String inheritedContexts,String inheritedType,String inheritedHideWhenDisabled) {
        int length=nodeList.getLength();
        for (int i=0;i<length;i++) {
            Node child=nodeList.item(i);
            if (child.getNodeType()==Node.ELEMENT_NODE) {
                JComponent addedUIElement=null;
                Element element=(Element)child;
                String nodeName=element.getNodeName();
                if ("button".equalsIgnoreCase(nodeName)) {
                    AbstractButton button=createToolbarButton(element,inheritedContexts,inheritedType,inheritedHideWhenDisabled);
                    addedUIElement=button;
                    String actionID=element.getAttribute("actionID");
                    if (actionID.length()!=0) {
                        SmartAction action=m_actions.getAction(actionID);
                        if (action==null)
                            button.setText("Invalid action ID '"+actionID+"'");
                        else
                            button.setAction(action);
                    }
                }
                else if ("separator".equalsIgnoreCase(nodeName)) {
                    addedUIElement=new JToolBar.Separator();
                    String contexts=element.getAttribute("contexts");
                    if (contexts.length()==0)
                        contexts=inheritedContexts;
                    if (contexts!=null)
                        addedUIElement.putClientProperty("guibase.contexts",","+contexts+",");
                }
                else if ("aggregate".equalsIgnoreCase(nodeName))
                    addedUIElement=createAggregate(element,inheritedContexts,inheritedType,inheritedHideWhenDisabled);
                else if ("group".equalsIgnoreCase(nodeName)) {
                    String newInheritedContexts=inheritedContexts;
                    String contexts=element.getAttribute("contexts");
                    if (contexts.length()!=0)
                        newInheritedContexts=contexts;
                    loadComponent(element.getChildNodes(),component,newInheritedContexts,inheritedType,inheritedHideWhenDisabled);
                }
                if (addedUIElement!=null) {
                    addedUIElement.setFocusable(false);
                    component.add(addedUIElement);
                }
            }
        }
    }
    /**
     * Creates an aggregate from XML definition. Node passed in must be an <code>aggregate</code> node.
     *
     * @param aggregateNode                     node defining the aggregate
     * @param inheritedContexts                 the names of inherited contexts
     * @param inheritedType                     the inherited component type
     * @param inheritedHideWhenDisabled         the inderited 'hide when disabled' flag
     * @return                                  menu item for the aggregate node
     */
    protected AbstractButton createAggregate(Element aggregateNode,String inheritedContexts,String inheritedType,String inheritedHideWhenDisabled) {
        AbstractButton button=createToolbarButton(aggregateNode,inheritedContexts,inheritedType,inheritedHideWhenDisabled);
        String actionID=aggregateNode.getAttribute("actionID");
        if (actionID.length()!=0) {
            SmartActionAggregate aggregate=(SmartActionAggregate)m_actions.getAction(actionID);
            if (aggregate==null) {
                aggregate=new SmartActionAggregate(actionID,m_smartActionManager,m_localizationManager);
                m_actions.put(actionID,aggregate);
            }
            NodeList nodeList=aggregateNode.getChildNodes();
            for (int childIndex=0;childIndex<nodeList.getLength();childIndex++) {
                Node aggregatedChild=nodeList.item(childIndex);
                if (aggregatedChild.getNodeType()==Node.ELEMENT_NODE) {
                    Element aggregatedChildElement=(Element)aggregatedChild;
                    if ("action".equalsIgnoreCase(aggregatedChildElement.getNodeName())) {
                        String aggregatedActionID=aggregatedChildElement.getAttribute("actionID");
                        SmartAction aggregatedAction=m_actions.getAction(aggregatedActionID);
                        aggregate.addAction(aggregatedAction);
                    }
                }
            }
            aggregate.updateAction();
            button.setAction(aggregate);
        }
        return button;
    }
    /**
     * Creates the toolbar button for the node specified.
     *
     * @param element                           node defining the properties of the button
     * @param inheritedContexts                 the names of inherited contexts
     * @param inheritedType                     the inherited component type
     * @param inheritedHideWhenDisabled         the inderited 'hide when disabled' flag
     * @return                                  the button
     */
    protected AbstractButton createToolbarButton(Element element,String inheritedContexts,String inheritedType,String inheritedHideWhenDisabled) {
        String type=element.getAttribute("type");
        if (type.length()==0)
            type=inheritedType;
        AbstractButton abstractButton;
        if ("toggle".equalsIgnoreCase(type))
            abstractButton=new ToolBarToggleButton(m_smartActionManager);
        else
            abstractButton=new ToolBarButton(m_smartActionManager);
        if ("true".equals(element.getAttribute("hideWhenDisabled")))
            abstractButton.putClientProperty("hideWhenDisabled","true");
        String hideWhenDisabled=element.getAttribute("hideWhenDisabled");
        if (hideWhenDisabled.length()==0)
            hideWhenDisabled=inheritedHideWhenDisabled;
        if ("true".equalsIgnoreCase(hideWhenDisabled))
            abstractButton.putClientProperty("hideWhenDisabled","true");
        else {
            String contexts=element.getAttribute("contexts");
            if (contexts.length()==0)
                contexts=inheritedContexts;
            if (contexts!=null)
                abstractButton.putClientProperty("guibase.contexts",","+contexts+",");
        }
        return abstractButton;
    }
    /**
     * Returns <code>true</code> if the component should be hidden if it is disabled.
     *
     * @param component             the component
     * @return                      <code>true</code> if component should be hidden if it is disabled
     */
    public static boolean hideWhenDisabled(JComponent component) {
        return "true".equals(component.getClientProperty("hideWhenDisabled"));
    }
    /**
     * Checks whether the component is visible in the specified context.
     *
     * @param component             the component
     * @param contextName           the name of the context
     * @return                      <code>true</code> if the component is visible in given context
     */
    public static boolean isVisibleInContext(JComponent component,String contextName) {
        String allowedContexts=(String)component.getClientProperty("guibase.contexts");
        if (allowedContexts!=null) {
            if (contextName==null)
                return false;
            else
                return allowedContexts.indexOf(contextName)!=-1;
        }
        else
            return true;
    }

    /**
     * Button that can be placed in toolbars.
     */
    protected static class ToolBarButton extends JButton {
        /** The manager. */
        protected SmartActionManager m_smartActionManager;

        /**
         * Creates an instance of this class.
         *
         * @param smartActionManager        the manager
         */
        public ToolBarButton(SmartActionManager smartActionManager) {
            m_smartActionManager=smartActionManager;
        }
        /**
         * Configures properties from given action.
         *
         * @param action            the action
         */
        protected void configurePropertiesFromAction(Action action) {
            if (action!=null) {
                Icon icon=(Icon)action.getValue(Action.SMALL_ICON);
                this.setIcon(icon);
                if (icon!=null)
                    this.setText(null);
                else
                    this.setText((String)action.getValue(Action.NAME));
                this.setEnabled(action.isEnabled());
                this.setToolTipText((String)action.getValue(Action.SHORT_DESCRIPTION));
                this.setSelected(((SmartAction)action).isSelected());
                boolean visible=isVisibleInContext(this,m_smartActionManager.getCurrentContextName());
                if (hideWhenDisabled(this) && !action.isEnabled())
                    visible=false;
                this.setVisible(visible);
            }
        }
        /**
         * Creates a property change listener for the button.
         *
         * @param action            action for which the listener is created
         * @return                  property change listener
         */
        protected PropertyChangeListener createActionPropertyChangeListener(Action action) {
            return new ButtonActionPropertyChangeListener(this,m_smartActionManager);
        }
    }

    /**
     * Toggle button that can be placed in toolbars.
     */
    protected static class ToolBarToggleButton extends JToggleButton {
        /** The manager. */
        protected SmartActionManager m_smartActionManager;

        /**
         * Creates an instance of this class.
         *
         * @param smartActionManager        the manager
         */
        public ToolBarToggleButton(SmartActionManager smartActionManager) {
            m_smartActionManager=smartActionManager;
        }
        /**
         * Creates this toolbar tobble button.
         */
        public ToolBarToggleButton() {
            setRequestFocusEnabled(false);
            setFocusable(false);
        }
        /**
         * Creates a property change listener for the button.
         *
         * @param action            action for which the listener is created
         * @return                  property change listener
         */
        protected PropertyChangeListener createActionPropertyChangeListener(Action action) {
            return new ButtonActionPropertyChangeListener(this,m_smartActionManager);
        }
        /**
         * Configures properties from given action.
         *
         * @param action            the action
         */
        protected void configurePropertiesFromAction(Action action) {
            if (action!=null) {
                Icon icon=(Icon)action.getValue(Action.SMALL_ICON);
                this.setIcon(icon);
                if (icon!=null)
                    this.setText(null);
                else
                    this.setText((String)action.getValue(Action.NAME));
                this.setEnabled(action.isEnabled());
                this.setToolTipText((String)action.getValue(Action.SHORT_DESCRIPTION));
                this.setSelected(((SmartAction)action).isSelected());
                boolean visible=isVisibleInContext(this,m_smartActionManager.getCurrentContextName());
                if (hideWhenDisabled(this) && !action.isEnabled())
                    visible=false;
                this.setVisible(visible);
            }
        }
    }

    /**
     * Property change listener for changes to action properties.
     */
    protected static class ButtonActionPropertyChangeListener implements PropertyChangeListener {
        /** Target of this listener. */
        protected AbstractButton m_target;
        /** The smart action manager. */
        protected SmartActionManager m_smartActionManager;

        /**
         * Creates this listener and attaches it to a button.
         *
         * @param target                    the target button
         * @param smartActionManager        the smart action manager
         */
        public ButtonActionPropertyChangeListener(AbstractButton target,SmartActionManager smartActionManager) {
            m_target=target;
            m_smartActionManager=smartActionManager;
        }
        /**
         * Called when property of an action has changed.
         *
         * @param event             event descripbing the type of change
         */
        public void propertyChange(PropertyChangeEvent event) {
            String propertyName=event.getPropertyName();
            if (Action.NAME.equals(propertyName)) {
                updateText();
                m_target.invalidate();
                m_target.repaint();
            }
            else if (Action.SMALL_ICON.equals(propertyName)) {
                Icon icon=(Icon)event.getNewValue();
                m_target.setIcon(icon);
                m_target.invalidate();
                m_target.repaint();
                updateText();
            }
            else if (Action.SHORT_DESCRIPTION.equals(propertyName)) {
                String text=(String)event.getNewValue();
                m_target.setToolTipText(text);
            }
            else if ("enabled".equals(propertyName)) {
                Boolean enabledState=(Boolean)event.getNewValue();
                m_target.setEnabled(enabledState.booleanValue());
                boolean visible=isVisibleInContext(m_target,m_smartActionManager.getCurrentContextName());
                if (hideWhenDisabled(m_target) && !enabledState.booleanValue())
                    visible=false;
                m_target.setVisible(visible);
                m_target.repaint();
            }
            else if (Action.MNEMONIC_KEY.equals(propertyName)) {
                Integer mnemonic=(Integer)event.getNewValue();
                m_target.setMnemonic(mnemonic.intValue());
                m_target.invalidate();
                m_target.repaint();
            }
            else if ("selected".equals(propertyName)) {
                Boolean selectedState=(Boolean)event.getNewValue();
                m_target.setSelected(selectedState.booleanValue());
                m_target.repaint();
            }
        }
        /**
         * Updates the text of the button.
         */
        protected void updateText() {
            String text;
            if (m_target.getIcon()!=null)
                text=null;
            else
                text=(String)m_target.getAction().getValue(Action.NAME);
            m_target.setText(text);
        }
    }
}
