package de.fzi.wim.guibase.tables;

import java.awt.Point;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Component;
import javax.swing.JViewport;
import javax.swing.JLabel;
import javax.swing.CellRendererPane;

/**
 * Implements a viewport that can show text in it. When {@link SmartTable} is placed in a scroll pane it will replace
 * default viewport with an instance of this class allowing users to specify a message that needs to appear under the
 * table in the viewport. Viewport will make every attempt to center the text vertically underneath the table.
 */
public class ViewportWithText extends JViewport {
    /** Text to be displayed in the viewport. */
    protected String m_text;
    /** Label that this table uses for painting. */
    protected JLabel m_labelForPainting;
    /** Renderer pane that can be used to render label that will display given text. */
    protected CellRendererPane m_cellRendererPane;

    /**
     * Creates and initializes an instance of this class. */
    public ViewportWithText() {
        m_labelForPainting=new JLabel();
        m_labelForPainting.setHorizontalAlignment(JLabel.CENTER);
        m_cellRendererPane=new CellRendererPane();
        m_cellRendererPane.add(m_labelForPainting);
    }
    /**
     * Updates the UI of this component.
     */
    public void updateUI() {
        super.updateUI();
        updateLabelColor();
    }
    /**
     * Sets the view of this viewport.
     *
     * @param view                  view component
     */
    public void setView(Component view) {
        super.setView(view);
        updateLabelColor();
    }
    /**
     * Updates the color of the viewport label.
     */
    protected void updateLabelColor() {
        Component view=getView();
        if (view!=null)
            m_labelForPainting.setForeground(view.getForeground());
    }
    /**
     * Sets the text to be displayed in the viewport.
     *
     * @param text                    text that should be displayed in the viewport
     */
    public void setText(String text) {
        m_text=text;
        m_labelForPainting.setText(m_text);
        repaint();
    }
    /**
     * Returns the text that is displayed in the viewport.
     *
     * @return                        text that is being displayed in the viewport
     */
    public String getText() {
        return m_text;
    }
    /**
     * Paints this viewport.
     *
     * @param g                        graphics objects where viewport should paint itself
     */
    public void paintComponent(Graphics g) {
        super.paintComponent(g);
        if (m_text!=null) {
            Point viewPosition=getViewPosition();
            Dimension viewSize=getViewSize();
            int startY=viewSize.height-viewPosition.y;
            Dimension viewportSize=getSize();
            int labelHeight=viewportSize.height-startY;
            if (labelHeight>0) {
                m_labelForPainting.setFont(getFont());
                m_cellRendererPane.paintComponent(g,this.m_labelForPainting,this,0,startY+1,viewportSize.width,labelHeight-1);
            }
        }
    }
}

