package de.fzi.wim.guibase.tables;

import java.awt.Insets;
import java.awt.Color;
import java.awt.Component;
import java.awt.Graphics;
import java.awt.event.MouseEvent;
import javax.swing.border.Border;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JTable;
import javax.swing.UIManager;
import javax.swing.event.TableColumnModelEvent;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.DefaultTableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.table.TableColumnModel;

/**
 * Manipulator that can be attached to the table header and that provides sorting behavior.
 */
public class SortableTableHeader extends JTableHeader {
    /** The sortedColumn property. */
    public static final String SORTED_COLUMN_PROPERTY="sortedColumn";
    /** The sortAscending property. */
    public static final String SORT_ASCENDING_PROPERTY="sortAscending";
    /** Icon showing sort up. */
    protected static final Icon s_sortUp=new ImageIcon(SortableTableHeader.class.getResource("res/sort_up.gif"));
    /** Icon showing sort down. */
    protected static final Icon s_sortDown=new ImageIcon(SortableTableHeader.class.getResource("res/sort_down.gif"));

    /** The table column over which the mouse is currently pressed. */
    protected TableColumn m_mouseSelectedColumn;
    /** The currently sorted table column. */
    protected TableColumn m_sortedColumn;
    /** Set to <code>true</code> if column is to be sorted ascending. */
    protected boolean m_sortAscending;

    /**
     * Creates an instance of this class.
     */
    public SortableTableHeader() {
        this(null);
    }
    /**
     * Creates an instance of this class.
     *
     * @param columnModel               the column model to use
     */
    public SortableTableHeader(TableColumnModel columnModel) {
        super(columnModel);
        enableEvents(MouseEvent.MOUSE_EVENT_MASK);
    }
    /**
     * Returns the column on which the sorting is currently set.
     *
     * @return                              the column on which the sorting is currently set (<code>null</code> if no sorting is set)
     */
    public TableColumn getSortedColumn() {
        return m_sortedColumn;
    }
    /**
     * Sets the new column on which the sort is to be performed. The sort direction flag is unchanged.
     *
     * @param sortedColumn                  the column on which sorting is set
     * @param sortAscending                 determines whether column should be sorted ascending
     */
    public void setSortedColumnAndDirection(TableColumn sortedColumn,boolean sortAscending) {
        TableColumn oldSortedColumn=m_sortedColumn;
        m_sortedColumn=sortedColumn;
        firePropertyChange(SORTED_COLUMN_PROPERTY,oldSortedColumn,m_sortedColumn);
        boolean oldSortAscending=m_sortAscending;
        m_sortAscending=sortAscending;
        firePropertyChange(SORT_ASCENDING_PROPERTY,oldSortAscending,m_sortAscending);
        repaint();
    }
    /**
     * Called to simulate the click on given column. If this column is not selected, the
     *
     * @param column                        column that was clicked
     */
    public void toggleColumnSort(TableColumn column) {
        setSortedColumnAndDirection(column,getSortedColumn()==column ? !getSortAscending() : true);
    }
    /**
     * If true, currently sorted column is sorted ascending.
     *
     * @return                              <code>true</code> if currently selected column is sorted ascending
     */
    public boolean getSortAscending() {
        return m_sortAscending;
    }
    /**
     * Creates the default renderer to use.
     *
     * @return                              the default table header renderer
     */
    protected TableCellRenderer createDefaultRenderer() {
        return new TableHeaderRenderer();
    }
    /**
     * Invoked when a column is removed from the table column model.
     *
     * @param e                             the event received
     */
    public void columnRemoved(TableColumnModelEvent e) {
        if (m_sortedColumn!=null && getColumnModel().getColumnIndex(m_sortedColumn)<0)
            setSortedColumnAndDirection(null,false);
        if (m_mouseSelectedColumn!=null && getColumnModel().getColumnIndex(m_mouseSelectedColumn)<0)
            m_mouseSelectedColumn=null;
        super.columnRemoved(e);
    }
    /**
      *  Sets the header's <code>draggedDistance</code> to <code>distance</code>. This method checks if dragging was significant
      *  and in that case turns off the sorting selection.
      *
      *  @param distance                    the distance dragged
      */
    public void setDraggedDistance(int distance) {
        super.setDraggedDistance(distance);
        if (distance>5 && m_mouseSelectedColumn!=null) {
            m_mouseSelectedColumn=null;
            repaint();
        }
    }
    /**
     * Processes a mouse event that occured in the header.
     *
     * @param e                             the event received
     */
    protected void processMouseEvent(MouseEvent e) {
        super.processMouseEvent(e);
        switch (e.getID()) {
        case MouseEvent.MOUSE_PRESSED:
            if (m_mouseSelectedColumn==null) {
                int index=columnAtPoint(e.getPoint());
                if (index!=-1) {
                    m_mouseSelectedColumn=getColumnModel().getColumn(index);
                    repaint();
                }
            }
            break;
        case MouseEvent.MOUSE_RELEASED:
            if (m_mouseSelectedColumn!=null) {
                int index=columnAtPoint(e.getPoint());
                if (index!=-1) {
                    TableColumn column=getColumnModel().getColumn(index);
                    if (column==m_mouseSelectedColumn)
                        toggleColumnSort(column);
                }
                m_mouseSelectedColumn=null;
                repaint();
            }
            break;
        }
    }

    /**
     * The renderer for table header.
     */
    protected class TableHeaderRenderer extends DefaultTableCellRenderer {
        protected boolean m_drawDepressed;
        protected Border m_normalHeaderBorder;

        public TableHeaderRenderer() {
            setHorizontalTextPosition(LEADING);
            setBorder(new HeaderBorder());
        }
	    public Component getTableCellRendererComponent(JTable table,Object value,boolean isSelected, boolean hasFocus,int row,int column) {
            JTableHeader header=table.getTableHeader();
            setForeground(header.getForeground());
            TableColumn columnObject=header.getColumnModel().getColumn(column);
            m_drawDepressed=(header.getDraggedColumn()==columnObject) || (m_mouseSelectedColumn==columnObject);
            setFont(header.getFont());
            setText(value==null ? "" : value.toString());
            if (columnObject==getSortedColumn())
                setIcon(getSortAscending() ? s_sortUp : s_sortDown);
            else
                setIcon(null);
            m_normalHeaderBorder=UIManager.getBorder("TableHeader.cellBorder");
	        return this;
        }

        /**
         * Border for the header.
         */
        protected class HeaderBorder implements Border {
            protected Insets m_insets=new Insets(0,0,0,0);

            public Insets getBorderInsets(Component c) {
                if (m_normalHeaderBorder==null)
                    m_insets.top=m_insets.left=m_insets.bottom=m_insets.right=0;
                else {
                    Insets insets=m_normalHeaderBorder.getBorderInsets(c);
                    m_insets.top=insets.top+2;
                    m_insets.left=insets.left+2;
                    m_insets.bottom=insets.bottom+2;
                    m_insets.right=insets.right+2;
                }
                if (m_drawDepressed) {
                    m_insets.top+=1;
                    m_insets.bottom-=1;
                    m_insets.left+=1;
                    m_insets.right-=1;
                }
                return m_insets;
            }
            public boolean isBorderOpaque() {
                return false;
            }
            public void paintBorder(Component c,Graphics g,int x,int y,int width,int height) {
                if (m_normalHeaderBorder!=null)
                    if (m_drawDepressed) {
                        Color oldColor=g.getColor();
                        g.setColor(UIManager.getColor("controlShadow"));
                        g.drawRect(x,y,width-1,height-1);
                        g.drawRect(x+1,y+1,width-3,height-3);
                        g.setColor(oldColor);
                    }
                    else
                        m_normalHeaderBorder.paintBorder(c,g,x,y,width,height);
            }
        }
    }
}
