package de.fzi.wim.guibase.tables;

import java.util.Vector;
import java.util.EventObject;
import java.text.NumberFormat;
import java.awt.FontMetrics;
import java.awt.Color;
import java.awt.Font;
import java.awt.Component;
import java.awt.Rectangle;
import java.awt.Graphics;
import java.awt.event.ActionEvent;
import javax.swing.AbstractAction;
import javax.swing.CellEditor;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.JComboBox;
import javax.swing.JCheckBox;
import javax.swing.JPasswordField;
import javax.swing.JTextArea;
import javax.swing.ListSelectionModel;
import javax.swing.UIManager;
import javax.swing.BorderFactory;
import javax.swing.border.Border;
import javax.swing.table.TableModel;
import javax.swing.table.TableColumnModel;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;

/**
 * Fixes standard table by registeing a boolean renderer that displays the focus border around it and by registering
 * smart cell editor for common types.
 */
public class SmartTable extends JTable {

    /**
     * Creates and initializes an empty table.
     */
    public SmartTable() {
        super();
        initialize();
    }
    /**
     * Creates and initializes a table with given number of rows and columns.
     *
     * @param numRows                           number of rows in the table
     * @param numColumns                        number of columns in the table
     */
    public SmartTable(int numRows,int numColumns) {
        super(numRows,numColumns);
        initialize();
    }
    /**
     * Creates a table and initializes it with data.
     *
     * @param rowData                            an array defining values for cells at given row and column intersection
     * @param columnNames                        an array of names for the columns
     */
    public SmartTable(Object[][] rowData,Object[] columnNames) {
        super(rowData,columnNames);
        initialize();
    }
    /**
     * Creates a table and attaches it to given model.
     *
     * @param dataModel                            data model for this table
     */
    public SmartTable(TableModel dataModel) {
        super(dataModel);
        initialize();
    }
    /**
     * Creates a table and attaches it to given data and column models.
     *
     * @param dataModel                            data model for this table
     * @param columnModel                        column model for this table
     */
    public SmartTable(TableModel dataModel,TableColumnModel columnModel) {
        super(dataModel,columnModel);
        initialize();
    }
    /**
     * Creates a table and attaches it to given data model, column model and list selection model.
     *
     * @param dataModel                            data model for this table
     * @param columnModel                        column model for this table
     * @param listSelectionModel                list selection model for this table
     */
    public SmartTable(TableModel dataModel,TableColumnModel columnModel,ListSelectionModel listSelectionModel) {
        super(dataModel,columnModel,listSelectionModel);
        initialize();
    }
    /**
     * Creates a tables and attached it to given vectors for data and column names.
     *
     * @param rowData                            vector of vectors containing row data
     * @param columnNames                        vector of column names
     */
    public SmartTable(Vector rowData,Vector columnNames) {
        super(rowData,columnNames);
        initialize();
    }
    /**
     * Initializes this object. Called from each constructor.
     */
    protected void initialize() {
        // if a subclass hasn't fixed the renderer, then simply fix it now
        if (!(getDefaultRenderer(Boolean.class) instanceof BooleanRendererFixed))
            setDefaultRenderer(Boolean.class,new BooleanRendererFixed());
        getActionMap().put("startEditing",new StartEditingAction());
    }
    /**
     * Adds tool-tip renderers for some standard types.
     */
    public void addToolTipRenderers() {
        setDefaultRenderer(String.class,new ToolTipCellRenderer());
        setDefaultRenderer(Number.class,new NumberRenderer());
        DoubleRenderer renderer=new DoubleRenderer();
        setDefaultRenderer(Float.class,renderer);
        setDefaultRenderer(Double.class,renderer);
    }
    /**
     * Returns the effective rectangle of the cell at given path and column.
     *
     * @param row                               row of the cell
     * @param column                            column of the cell
     * @param includeSpacing                    if <code>false</code>, return the true cell bounds - computed by subtracting the intercell spacing
     *                                          from the height and widths of the column and row models
     * @return                                  effective cell rectangle at given position
     */
    public Rectangle getEffectiveCellRect(int row,int column,boolean includeSpacing) {
        return getCellRect(row,column,includeSpacing);
    }
    /**
     * Overrides default Swing implementation to set focus to the editor component after table enters edit mode.
     *
     * @param row                                row of the cell being edited
     * @param column                            column of cell being edited
     * @param e                                    event that triggered editing
     * @return                                    <code>true</code> if editing was started successfully
     */
    public boolean editCellAt(int row,int column,EventObject e) {
        if (super.editCellAt(row,column,e)) {
            Component component=getEditorComponent();
            if (component!=null)
                if (component instanceof JComboBox) {
                    JComboBox comboBox=(JComboBox)component;
                    if (comboBox.getEditor()!=null)
                        comboBox.getEditor().getEditorComponent().requestFocus();
                    else
                        comboBox.requestFocus();
                }
                else
                    component.requestFocus();
            return true;
        }
        else
            return false;
    }
    /**
     * Discards the editor object and frees the real estate it used for cell rendering. This method is overridden to prevent
     * the table from gaining the input focus when editing is stopped. It is the cell editor who should request focus to
     * the table.
     */
    public void removeEditor() {
        TableCellEditor editor=getCellEditor();
        if (editor!=null) {
            editor.removeCellEditorListener(this);
            if (!(editorComp instanceof JComponent) || ((JComponent)editorComp).getClientProperty("smartEditor")==null)
                requestFocus();
            if (editorComp!=null)
                remove(editorComp);
            Rectangle cellRect=getCellRect(editingRow,editingColumn,false);
            setCellEditor(null);
            setEditingColumn(-1);
            setEditingRow(-1);
            editorComp=null;
            repaint(cellRect);
        }
    }

    /**
     * Action that is executed to start editing in the table. This class replaces default Swing implementation in order not
     * to set focus to the editor component. It is <code>SmartTable</code> that does this.
     */
    protected static class StartEditingAction extends AbstractAction {
        /**
         * Called when action is invoked.
         *
         * @param e                                Swing event object
         */
        public void actionPerformed(ActionEvent e) {
            JTable table=(JTable)e.getSource();
            if (!table.hasFocus()) {
                CellEditor cellEditor=table.getCellEditor();
                if (cellEditor!=null && !cellEditor.stopCellEditing())
                    return;
                table.requestFocus();
                return;
            }
            ListSelectionModel rowSelectionModel=table.getSelectionModel();
            int anchorRow=rowSelectionModel.getAnchorSelectionIndex();
            ListSelectionModel columnSelectionModel=table.getColumnModel().getSelectionModel();
            int anchorColumn=columnSelectionModel.getAnchorSelectionIndex();
            table.editCellAt(anchorRow,anchorColumn);
        }
    }

    /**
     * A renderer class that fxies Swing's problems with rendering boolean values. It adds a selection border if cell is selected.
     */
    static protected class BooleanRendererFixed extends JCheckBox implements TableCellRenderer {
        /** A staric instance of a border that is displayed when an item doesn't have selection. */
        protected static final Border s_noFocusBorder=BorderFactory.createEmptyBorder(1,1,1,1);

        /**
         * Creates and initializes an instance of this class.
         */
        public BooleanRendererFixed() {
            super("");
            setHorizontalAlignment(JCheckBox.CENTER);
            setBorder(s_noFocusBorder);
            setBorderPainted(true);
            setBorderPaintedFlat(false);
        }
        /**
         * Called to get renderer component.
         *
         * @param table                            table where renderer is used
         * @param value                            value to be rendered
         * @param isSelected                    <code>true</code> if cell should be painted as selected
         * @param hasFocus                        <code>true</code> if cell should be painted with focus
         * @param row                            row for the cell
         * @param column                        column for the cell
         * @return                                component used for painting a cell
         */
        public Component getTableCellRendererComponent(JTable table,Object value,boolean isSelected,boolean hasFocus,int row,int column) {
            setEnabled(table.isEnabled());
            setSelected((value!=null && ((Boolean)value).booleanValue()));
            if (isSelected) {
                if (hasFocus && table.isCellEditable(row,column)) {
                    setForeground(UIManager.getColor("Table.focusCellForeground"));
                    setBackground(UIManager.getColor("Table.focusCellBackground"));
                }
                else {
                    setForeground(table.getSelectionForeground());
                    setBackground(table.getSelectionBackground());
                }
            }
            else {
                setForeground(table.getForeground());
                setBackground(table.getBackground());
            }
            if (hasFocus)
                setBorder(UIManager.getBorder("Table.focusCellHighlightBorder"));
            else
                setBorder(s_noFocusBorder);
            return this;
        }
    }

    /**
     * Cell renderer that implements tooltips.
     */
    public static class ToolTipCellRenderer extends JLabel implements TableCellRenderer {
        /** Border for cells with no focus. */
        protected static final Border s_noFocusBorder=BorderFactory.createEmptyBorder(1,1,1,1);

        /**
         * Creates this renderer.
         */
        public ToolTipCellRenderer() {
            setOpaque(true);
        }
        /**
         * Returns the default table cell renderer.
         *
         * @param table             the <code>JTable</code>
         * @param value             the value to assign to the cell at <code>[row, column]</code>
         * @param isSelected        <code>true</code> if cell is selected
         * @param hasFocus          <code>true</code> if cell has focus
         * @param row               the row of the cell to render
         * @param column            the column of the cell to render
         * @return                  the default table cell renderer
         */
        public Component getTableCellRendererComponent(JTable table,Object value,boolean isSelected,boolean hasFocus,int row,int column) {
            setEnabled(table.isEnabled());
            setForeground(getCellForeground(table,value,isSelected,hasFocus,row,column));
            setBackground(getCellBackground(table,value,isSelected,hasFocus,row,column));
            setFont(getCellFont(table,value,isSelected,hasFocus,row,column));
            setBorder(getCellBorder(table,value,isSelected,hasFocus,row,column));
            setValue(table,value,isSelected,hasFocus,row,column);
            updateToolTip(table,value,row,column);
            // optimization for avoinding painting of cell background
            Color back=getBackground();
            boolean colorMatch=(back!=null) && (back.equals(table.getBackground())) && table.isOpaque();
            setOpaque(!colorMatch);
            return this;
        }
        /**
         * Returns desired foreground color of a cell.
         */
        protected Color getCellForeground(JTable table,Object value,boolean isSelected,boolean hasFocus,int row,int column) {
            if (isSelected)
               return table.getSelectionForeground();
            else if (hasFocus && table.isCellEditable(row,column))
                return UIManager.getColor("Table.focusCellForeground");
            else
                return table.getForeground();
        }
        /**
         * Returns desired background color of a cell.
         */
        protected Color getCellBackground(JTable table,Object value,boolean isSelected,boolean hasFocus,int row,int column) {
            if (isSelected)
               return table.getSelectionBackground();
            else if (hasFocus && table.isCellEditable(row,column))
                return UIManager.getColor("Table.focusCellBackground");
            else
                return table.getBackground();
        }
        /**
         * Returns desired cell border.
         */
        protected Border getCellBorder(JTable table,Object value,boolean isSelected,boolean hasFocus,int row,int column) {
            if (hasFocus)
                return UIManager.getBorder("Table.focusCellHighlightBorder");
            else
                return s_noFocusBorder;
        }
        /**
         * Returns desired cell font.
         */
        protected Font getCellFont(JTable table,Object value,boolean isSelected,boolean hasFocus,int row,int column) {
            return table.getFont();
        }
        /**
         * Updates the tooltip of this renderer.
         *
         * @param table             table in which renderer is being used
         * @param value             value being rendered
         * @param row               row being rendered
         * @param column            column being rendered
         */
        protected void updateToolTip(JTable table,Object value,int row,int column) {
            String cellText=getCellText(value);
            if (cellText==null)
                setToolTipText(null);
            else {
                Graphics g=table.getGraphics();
                g.setFont(getFont());
                FontMetrics fontMetrics=g.getFontMetrics();
                if (fontMetrics.stringWidth(cellText)>getAvailableTextWidth(table,row,column))
                    setToolTipText(cellText);
                else
                    setToolTipText(null);
                g.dispose();
            }
        }
        /**
         * Sets the value of this renderer.
         */
        protected void setValue(JTable table,Object value,boolean isSelected,boolean hasFocus,int row,int column) {
            setText(getCellText(value));
        }
        /**
         * Returns the effective width of the text part of the cell.
         *
         * @param table         table for which width is computed
         * @param row           row
         * @param column        column
         */
        protected int getAvailableTextWidth(JTable table,int row,int column) {
            return ((SmartTable)table).getEffectiveCellRect(row,column,false).width;
        }
        /**
         * Returns the text that is displayed in the control.
         *
         * @param value         value being rendered
         */
        protected String getCellText(Object value) {
            return value==null ? null : value.toString();
        }
        /**
         * Overridden for performance reasons.
         */
        public void validate() {
        }
        /**
         * Overridden for performance reasons.
         */
        public void revalidate() {
        }
        /**
         * Overridden for performance reasons.
         */
        public void repaint(long tm,int x,int y,int width,int height) {
        }
        /**
         * Overridden for performance reasons.
         */
        public void repaint(Rectangle r) {
        }
    }

    /**
     * Renderer for numbers.
     */
    public static class NumberRenderer extends ToolTipCellRenderer {
        /**
         * Creates a number renderer.
         */
        public NumberRenderer() {
            super();
            setHorizontalAlignment(JLabel.RIGHT);
        }
    }

    /**
     * Renderer for double values.
     */
    public static class DoubleRenderer extends NumberRenderer {
        /** Formatter for double numbers. */
        protected NumberFormat m_formatter;

        /**
         * Retruns a text representation of given value.
         *
         * @param value         value being rendered
         */
        public String getCellText(Object value) {
            if (m_formatter==null)
                m_formatter=NumberFormat.getInstance();
            return (value == null) ? null : m_formatter.format(value);
        }
    }

    /**
     * Custom renderer for the password field in the table
     */
    public static class PasswordRenderer extends JPasswordField implements TableCellRenderer {
        /** Border for cells with no focus. */
        protected Border s_noFocusBorder=BorderFactory.createEmptyBorder(1,1,1,1);

        public PasswordRenderer() {
            super();
        }
        public Component getTableCellRendererComponent(JTable table,Object value,boolean isSelected,boolean hasFocus,int row,int column) {
            // set the text
            setText(value.toString());
            // adjust the properties to the table properties
            setEnabled(table.isEnabled());
            setForeground(getCellForeground(table,value,isSelected,hasFocus,row,column));
            setBackground(getCellBackground(table,value,isSelected,hasFocus,row,column));
            setFont(getCellFont(table,value,isSelected,hasFocus,row,column));
            setBorder(getCellBorder(table,value,isSelected,hasFocus,row,column));
            // optimization for avoinding painting of cell background
            Color back=getBackground();
            boolean colorMatch=(back!=null) && (back.equals(table.getBackground())) && table.isOpaque();
            setOpaque(!colorMatch);
            return this;
        }
        /**
         * Returns desired foreground color of a cell.
         */
        protected Color getCellForeground(JTable table,Object value,boolean isSelected,boolean hasFocus,int row,int column) {
            if (isSelected)
               return table.getSelectionForeground();
            else if (hasFocus && table.isCellEditable(row,column))
                return UIManager.getColor("Table.focusCellForeground");
            else
                return table.getForeground();
        }
        /**
         * Returns desired background color of a cell.
         */
        protected Color getCellBackground(JTable table,Object value,boolean isSelected,boolean hasFocus,int row,int column) {
            if (isSelected)
               return table.getSelectionBackground();
            else if (hasFocus && table.isCellEditable(row,column))
                return UIManager.getColor("Table.focusCellBackground");
            else
                return table.getBackground();
        }
        /**
         * Returns desired cell border.
         */
        protected Border getCellBorder(JTable table,Object value,boolean isSelected,boolean hasFocus,int row,int column) {
            if (hasFocus)
                return UIManager.getBorder("Table.focusCellHighlightBorder");
            else
                return s_noFocusBorder;
        }
        /**
         * Returns desired cell font.
         */
        protected Font getCellFont(JTable table,Object value,boolean isSelected,boolean hasFocus,int row,int column) {
            return table.getFont();
        }
    }

    /**
     * The renderer showing multiple lines of the text.
     */
    public static class MultiLineCellRenderer extends JTextArea implements TableCellRenderer {
        /** Border for cells with no focus. */
        protected Border s_noFocusBorder=BorderFactory.createEmptyBorder(1,1,1,1);

        public MultiLineCellRenderer() {
            setLineWrap(false);
            setWrapStyleWord(true);
            setOpaque(true);
        }

        public Component getTableCellRendererComponent(JTable table,Object value,boolean isSelected,boolean hasFocus,int row,int column) {
            // set the text
            setText((value==null) ? "" : value.toString());
            // adjust the properties to the table properties
            setEnabled(table.isEnabled());
            setForeground(getCellForeground(table,value,isSelected,hasFocus,row,column));
            setBackground(getCellBackground(table,value,isSelected,hasFocus,row,column));
            setFont(getCellFont(table,value,isSelected,hasFocus,row,column));
            setBorder(getCellBorder(table,value,isSelected,hasFocus,row,column));
            // optimization for avoinding painting of cell background
            Color back=getBackground();
            boolean colorMatch=(back!=null) && (back.equals(table.getBackground())) && table.isOpaque();
            setOpaque(!colorMatch);
            return this;
        }
        /**
         * Returns desired foreground color of a cell.
         */
        protected Color getCellForeground(JTable table,Object value,boolean isSelected,boolean hasFocus,int row,int column) {
            if (isSelected)
               return table.getSelectionForeground();
            else if (hasFocus && table.isCellEditable(row,column))
                return UIManager.getColor("Table.focusCellForeground");
            else
                return table.getForeground();
        }
        /**
         * Returns desired background color of a cell.
         */
        protected Color getCellBackground(JTable table,Object value,boolean isSelected,boolean hasFocus,int row,int column) {
            if (isSelected)
               return table.getSelectionBackground();
            else if (hasFocus && table.isCellEditable(row,column))
                return UIManager.getColor("Table.focusCellBackground");
            else
                return table.getBackground();
        }
        /**
         * Returns desired cell border.
         */
        protected Border getCellBorder(JTable table,Object value,boolean isSelected,boolean hasFocus,int row,int column) {
            if (hasFocus)
                return UIManager.getBorder("Table.focusCellHighlightBorder");
            else
                return s_noFocusBorder;
        }
        /**
         * Returns desired cell font.
         */
        protected Font getCellFont(JTable table,Object value,boolean isSelected,boolean hasFocus,int row,int column) {
            return table.getFont();
        }
    }
}
