package de.fzi.wim.guibase.tables;

import java.util.EventObject;
import java.awt.Color;
import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.MouseEvent;
import javax.swing.JComponent;
import javax.swing.AbstractCellEditor;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.ComboBoxEditor;
import javax.swing.ComboBoxModel;
import javax.swing.JTextField;
import javax.swing.JTable;
import javax.swing.BorderFactory;
import javax.swing.table.TableCellEditor;

/**
 * Extends standard Swing's cell editor to fix some of the important problems. Here is the list of problems fixed:
 * <ul>
 *  <li>editor stops editing when component loses focus,
 *  <li>editor recognises the escape key and cancels editing,
 *  <li>when cell editing is stopped, editor can select next editable cell in the table,
 * </ul>
 */
public class SmartCellEditor extends AbstractCellEditor implements FocusListener,KeyListener,ActionListener,TableCellEditor {
    /** Component for editing a value. */
    protected JComponent m_component;
    /** Number of clicks to start editing. */
    protected int m_clickCountToStart;
    /** Flag set to <code>true</code> whenever cell editor is stopping the edit because control lost focus. */
    protected boolean m_controlLostFocus;
    /** Flag indicating whether next cell should be selected when editing is stopped. */
    protected boolean m_onInputSelectNextCell=false;
    /** Reference to the table that this editor has last been used in. */
    protected JTable m_lastTable;
    /** Flag used to stop recursive event invocations. */
    protected boolean m_disableEvents=false;
    /** Set to <code>1</code> if next cell should be selected when editing is stopped, <code>-1</code> if previous cells is to be selected or to <code>0</code> if no cell should be selected. */
    protected int m_moveToNextCell;
    /** Deletegate that regulates how to interact with the control. */
    protected EditorDelegate m_delegate;

    /**
     * Creates a cell editor for a check box.
     *
     * @param checkBox                            check box that will be used for editing a value
     */
    public SmartCellEditor(final JCheckBox checkBox) {
        checkBox.setHorizontalAlignment(JCheckBox.CENTER);
        m_clickCountToStart=1;
        m_component=checkBox;
        m_delegate=new EditorDelegate() {
            public void setValue(Object value) {
                boolean selected=false;
                if (value instanceof Boolean)
                    selected=((Boolean)value).booleanValue();
                else if (value instanceof String)
                    selected=value.equals("true");
                checkBox.setSelected(selected);
            }
            public Object getValue() {
                return new Boolean(checkBox.isSelected());
            }
        };
        checkBox.putClientProperty("smartEditor","true");
        checkBox.addActionListener(this);
        checkBox.addFocusListener(this);
        checkBox.addKeyListener(this);
        checkBox.setBorder(BorderFactory.createLineBorder(Color.black));
    }
    /**
     * Creates a cell editor for a combo box.
     *
     * @param comboBox                            combo box that will be used for editing a value
     */
    public SmartCellEditor(final JComboBox comboBox) {
        m_clickCountToStart=1;
        m_component=comboBox;
        comboBox.putClientProperty("JComboBox.lightweightKeyboardNavigation","Lightweight");
        m_delegate=new EditorDelegate() {
            public void setValue(Object value) {
                comboBox.setSelectedItem(value);
            }
            public Object getValue() {
                return comboBox.getSelectedItem();
            }
            public boolean shouldSelectCell(EventObject anEvent) {
                if (anEvent instanceof MouseEvent) {
                    MouseEvent e=(MouseEvent)anEvent;
                    return e.getID()!=MouseEvent.MOUSE_DRAGGED;
                }
                return true;
            }
        };
        comboBox.putClientProperty("smartEditor","true");
        comboBox.addActionListener(this);
        comboBox.addFocusListener(this);
        comboBox.addKeyListener(this);
        comboBox.setBorder(BorderFactory.createLineBorder(Color.black));
        if (comboBox.getEditor()!=null && (comboBox.getEditor().getEditorComponent() instanceof JComponent)) {
            JComponent component=(JComponent)comboBox.getEditor().getEditorComponent();
            component.addFocusListener(this);
            component.addKeyListener(this);
        }
    }
    /**
     * Creates a cell editor for a text field.
     *
     * @param textField                             text field that will be used for editing a value
     */
    public SmartCellEditor(final JTextField textField) {
        m_component=textField;
        m_clickCountToStart=2;
        m_delegate=new EditorDelegate() {
            public void setValue(Object value) {
                textField.setText((value!=null) ? value.toString() : "");
            }
            public Object getValue() {
                return textField.getText();
            }
            public void focusGained() {
                textField.selectAll();
            }
        };
        textField.putClientProperty("smartEditor","true");
        textField.addFocusListener(this);
        textField.addKeyListener(this);
        textField.setBorder(BorderFactory.createLineBorder(Color.black));
    }
    /**
     * Creates a cell editor for any component.
     *
     * @param component                             component that will be used for editing a value
     */
    public SmartCellEditor(JComponent component) {
        m_clickCountToStart=1;
        m_component=component;
        m_delegate=new EditorDelegate() {
            public void setValue(Object value) {
            }
            public Object getValue() {
                return null;
            }
        };
        component.putClientProperty("smartEditor","true");
        component.addFocusListener(this);
        component.addKeyListener(this);
    }
    /**
     * Creates any editor.
     */
    protected SmartCellEditor() {
    }
    /**
     * Specifies the number of clicks needed to start editing.
     *
     * @param count                                 an int specifying the number of clicks needed to start editing
     * @see #getClickCountToStart
     */
    public void setClickCountToStart(int count) {
        m_clickCountToStart=count;
    }
    /**
     *  Returns number of clicks required to start editing.
     *
     * @return                                      number of clicks needed to start editing
     */
    public int getClickCountToStart() {
        return m_clickCountToStart;
    }
    /**
     * An implementation side-effect since this class implements <code>FocusListner</code> interface.
     *
     * @param e                                     Swing event
     */
    public void focusGained(FocusEvent e) {
        m_delegate.focusGained();
    }
    /**
     * An implementation side-effect since this class implements <code>FocusListner</code> interface. This method
     * will stop editing when control loses focus.
     *
     * @param e                                     Swing event
     */
    public void focusLost(FocusEvent e) {
        if (!m_disableEvents) {
            m_controlLostFocus=true;
            stopCellEditing();
            m_controlLostFocus=false;
        }
    }
    /**
     * An implementation side-effect since this class implements <code>KeyListener</code> interface. If pressed key is the
     * Escape key, cell editing is canceled.
     *
     * @param e                                     Swing event
     */
    public void keyPressed(KeyEvent e) {
        if (e.getKeyCode()==KeyEvent.VK_ESCAPE) {
            cancelCellEditing();
            e.consume();
        }
    }
    /**
     * An implementation side-effect since this class implements <code>KeyListener</code> interface.
     *
     * @param e                                     Swing event
     */
    public void keyReleased(KeyEvent e) {
    }
    /**
     * An implementation side-effect since this class implements <code>KeyListener</code> interface.
     *
     * @param e                                     Swing event
     */
    public void keyTyped(KeyEvent e) {
    }
    /**
     * This method is an implementation side-effect since this class implements <code>ActionListener</code> interface.
     *
     * @param e                                     Swing event
     */
    public void actionPerformed(ActionEvent e) {
        if (!m_disableEvents) {
            m_moveToNextCell=(m_component instanceof JComboBox) ? 1 : 0;
            stopCellEditing();
        }
    }
    /**
     * Returns the value of this cell editor.
     *
     * @return                                      value of this cell editor
     */
    public Object getCellEditorValue() {
        return m_delegate.getValue();
    }
    /**
     * Returns <code>true</code> if this cell is editable.
     *
     * @param anEvent                               the event
     * @return                                      <code>true</code> if this cell is editable
     */
    public boolean isCellEditable(EventObject anEvent) {
        if (anEvent instanceof MouseEvent)
            return ((MouseEvent)anEvent).getClickCount()>=m_clickCountToStart;
        return true;
    }
    /**
     * Returns <code>true</code> if this cell should be selected.
     *
     * @param anEvent                               the event
     * @return                                      <code>true</code> if this cell should be selected
     */
    public boolean shouldSelectCell(EventObject anEvent) {
        return m_delegate.shouldSelectCell(anEvent);
    }
    /**
     * Determines whether next cell should be selected when editing is stopped.
     *
     * @param onInputSelectNextCell                 <code>true</code> if next cell should be selected when editing is stopped
     */
    public void onInputSelectNextCell(boolean onInputSelectNextCell) {
        m_onInputSelectNextCell=onInputSelectNextCell;
    }
    /**
     * Called to get editor component. This method prepared the editor by getting the object from the data source,
     * setting it into table's instance manager and reloading appropriate control.
     *
     * @param table                                 table where renderer is used
     * @param value                                 value to be rendered
     * @param isSelected                            <code>true</code> if cell should be edited as selected
     * @param row                                   row for the cell
     * @param column                                column for the cell
     * @return                                      component used for editing a cell
     */
    public Component getTableCellEditorComponent(JTable table,Object value,boolean isSelected,int row,int column) {
        m_disableEvents=true;
        m_lastTable=table;
        m_moveToNextCell=0;
        m_delegate.setValue(value);
        m_disableEvents=false;
        return m_component;
    }
    /**
     * Called to stop cell editing. This method will select next cell in the associated table if needed. This method fixes
     * several issues of Swing event ordering so it should NOT be overriden. Subclasses should override
     * {@link #stopCellEditingInternal()} method to provide their own stopping mechanisms.
     *
     * @return                                      <code>true</code> if editing was successfully stopped
     */
    public boolean stopCellEditing() {
        if (m_lastTable==null || !m_lastTable.isEditing() || m_disableEvents)
            return false;
        try {
            m_disableEvents=true;
            // There is a problem with event ordering in the combo. Our focus listener gets executed before internal focus listener
            // in Swing executes and updates the current item in the combo. Therefore we need to make sure we execute this ourselves.
            if (m_component instanceof JComboBox) {
                JComboBox comboBox=(JComboBox)m_component;
                ComboBoxEditor editor=comboBox.getEditor();
                if (comboBox.isEditable() && editor!=null) {
                    Object item=editor.getItem();
                    ComboBoxModel model=comboBox.getModel();
                    if (item!=null && !item.equals(model.getSelectedItem()))
                        model.setSelectedItem(item);
                }
            }
            // invoke sub-class' stop editing method
            int editingRow=m_lastTable.getEditingRow();
            int editingColumn=m_lastTable.getEditingColumn();
            if (!stopCellEditingInternal())
                return false;
            fireEditingStopped();
            if (!m_controlLostFocus)
                m_lastTable.requestFocus();
            if (m_onInputSelectNextCell && m_moveToNextCell!=0 && !m_controlLostFocus)
                selectNextEditableCell(editingRow,editingColumn,m_moveToNextCell>0);
            m_lastTable=null;
        }
        finally {
            m_disableEvents=false;
        }
        return true;
    }
    /**
     * Subclasses should override this method to provice their own cell editing processing. This method is guarranteed not to
     * execute recursively because of an 'event storm'.
     *
     * @return                                      <code>true</code> is editing should be stopped
     */
    protected boolean stopCellEditingInternal() {
        return true;
    }
    /**
     * Called to cancel cell editing.
     */
    public void cancelCellEditing() {
        if (m_lastTable!=null && !m_disableEvents)
            try {
                m_disableEvents=true;
                fireEditingCanceled();
            }
            finally {
                m_disableEvents=false;
                m_lastTable=null;
            }
    }
    /**
     * Utility method that will locate and select next (previous) editable cell in the last table.
     *
     * @param row                                   row of the starting cell
     * @param column                                column of the starting cell
     * @param moveNext                              <code>true</code> if next cell is requested (if <code>false</code> previous cell will be selected
     */
    protected void selectNextEditableCell(int row,int column,boolean moveNext) {
        if (moveNext)
            column++;
        else
            column--;
        int numberOfRows=m_lastTable.getRowCount();
        int numberOfColumns=m_lastTable.getColumnCount();
        while (0<=row && row<numberOfRows) {
            while (0<=column && column<numberOfColumns) {
                if (m_lastTable.isCellEditable(row,column)) {
                    m_lastTable.changeSelection(row,column,false,false);
                    return;
                }
                if (moveNext)
                    column++;
                else
                    column--;
            }
            column=0;
            if (moveNext)
                row++;
            else
                row--;
        }
    }

    /**
     * Utility class that implements behavior that differs across components.
     */
    protected abstract class EditorDelegate {

        public abstract Object getValue();
        public abstract void setValue(Object value);
        public void focusGained() {
        }
        public boolean shouldSelectCell(EventObject anEvent) {
            return true;
        }
    }
}
