package de.fzi.wim.guibase.sidebar;

import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.Insets;
import java.awt.Dimension;
import java.awt.Component;
import java.awt.Container;
import java.awt.LayoutManager;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JLayeredPane;

/**
 * A component looking similar to the "outlookbar" from MS-Outlook.
 */
public class JSideBar extends JPanel {
    /** The default icon for the remove button. */
    protected static final Icon s_defaultRemoveIcon=new ImageIcon(JSideBar.class.getResource("images/delete.gif"));
    /** The margin from the inner button to the outer one. */
    protected static final int INNER_BUTTON_MARGIN=5;

    /** The remove icon. */
    protected Icon m_removeIcon;
    /** The text for the remove button. */
    protected String m_removeText;
    /** The index of the currently selected item. */
    protected int m_selectedIndex;
    /** The handler for item removals. */
    protected ActionListener m_itemRemovedListener;
    /** The  handled for item selection. */
    protected ActionListener m_itemSelectedListener;
    /** The layout manager for this component. */
    protected GridBagLayout m_layout;

    /**
     * Creates a new instance of SideBar.
     */
    public JSideBar() {
        super(null);
        m_layout=new GridBagLayout();
        setLayout(m_layout);
        m_removeIcon=s_defaultRemoveIcon;
        m_selectedIndex=-1;
        m_itemRemovedListener=new ActionListener() {
            public void actionPerformed(ActionEvent event) {
                for (int i=0;i<getItemCount();i++) {
                    TitlePanel titlePanel=getTitlePanel(i);
                    if (titlePanel.m_removeButton==event.getSource()) {
                        removeItem(i);
                        break;
                    }
                }
            }
        };
        m_itemSelectedListener=new ActionListener() {
            public void actionPerformed(ActionEvent event) {
                for (int i=0;i<getItemCount();i++) {
                    TitlePanel titlePanel=getTitlePanel(i);
                    if (titlePanel.m_selectButton==event.getSource()) {
                        setSelectedIndex(i);
                        break;
                    }
                }
            }
        };
    }
    /**
     * Returns the remove icon.
     *
     * @return                      the remove icon
     */
    public Icon getRemoveIcon() {
        return m_removeIcon;
    }
    /**
     * Sets the icon for the remove button.
     *
     * @param icon                  the icon for the remove button
     */
    public void setRemoveIcon(Icon icon) {
        Icon oldRemoveIcon=m_removeIcon;
        m_removeIcon=icon;
        for (int i=0;i<getItemCount();i++) {
            TitlePanel titlePanel=getTitlePanel(i);
            titlePanel.m_removeButton.setIcon(m_removeIcon);
        }
        firePropertyChange("removeIcon",oldRemoveIcon,m_removeIcon);
    }
    /**
     * Returns the text for the removal button.
     *
     * @return                      the remove button text
     */
    public String getRemoveText() {
        return m_removeText;
    }
    /**
     * Sets the rexr for the remove button.
     *
     * @param removeText            the text for the remove button
     */
    public void setRemoveText(String removeText) {
        String oldRemoveText=m_removeText;
        m_removeText=removeText;
        for (int i=0;i<getItemCount();i++) {
            TitlePanel titlePanel=getTitlePanel(i);
            titlePanel.m_removeButton.setToolTipText(m_removeText);
        }
        firePropertyChange("removeText",oldRemoveText,m_removeText);
    }
    /**
     * Add a new item to the <code>SideBar</code>.
     *
     * @param label                 label of the button for this item, may be <code>null</code>
     * @param icon                  the icon for the item, may be <code>null</code>
     * @param description           the description text for the item
     * @param component             component to display when the button is pressed
     * @param removeButton          <code>true</code> if the remove button should be activated
     */
    public void addItem(String label,Icon icon,String description,Component component,boolean removeButton) {
        insertItem(getItemCount(),label,icon,description,component,removeButton);
    }
    /**
     * Insert a new item to the <code>SideBar</code>.
     *
     * @param index                 index of the position where the item is to be inserted
     * @param label                 label of the button for this item, may be <code>null</code>
     * @param icon                  the icon for the item, may be <code>null</code>
     * @param description           the description text for the item
     * @param component             component to display when the button is pressed
     * @param removeButton          <code>true</code> if the remove button should be activated
     */
    public void insertItem(int index,String label,Icon icon,String description,Component component,boolean removeButton) {
        component.setVisible(false);
        TitlePanel titlePanel=new TitlePanel(m_itemSelectedListener,m_itemRemovedListener);
        titlePanel.m_selectButton.setText(label);
        titlePanel.m_selectButton.setIcon(icon);
        titlePanel.m_selectButton.setToolTipText(description);
        titlePanel.m_removeButton.setIcon(m_removeIcon);
        titlePanel.m_removeButton.setToolTipText(m_removeText);
        titlePanel.setRemovalButtonState(removeButton);
        add(titlePanel,index*2);
        add(component,index*2+1);
        updateLayoutConstraints();
        if (m_selectedIndex>=index)
            setSelectedIndex(m_selectedIndex+1);
        else if (m_selectedIndex<0)
            setSelectedIndex(0);
    }
    /**
     * Removes the item with the supplied index.
     *
     * @param index                         index of the item to be removed
     */
    public void removeItem(int index) {
        remove(2*index+1);
        remove(2*index);
        updateLayoutConstraints();
        if (index<m_selectedIndex)
            setSelectedIndex(m_selectedIndex-1);
        else if (index==m_selectedIndex) {
            if (m_selectedIndex>0)
                setSelectedIndex(m_selectedIndex-1);
            else if (getItemCount()>0)
                setSelectedIndex(0);
            else
                setSelectedIndex(-1);
        }
    }
    /**
     * Returns the index to the item that contains the specified component.
     *
     * @param component             the item to search for
     * @return                      index of the found item, <code>-1</code> if the component could not be found
     */
    public int findComponentIndex(Component component) {
        for (int i=0;i<getItemCount();i++)
            if (getItemComponent(i)==component)
                return i;
        return -1;
    }
    /**
     * Returns the component at specified index.
     *
     * @param index                 the index of the component
     * @return                      the component with given index
     */
    public Component getItemComponent(int index) {
        return getComponent(index*2+1);
    }
    /**
     * Selects the item with the supplied index.
     *
     * @param index                 index of the item to be selected
     */
    public void setSelectedIndex(int index) {
        if (m_selectedIndex!=index) {
            int oldSelectedIndex=m_selectedIndex;
            m_selectedIndex=index;
            if (oldSelectedIndex>=0 && oldSelectedIndex*2+1<getComponentCount())
                getItemComponent(oldSelectedIndex).setVisible(false);
            if (m_selectedIndex>=0 && m_selectedIndex*2+1<getComponentCount())
                getItemComponent(m_selectedIndex).setVisible(true);
            updateLayoutConstraints();
            firePropertyChange("selectedIndex",oldSelectedIndex,m_selectedIndex);
        }
    }
    /**
     * Returns the index of the selected item.
     *
     * @return                      the indx of the currently selected item
     */
    public int getSelectedIndex() {
        return m_selectedIndex;
    }
    /**
     * Returns the number of items currently added to the sidebar.
     *
     * @return      number of items
     */
    public int getItemCount() {
        return getComponentCount()/2;
    }
    /**
     * Returns the TitlePanel for item with given index.
     *
     * @param index                 the index of the item
     * @return                      the TitlePanel of the item
     */
    protected TitlePanel getTitlePanel(int index) {
        return (TitlePanel)getComponent(index*2);
    }
    /**
     * Returns the index of the item where the point points to.
     *
     * @param x                     the X coordinate
     * @param y                     the Y coordinate
     * @return                      the index of the item
     */
    public int getItemIndexAtPoint(int x,int y) {
        for (int i=0;i<getItemCount();i++) {
            if (getTitlePanel(i).contains(x,y))
                return i;
            if (getItemComponent(i).contains(x,y))
                return i;
        }
        return -1;
    }
    /**
     * Sets the label for given item.
     *
     * @param index                 the index of the item
     * @param label                 the label of the item
     */
    public void setItemLabel(int index,String label) {
        getTitlePanel(index).m_selectButton.setText(label);
    }
    /**
     * Sets the description for given item.
     *
     * @param index                 the index of the item
     * @param description           the description of the item
     */
    public void setItemDescription(int index,String description) {
        getTitlePanel(index).m_selectButton.setToolTipText(description);
    }
    /**
     * Sets the icon for given item.
     *
     * @param index                 the index of the item
     * @param icon                  the icon of the item
     */
    public void setItemIcon(int index,Icon icon) {
        getTitlePanel(index).m_selectButton.setIcon(icon);
    }
    /**
     * Updates the constraints of all elements in the container.
     */
    protected void updateLayoutConstraints() {
        for (int i=0;i<getItemCount();i++) {
            m_layout.setConstraints(getTitlePanel(i),new GridBagConstraints(0,i*2,1,1,1.0,0.0,GridBagConstraints.CENTER,GridBagConstraints.HORIZONTAL,new Insets(0,0,0,0),0,0));
            if (i==getSelectedIndex())
                m_layout.setConstraints(getItemComponent(i),new GridBagConstraints(0,i*2+1,1,1,1.0,1.0,GridBagConstraints.CENTER,GridBagConstraints.BOTH,new Insets(0,0,0,0),0,0));
            else
                m_layout.setConstraints(getItemComponent(i),new GridBagConstraints(0,i*2+1,1,1,1.0,0.0,GridBagConstraints.CENTER,GridBagConstraints.HORIZONTAL,new Insets(0,0,0,0),0,0));
        }
        revalidate();
    }

    /**
     * Button of each item.
     */
    protected static class TitlePanel extends JLayeredPane implements LayoutManager {
        /** The selection button. */
        protected JButton m_selectButton;
        /** The button used to remove this item from the tab. */
        protected JButton m_removeButton;

        public TitlePanel(ActionListener selectListener,ActionListener removeListener) {
            setLayout(this);
            m_removeButton=new JButton();
            m_removeButton.addActionListener(removeListener);
            m_removeButton.setMargin(new Insets(1,1,1,1));
            m_removeButton.setIconTextGap(0);
            m_removeButton.setOpaque(false);
            m_removeButton.setBorderPainted(false);
            m_removeButton.setFocusable(false);
            m_removeButton.setVisible(false);
            m_removeButton.addMouseListener(new MouseAdapter() {
                public void mouseEntered(MouseEvent mouseEvent) {
                    m_removeButton.setBorderPainted(true);
                }
                public void mouseExited(MouseEvent mouseEvent) {
                    m_removeButton.setBorderPainted(false);
                }
            });
            m_selectButton=new JButton();
            m_selectButton.addActionListener(selectListener);
            m_selectButton.setHorizontalAlignment(JButton.LEFT);
            add(m_selectButton,new Integer(0));
            add(m_removeButton,new Integer(1));
        }
        public void setRemovalButtonState(boolean visible) {
            m_removeButton.setVisible(visible);
            m_selectButton.setMargin(null);
            if (visible) {
                Insets margin=m_selectButton.getMargin();
                Dimension removeSize=m_removeButton.getPreferredSize();
                if (removeSize!=null && margin.right<2*INNER_BUTTON_MARGIN+removeSize.width) {
                    margin.right=2*INNER_BUTTON_MARGIN+removeSize.width;
                    m_selectButton.setMargin(margin);
                }
            }
            revalidate();
        }
        public void addLayoutComponent(String name,Component component) {
        }
        public void removeLayoutComponent(Component component) {
        }
        public void layoutContainer(Container parent) {
            m_selectButton.setBounds(0,0,getWidth(),getHeight());
            Dimension preferredSize=m_removeButton.getPreferredSize();
            m_removeButton.setBounds(getWidth()-INNER_BUTTON_MARGIN-preferredSize.width,(getHeight()-preferredSize.height)/2,preferredSize.width,preferredSize.height);
        }
        public Dimension minimumLayoutSize(Container parent) {
            Dimension result=m_selectButton.getMinimumSize();
            Dimension inner=m_removeButton.getMinimumSize();
            if (result.height<inner.height+2*INNER_BUTTON_MARGIN)
                result.height=inner.height+2*INNER_BUTTON_MARGIN;
            return result;
        }
        public Dimension preferredLayoutSize(Container parent) {
            Dimension result=m_selectButton.getPreferredSize();
            Dimension inner=m_removeButton.getPreferredSize();
            if (result.height<inner.height+2*INNER_BUTTON_MARGIN)
                result.height=inner.height+2*INNER_BUTTON_MARGIN;
            return result;
        }
    }
}
