package de.fzi.wim.guibase.selectionsource;

import java.util.Collection;
import java.util.Collections;
import java.util.EventListener;
import javax.swing.event.EventListenerList;

/**
 * The manager of selection sources.
 */
public class SelectionSourceManager {
    /** The event listener list. */
    protected EventListenerList m_eventListenerList;
    /** The active selection source. */
    protected SelectionSource m_activeSelectionSource;
    /** The active selection. */
    protected Collection m_activeSelection;

    /**
     * Creates the selection source.
     */
    public SelectionSourceManager() {
        m_eventListenerList=new EventListenerList();
        reloadActiveSelection();
    }
    /**
     * Adds a listener to this manager.
     *
     * @param selectionSourceManagerListener        the listener
     */
    public void addSelectionSourceManagerListener(SelectionSourceManagerListener selectionSourceManagerListener) {
        m_eventListenerList.add(SelectionSourceManagerListener.class,selectionSourceManagerListener);
    }
    /**
     * Removes a listener from this manager.
     *
     * @param selectionSourceManagerListener        the listener
     */
    public void removeSelectionSourceManagerListener(SelectionSourceManagerListener selectionSourceManagerListener) {
        m_eventListenerList.remove(SelectionSourceManagerListener.class,selectionSourceManagerListener);
    }
    /**
     * Fires the notification that the active selection source changed.
     */
    protected void fireActiveSelectionSourceChanged() {
        EventListener[] listeners=m_eventListenerList.getListeners(SelectionSourceManagerListener.class);
        for (int i=0;i<listeners.length;i++)
            ((SelectionSourceManagerListener)listeners[i]).activeSelectionSourceChanged(this,m_activeSelectionSource);
    }
    /**
     * Makes supplied selection source active.
     *
     * @param selectionSource                       the selection source to make active
     */
    public void setActiveSelectionSource(SelectionSource selectionSource) {
        if (m_activeSelectionSource!=selectionSource) {
            if (m_activeSelectionSource!=null)
                m_activeSelectionSource.notifySelectionSecondary();
            m_activeSelectionSource=selectionSource;
            reloadActiveSelection();
            if (m_activeSelectionSource!=null)
                m_activeSelectionSource.notifySelectionPrimary();
            fireActiveSelectionSourceChanged();
        }
    }
    /**
     * Returns the active selection source.
     *
     * @return                                      returns the active selection source
     */
    public SelectionSource getActiveSelectionSource() {
        return m_activeSelectionSource;
    }
    /**
     * Returns the selection of the active selection source.
     *
     * @return                                      the selection of the active selection source
     */
    public Collection getActiveSelection() {
        return m_activeSelection;
    }
    /**
     * Notifies this manager that the selection in the source has changed.
     *
     * @param selectionSource                       the selection source
     */
    public void notifySelectionChanged(SelectionSource selectionSource) {
        boolean isActive=(selectionSource==m_activeSelectionSource);
        if (isActive)
            reloadActiveSelection();
        EventListener[] listeners=m_eventListenerList.getListeners(SelectionSourceManagerListener.class);
        for (int i=0;i<listeners.length;i++)
            ((SelectionSourceManagerListener)listeners[i]).selectionChanged(this,selectionSource,isActive);
    }
    /**
     * Reloads the active selection from the active selection source.
     */
    protected void reloadActiveSelection() {
        if (m_activeSelectionSource==null)
            m_activeSelection=Collections.EMPTY_LIST;
        else
            m_activeSelection=m_activeSelectionSource.getSelection();
    }
}
