package de.fzi.wim.guibase.parameters;

import java.util.Map;
import java.util.HashMap;
import java.util.Iterator;
import java.awt.Dimension;
import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.JTabbedPane;
import javax.swing.JDialog;
import javax.swing.JPasswordField;
import javax.swing.JPanel;
import javax.swing.JButton;
import javax.swing.JScrollPane;
import javax.swing.JFrame;
import javax.swing.table.TableModel;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableCellEditor;
import javax.swing.BorderFactory;

import de.fzi.wim.guibase.localization.*;
import de.fzi.wim.guibase.tables.*;

/**
 * Simple dialog for completing connection parameters.
 */
public class ParametersCompletionDlg extends JDialog {
    /** Set to <code>true</code> if dialog is confirmed. */
    protected boolean m_confirmed;
    /** the list of parameter maps. */
    protected Map m_connectionParameters;
    /** the current localization manager instance */
    protected LocalizationManager m_localizationManager;
    /** the tabbed pane to list the parameters maps */
    protected JTabbedPane m_tabbedPane;
    /** the ID of the title */
    protected String m_titleID;

    /**
     * Creates an instance of this class.
     *
     * @param owner                                 the owner
     * @param localizationManager                   localization manager
     * @param titleID                               the ID of the title
     * @param parametersInfos                       terator of ParametersInfos map of connection parameters
     */
    public ParametersCompletionDlg(JFrame owner,LocalizationManager localizationManager,String titleID,Iterator parametersInfos) {
        super(owner,localizationManager.getPhrase(titleID),true);
        m_localizationManager=localizationManager;
        m_connectionParameters=new HashMap();
        m_titleID=titleID;
        setDefaultCloseOperation(DISPOSE_ON_CLOSE);
        // buttons
        JButton okButton=localizationManager.getButton(titleID+".OK");
        okButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                m_confirmed=true;
                dispose();
            }
        });
        JButton cancelButton=localizationManager.getButton(titleID+".cancel");
        cancelButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                dispose();
            }
        });
        // create the buttons pane
        JPanel buttonsPane=new JPanel(new FlowLayout(FlowLayout.RIGHT));
        buttonsPane.add(okButton);
        buttonsPane.add(cancelButton);
        buttonsPane.setBorder(BorderFactory.createEmptyBorder(0,5,5,0));
        //create the tabbed pane for the parameters info tables
        m_tabbedPane=new JTabbedPane();
        addParametersInfos(parametersInfos);
        m_tabbedPane.setSelectedIndex(0);
        // and the root pane to which buttons and the list will be added
        JPanel rootPane=new JPanel(new BorderLayout());
        rootPane.add(m_tabbedPane,BorderLayout.CENTER);
        rootPane.add(buttonsPane,BorderLayout.SOUTH);
        setContentPane(rootPane);
        // finish the dialog
        getRootPane().setDefaultButton(okButton);
        pack();
        setLocationRelativeTo(owner);
    }
    /**
     * Returns whether dialog was confirmed with OK.
     *
     * @return                      <code>true</code> if dialog has been confirmed
     */
    public boolean isConfirmed() {
        return m_confirmed;
    }
    /**
     * Returns the map of parameters for establishing the KAON connection.
     *
     * @param name                  the name of the map to retrieve
     * @return                      the KAON connection parameters <code>null</code> if map cannot be found
     */
    public Map getConnectionParameters(String name) {
        ParametersInfo parametersInfo=(ParametersInfo)m_connectionParameters.get(name);
        if (parametersInfo!=null)
            return parametersInfo.getConnectionParameters();
        return null;
    }
    /**
     * Adds some ParametersInfos to the dialog
     *
     * @param parametersInfos        iterator of ParametersInfos map of connection parameters
     * @return                       returns <code>true</code> on success, <code>false</code> on error adding the parametersinfos
     */
    public boolean addParametersInfos(Iterator parametersInfos) {
        boolean success=true;
        // and add all parameter infos
        while (parametersInfos.hasNext()) {
            ParametersInfo parametersInfo=(ParametersInfo)parametersInfos.next();
            String name=parametersInfo.getName();
            if (name==null || "".equals(name) || m_connectionParameters.containsKey(name))
                success=false;
            else {
                ParametersInfoComponent parametersInfoComponent=new ParametersInfoComponent(parametersInfo,m_localizationManager.getPhrase(m_titleID+".connectionKey"),m_localizationManager.getPhrase(m_titleID+".connectionValue"),m_localizationManager.getPhrase(m_titleID+".missingParameters"));
                m_connectionParameters.put(name,parametersInfoComponent);
                m_tabbedPane.addTab(name,null,parametersInfoComponent,m_localizationManager.getPhrase(m_titleID+".missingParameters")+" for '"+name+"'" );
            }
        }
        return success;
    }
    /**
     * Tests whether parameters need to be completed.
     *
     * @param parameters            the paramteters
     * @return                      <code>true</code> if parameters should be completed
     */
    public static boolean needsCompletion(Map parameters) {
        Iterator values=parameters.values().iterator();
        while (values.hasNext()) {
            if ("???".equals(values.next().toString()))
                return true;
        }
        return false;
    }

    /**
     * Interface for ParametersInfo classes.
     */
    public interface ParametersInfo {
        /**
         * Returns the name of the parameters map.
         *
         * @return             the name of the map
         */
        String getName();
        /**
         * Returns the map of parameters for establishing the KAON connection.
         *
         * @return                      the KAON connection parameters
         */
        public Map getConnectionParameters();
    }

    /**
     * Component class to store and edit the information about a parameters map
     */
    protected class ParametersInfoComponent extends JPanel implements ParametersInfo {
        /** parameters info */
        ParametersInfo m_parametersInfo;
        /** Table component for the connection parameters. */
        protected ParametersTable m_parameters;

        /**
         * Creates an instance of this class.
         *
         * @param parametersinfo                        the parameters info
         * @param connectionkey                         the string for the connection key table header
         * @param connectionvalue                       the string for the connection value table header
         * @param missingparameters                     the string for the border frame
        */
        public ParametersInfoComponent(ParametersInfo parametersinfo,String connectionkey,String connectionvalue,String missingparameters) {
            super(new BorderLayout());
            m_parametersInfo=parametersinfo;
            // parameters table
            DefaultTableModel tableModel=new DefaultTableModel(new Object[] { connectionkey,connectionvalue },0);
            Iterator keys=m_parametersInfo.getConnectionParameters().keySet().iterator();
            while (keys.hasNext()) {
                String key=(String)keys.next();
                String value=(String)m_parametersInfo.getConnectionParameters().get(key);
                if ("???".equals(value))
                    tableModel.addRow(new Object[] { key,"" });
            }
            m_parameters=new ParametersTable(tableModel);
            m_parameters.getTableHeader().setReorderingAllowed(false);
            JScrollPane parametersScrollPane=new JScrollPane(m_parameters);
            parametersScrollPane.getViewport().setBackground(m_parameters.getBackground());
            parametersScrollPane.setPreferredSize(new Dimension(300,200));
            parametersScrollPane.setBorder(BorderFactory.createTitledBorder(missingparameters));
            add(parametersScrollPane,BorderLayout.CENTER);
        }
        /**
         * Returns the name of the parameters map.
         *
         * @return             the name of the map
         */
        public String getName() {
            return m_parametersInfo.getName();
        }
        /**
         * Returns the map of parameters for establishing the KAON connection.
         *
         * @return                      the KAON connection parameters should never be <code>null</code>
         */
        public Map getConnectionParameters() {
            Map parameters=new HashMap(m_parametersInfo.getConnectionParameters());
            for (int i=0;i<m_parameters.getRowCount();i++) {
                String key=(String)m_parameters.getValueAt(i,0);
                String value=((String)m_parameters.getValueAt(i,1)).trim();
                parameters.put(key,value);
            }
            return parameters;
        }
    }

    /**
     * Table allowing entering of connection parameters.
     */
    protected static class ParametersTable extends SmartTable {
        protected TableCellRenderer m_passwordRenderer;
        protected SmartCellEditor m_passwordEditor;

        public ParametersTable(TableModel model) {
            super(model);
            m_passwordRenderer=new PasswordRenderer();
            m_passwordEditor=new SmartCellEditor(new JPasswordField());
        }
        public TableCellRenderer getCellRenderer(int row,int column) {
            if (isPasswordField(row,column))
                return m_passwordRenderer;
            else
                return super.getCellRenderer(row,column);
        }
        public TableCellEditor getCellEditor(int row,int column) {
            if (isPasswordField(row,column))
                return m_passwordEditor;
            else
                return super.getCellEditor(row,column);
        }
        protected boolean isPasswordField(int row,int column) {
            if (column==1) {
                String value=(String)getValueAt(row,0);
                return value.indexOf("PASSWORD")!=-1;
            }
            else
                return false;
        }
        public boolean isCellEditable(int row,int column) {
            return column==1;
        }
    }

    /**
     * Default ParametersInfo class
     */
    public static class DefaultCompletionParametersInfo implements ParametersCompletionDlg.ParametersInfo {
        /** name of the parameters info */
        protected String m_name;
        /** connection parameters map */
        protected Map m_connectionParametersMap;

        /**
         * Creates an instance of this class.
         *
         * @param name                                  name of the parameters info
         * @param connectionParametersMap               connection parameters map
         */
        public DefaultCompletionParametersInfo(String name,Map connectionParametersMap) {
            setName(name);
            setConnectionParameters(connectionParametersMap);
        }
        /**
         * Returns the name of the parameters map.
         *
         * @return             the name of the map
         */
        public String getName() {
            return m_name;
        }
        /**
         * Sets the name of the parameters map.
         *
         * @param name                                  name of the parameters info
         */
        public void setName(String name) {
            m_name=name;
        };
        /**
         * Returns the map of parameters for establishing the KAON connection.
         *
         * @return                      the KAON connection parameters
         */
        public Map getConnectionParameters() {
            return m_connectionParametersMap;
        }
        /**
         * Sets the parameters map.
         *
         * @param connectionParametersMap               connection parameters map
         */
        public void setConnectionParameters(Map connectionParametersMap) {
            m_connectionParametersMap=connectionParametersMap;
        }
    }
}
