package de.fzi.wim.guibase.menus;

import java.awt.Component;
import java.awt.event.ContainerEvent;
import java.awt.event.ContainerListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseAdapter;
import javax.swing.Action;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JMenu;

/**
 * Implements a menu tracker that will display menu help text in the status line when the user drags the mouse over a menu item.
 */
public class MenuHelpTracker {
    /** Menu that is being tracked. */
    protected JMenuBar m_menuBar;
    /** Place where help will be shown. */
    protected JLabel m_helpControl;
    /** Default text in the help control. */
    protected String m_defaultText;
    /** Component listener that watches for modifications of the menu bar. */
    protected ContainerListener m_containerListener;
    /** Mouse listener that listens for mouse motion events in menu items. */
    protected MouseListener m_mouseListener;

    /**
     * Creates menu tracker and attaches it to a menu bar.
     *
     * @param helpControl                   palce where help will be shown
     */
    public MenuHelpTracker(JLabel helpControl) {
        m_helpControl=helpControl;
        m_defaultText=m_helpControl.getText();
        m_containerListener=createContainerListener();
        m_mouseListener=createMouseListener();
    }
    /**
     * Adds specified menu bar for tracking. If the parameter is <code>null</code>, current menu bar is not
     * tracked any more.
     *
     * @param menuBar                       menu bar that is tracked
     */
    public void trackMenuBar(JMenuBar menuBar) {
        if (m_menuBar!=null)
            uninstallListeners(m_menuBar);
        m_menuBar=menuBar;
        if (m_menuBar!=null)
            installListeners(m_menuBar);
    }
    /**
     * Creates a container listener that watches for modifications to the menu bar structure.
     *
     * @return                              container listener
     */
    protected ContainerListener createContainerListener() {
        return new ContainerListener() {
            public void componentAdded(ContainerEvent e) {
                if (e.getComponent() instanceof JComponent)
                    installListeners((JComponent)e.getComponent());
            }
            public void componentRemoved(ContainerEvent e) {
                if (e.getComponent() instanceof JComponent)
                    uninstallListeners((JComponent)e.getComponent());
            }
        };
    }
    /**
     * Creates a mouse listener that watches for mouse events.
     *
     * @return                              mouse listener
     */
    protected MouseListener createMouseListener() {
        return new MouseAdapter() {
            public void mouseEntered(MouseEvent e) {
                mouseEnteredItem((JMenuItem)e.getSource());
            }
            public void mouseExited(MouseEvent e) {
                mouseExitedItem((JMenuItem)e.getSource());
            }
        };
    }
    /**
     * Installs listeners to supplied component and all of its chidlren.
     *
     * @param component                     component that listeners need to be added to
     */
    protected void installListeners(JComponent component) {
        component.addContainerListener(m_containerListener);
        if (component instanceof JMenuItem)
            component.addMouseListener(m_mouseListener);
        int numberOfComponents=component.getComponentCount();
        for (int i=0;i<numberOfComponents;i++) {
            Component child=component.getComponent(i);
            if (child instanceof JComponent)
                installListeners((JComponent)child);
        }
        if (component instanceof JMenu)
            installListeners(((JMenu)component).getPopupMenu());
    }
    /**
     * Removes listeners from supplied component and all of its chidlren.
     *
     * @param component                     component that listeners need to be removed from
     */
    protected void uninstallListeners(JComponent component) {
        component.removeContainerListener(m_containerListener);
        if (component instanceof JMenuItem)
            component.removeMouseListener(m_mouseListener);
        int numberOfComponents=component.getComponentCount();
        for (int i=0;i<numberOfComponents;i++) {
            Component child=component.getComponent(i);
            if (child instanceof JComponent)
                uninstallListeners((JComponent)child);
        }
        if (component instanceof JMenu)
            uninstallListeners(((JMenu)component).getPopupMenu());
    }
    /**
     * Called when mouse has entered into an area of a menu item.
     *
     * @param menuItem                      item that was entered
     */
    protected void mouseEnteredItem(JMenuItem menuItem) {
        String newText=(String)menuItem.getClientProperty(Action.LONG_DESCRIPTION);
        if (newText==null && menuItem.getAction()!=null)
            newText=(String)menuItem.getAction().getValue(Action.LONG_DESCRIPTION);
        if (newText!=null && newText.length()!=0)
            m_helpControl.setText(newText);
        else
            m_helpControl.setText(m_defaultText);
    }
    /**
     * Called when mouse has exited an area of a menu item.
     *
     * @param menuItem                      item that was exited
     */
    protected void mouseExitedItem(JMenuItem menuItem) {
        m_helpControl.setText(m_defaultText);
    }
}
