package de.fzi.wim.guibase.menus;

import java.util.Map;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.Icon;
import javax.swing.Action;
import javax.swing.AbstractButton;
import javax.swing.DefaultButtonModel;
import javax.swing.JMenuBar;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JComponent;
import javax.swing.KeyStroke;
import javax.swing.event.PopupMenuEvent;
import javax.swing.event.PopupMenuListener;
import org.w3c.dom.Node;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import de.fzi.wim.guibase.localization.LocalizationManager;
import de.fzi.wim.guibase.actions.ModalSmartAction;
import de.fzi.wim.guibase.actions.SmartActionDecorator;
import de.fzi.wim.guibase.actions.SmartAction;
import de.fzi.wim.guibase.actions.SmartActionMap;
import de.fzi.wim.guibase.actions.SmartActionManager;
import de.fzi.wim.guibase.actions.SmartActionAggregate;

/**
 * Utility class that can be used to build menus. Structure of menus is defined by an XML structure.
 * Instance of this class must be given a localization manager and an action map of all actions known to the application.
 * Menu builder will use these actions to build menu items that reference particular action.
 * <p>Menu structure is defined using following element (each can have actionID attribute defining the ID of the action):
 * <ul>
 *  <li>&lt;menubar&gt; - defines menu bar contents (can contain &lt;menuitem&gt; or &lt;menu&gt; elements)
 *  <li>&lt;menu&gt; - defines menu (can contain &lt;menuitem&gt; elements)
 *  <li>&lt;menuitem&gt; - defines menu idem (can't contain any elements)
 *  <li>&lt;separator&gt; - defines a separator in the menu
 *  <li>&lt;aggregate&gt; - defines an aggregate of the actions
 * </ul>
 */
public class LocalizedMenuBuilder {
    /** Localization manager that is used to build menu items. */
    protected LocalizationManager m_localizationManager;
    /** Map of all actions (mapped by their action ID). */
    protected SmartActionMap m_actions;
    /** The manager. */
    protected SmartActionManager m_smartActionManager;

    /**
     * Creates an instalce of this class and initializes it with a localization manager and an action map.
     *
     * @param localizationManager               source of the localization information
     * @param smartActionManager                the manager
     * @param actions                           map of actions (mapped by their action ID)
     */
    public LocalizedMenuBuilder(LocalizationManager localizationManager,SmartActionManager smartActionManager,SmartActionMap actions) {
        m_localizationManager=localizationManager;
        m_smartActionManager=smartActionManager;
        m_actions=actions;
    }
    /**
     * Creates a popup menu from its XML definition. Node passed in must be a <code>popupmenu</code> node.
     *
     * @param popupMenuNode                     node defining the popup
     * @return                                  the popup menu defined by the node
     * @exception IllegalArgumentException      thrown if there is a problem with the menu bar specification
     */
    public JPopupMenu createPopupMenu(Element popupMenuNode) throws IllegalArgumentException {
        if (!"popupmenu".equalsIgnoreCase(popupMenuNode.getNodeName()))
            throw new IllegalArgumentException("Popup menu must be created from a <popupmenu> element.");
        JPopupMenu popupMenu=new JPopupMenu();
        loadComponent(popupMenuNode.getChildNodes(),popupMenu,null,"",null);
        addPopupMenuListeners(popupMenu);
        return popupMenu;
    }
    /**
     * Creates a menu bar from XML definition. Node passed in must be a <code>menubar</code> node.
     *
     * @param menuBarNode                       node defining the menu bar
     * @return                                  menu bar defined by this node
     * @exception IllegalArgumentException      thrown if there is a problem with the menu bar specification
     */
    public JMenuBar createMenuBar(Element menuBarNode) throws IllegalArgumentException {
        if (!"menubar".equalsIgnoreCase(menuBarNode.getNodeName()))
            throw new IllegalArgumentException("Menu bar must be created from a <menubar> element.");
        JMenuBar menuBar=new JMenuBar();
        loadComponent(menuBarNode.getChildNodes(),menuBar,null,"",null);
        return menuBar;
    }
    /**
     * Creates a menu item from XML definition. Node passed in must be a <code>menuitem</code> node.
     *
     * @param menuItemNode                      node definining the menu item
     * @param inheritedContexts                 the names of inherited contexts
     * @param inheritedType                     the inherited component type
     * @param inheritedHideWhenDisabled         the inderited 'hide when disabled' flag
     * @return                                  menu item defined by this node
     */
    protected JMenuItem createMenuItem(Element menuItemNode,String inheritedContexts,String inheritedType,String inheritedHideWhenDisabled) {
        JMenuItem menuItem=createMenuItemControl(menuItemNode,inheritedContexts,inheritedType,inheritedHideWhenDisabled);
        String actionID=menuItemNode.getAttribute("actionID");
        if (actionID.length()==0)
            menuItem.setText("Menu item doesn't have actionID");
        else {
            SmartAction action=getAction(actionID);
            if (action==null)
                updateMenuItemText(menuItem,actionID);
            else
                menuItem.setAction(action);
        }
        return menuItem;
    }
    /**
     * Creates an aggregate from XML definition. Node passed in must be an <code>aggregate</code> node.
     *
     * @param aggregateNode                     node defining the aggregate
     * @param inheritedContexts                 the names of inherited contexts
     * @param inheritedType                     the inherited component type
     * @param inheritedHideWhenDisabled         the inderited 'hide when disabled' flag
     * @return                                  menu item for the aggregate node
     */
    protected JMenuItem createAggregate(Element aggregateNode,String inheritedContexts,String inheritedType,String inheritedHideWhenDisabled) {
        JMenuItem menuItem=createMenuItemControl(aggregateNode,inheritedContexts,inheritedType,inheritedHideWhenDisabled);
        String actionID=aggregateNode.getAttribute("actionID");
        if (actionID.length()==0)
            menuItem.setText("Aggregate doesn't have actionID");
        else {
            SmartActionAggregate aggregate=(SmartActionAggregate)m_actions.getAction(actionID);
            if (aggregate==null) {
                aggregate=new SmartActionAggregate(actionID,m_smartActionManager,m_localizationManager);
                m_actions.put(actionID,aggregate);
            }
            menuItem.setAction(aggregate);
            NodeList nodeList=aggregateNode.getChildNodes();
            for (int i=0;i<nodeList.getLength();i++) {
                Node child=nodeList.item(i);
                if (child.getNodeType()==Node.ELEMENT_NODE) {
                    Element element=(Element)child;
                    if ("action".equalsIgnoreCase(element.getNodeName())) {
                        String aggregatedActionID=element.getAttribute("actionID");
                        SmartAction aggregatedAction=getAction(aggregatedActionID);
                        aggregate.addAction(aggregatedAction);
                    }
                }
            }
            aggregate.updateAction();
        }
        return menuItem;
    }
    /**
     * Creates a menu from XML definition. Node passed in must be a <code>menu</code> node.
     *
     * @param menuNode                          node defining the menu
     * @param inheritedContexts                 the names of inherited contexts
     * @param inheritedType                     the inherited component type
     * @param inheritedHideWhenDisabled         the inderited 'hide when disabled' flag
     * @return                                  menu defined by this node
     */
    protected JMenu createMenu(Element menuNode,String inheritedContexts,String inheritedType,String inheritedHideWhenDisabled) {
        JMenu menu=(JMenu)createMenuItemControl(menuNode,inheritedContexts,inheritedType,inheritedHideWhenDisabled);
        String actionID=menuNode.getAttribute("actionID");
        if (actionID.length()!=0)
            updateMenuItemText(menu,actionID);
        loadComponent(menuNode.getChildNodes(),menu,inheritedContexts,inheritedType,inheritedHideWhenDisabled);
        addPopupMenuListeners(menu.getPopupMenu());
        return menu;
    }
    /**
     * Creates a menu from XML definition attached to the modal action. Node passed in must be a <code>modalmenu</code> node.
     *
     * @param menuNode                          node defining the menu
     * @param inheritedContexts                 the names of inherited contexts
     * @param inheritedType                     the inherited component type
     * @param inheritedHideWhenDisabled         the inderited 'hide when disabled' flag
     * @return                                  menu defined by this node
     */
    protected JMenu createModalMenu(Element menuNode,String inheritedContexts,String inheritedType,String inheritedHideWhenDisabled) {
        JMenu menu=(JMenu)createMenuItemControl(menuNode,inheritedContexts,inheritedType,inheritedHideWhenDisabled);
        String actionID=menuNode.getAttribute("actionID");
        if (actionID.length()==0)
            menu.setText("Menu item doesn't have actionID");
        else {
            SmartAction action=getAction(actionID);
            if (action instanceof ModalSmartAction) {
                String value=menuNode.getAttribute("type");
                String type=value.length()==0 ? inheritedType : value;
                int typeValue=0;
                if ("check".equals(type))
                    typeValue=1;
                else if ("radio".equals(type))
                    typeValue=2;
                ModalMenuManager modalMenuManager=new ModalMenuManager(actionID,m_localizationManager,(ModalSmartAction)action,menu.getPopupMenu(),typeValue);
                menu.setAction(modalMenuManager);
            }
            else
                menu.setText("Action "+actionID+" isn't a modal smart action.");
        }
        return menu;
    }
    /**
     * Registers actions as listeners to the popup menu.
     *
     * @param popupMenu                         the popup menu
     */
    protected void addPopupMenuListeners(JPopupMenu popupMenu) {
        for (int i=0;i<popupMenu.getComponentCount();i++) {
            JComponent child=(JComponent)popupMenu.getComponent(i);
            if (child instanceof AbstractButton) {
                AbstractButton button=(AbstractButton)child;
                Action itemAction=button.getAction();
                if (itemAction instanceof SmartAction)
                    popupMenu.addPopupMenuListener((SmartAction)itemAction);
            }
        }
    }
    /**
     * Updates a menu item text from localization source.
     *
     * @param menuItem                          menu item to be updated
     * @param actionID                          action ID of the node
     */
    protected void updateMenuItemText(JMenuItem menuItem,String actionID) {
        Map actionMap=m_localizationManager.getActionMap(actionID);
        if (actionMap!=null) {
            menuItem.setText((String)actionMap.get(Action.NAME));
            menuItem.setIcon((Icon)actionMap.get(Action.SMALL_ICON));
            Integer mnemonic=(Integer)actionMap.get(Action.MNEMONIC_KEY);
            if (mnemonic!=null)
                menuItem.setMnemonic(mnemonic.intValue());
            if (!(menuItem instanceof JMenu))
                menuItem.setAccelerator((KeyStroke)actionMap.get(Action.ACCELERATOR_KEY));
            String longDescription=(String)actionMap.get(Action.LONG_DESCRIPTION);
            if (longDescription!=null)
                menuItem.putClientProperty(Action.LONG_DESCRIPTION,longDescription);
        }
        else
            menuItem.setText("Menu item with action ID '"+actionID+"' could not be found.");
    }
    /**
     * Parses nodes under given root and adds all elements to the supplied container.
     *
     * @param nodeList                          list of nodes to parse
     * @param component                         component to which elements are added
     * @param inheritedContexts                 the inherited contexts
     * @param inheritedType                     the inherited component type
     * @param inheritedHideWhenDisabled         the inderited 'hide when disabled' flag
     */
    protected void loadComponent(NodeList nodeList,JComponent component,String inheritedContexts,String inheritedType,String inheritedHideWhenDisabled) {
        int length=nodeList.getLength();
        for (int i=0;i<length;i++) {
            Node child=nodeList.item(i);
            if (child.getNodeType()==Node.ELEMENT_NODE) {
                JComponent newItem=null;
                Element element=(Element)child;
                String nodeName=element.getNodeName();
                if ("menuitem".equalsIgnoreCase(nodeName))
                    newItem=createMenuItem(element,inheritedContexts,inheritedType,inheritedHideWhenDisabled);
                else if ("menu".equalsIgnoreCase(nodeName))
                    newItem=createMenu(element,inheritedContexts,inheritedType,inheritedHideWhenDisabled);
                else if ("modalmenu".equalsIgnoreCase(nodeName))
                    newItem=createModalMenu(element,inheritedContexts,inheritedType,inheritedHideWhenDisabled);
                else if ("separator".equalsIgnoreCase(nodeName)) {
                    if (component instanceof JMenu) {
                        newItem=new JPopupMenu.Separator();
                        String contexts=element.getAttribute("contexts");
                        if (contexts.length()==0)
                            contexts=inheritedContexts;
                        if (contexts!=null)
                            newItem.putClientProperty("guibase.contexts",","+contexts+",");
                    }
                }
                else if ("aggregate".equalsIgnoreCase(nodeName))
                    newItem=createAggregate(element,inheritedContexts,inheritedType,inheritedHideWhenDisabled);
                else if ("group".equalsIgnoreCase(nodeName)) {
                    String newInheritedContexts=inheritedContexts;
                    String value=element.getAttribute("contexts");
                    if (value.length()!=0)
                        newInheritedContexts=value;
                    String newInheritedType=inheritedType;
                    value=element.getAttribute("type");
                    if (value.length()!=0)
                        newInheritedType=value;
                    String newInheritedHideWhenDisabled=inheritedHideWhenDisabled;
                    value=element.getAttribute("hideWhenDisabled");
                    if (value.length()!=0)
                        newInheritedHideWhenDisabled=value;
                    loadComponent(element.getChildNodes(),component,newInheritedContexts,newInheritedType,newInheritedHideWhenDisabled);
                }
                if (newItem!=null)
                    component.add(newItem);
            }
        }
    }
    /**
     * Creates a menu item from nore specification.
     *
     * @param element                           the element of the control
     * @param inheritedContexts                 the names of inherited contexts
     * @param inheritedType                     the inherited component type
     * @param inheritedHideWhenDisabled         the inderited 'hide when disabled' flag
     * @return                                  menu item for the node
     */
    protected JMenuItem createMenuItemControl(Element element,String inheritedContexts,String inheritedType,String inheritedHideWhenDisabled) {
        JMenuItem menuItem;
        if ("menu".equals(element.getNodeName()) || "modalmenu".equals(element.getNodeName()))
            menuItem=new SmartMenu(m_smartActionManager);
        else {
            String type=element.getAttribute("type");
            if (type.length()==0)
                type=inheritedType;
            if ("check".equalsIgnoreCase(type))
                menuItem=new SmartCheckBoxMenuItem(m_smartActionManager);
            else if ("radio".equalsIgnoreCase(type))
                menuItem=new SmartRadioButtonMenuItem(m_smartActionManager);
            else
                menuItem=new SmartMenuItem(m_smartActionManager);
        }
        String hideWhenDisabled=element.getAttribute("hideWhenDisabled");
        if (hideWhenDisabled.length()==0)
            hideWhenDisabled=inheritedHideWhenDisabled;
        if ("true".equalsIgnoreCase(hideWhenDisabled))
            menuItem.putClientProperty("hideWhenDisabled","true");
        else {
            String contexts=element.getAttribute("contexts");
            if (contexts.length()==0)
                contexts=inheritedContexts;
            if (contexts!=null)
                menuItem.putClientProperty("guibase.contexts",","+contexts+",");
        }
        return menuItem;
    }
    /**
     * Returns <code>true</code> if the component should be hidden if it is disabled.
     *
     * @param component             the component
     * @return                      <code>true</code> if component should be hidden if it is disabled
     */
    public static boolean hideWhenDisabled(JComponent component) {
        return "true".equals(component.getClientProperty("hideWhenDisabled"));
    }
    /**
     * Checks whether the component is visible in the specified context.
     *
     * @param component             the component
     * @param smartActionManager    the smart action manager
     * @return                      <code>true</code> if the component is visible in given context
     */
    public static boolean isVisibleInContext(JComponent component,SmartActionManager smartActionManager) {
        if (smartActionManager!=null) {
            String contextName=smartActionManager.getCurrentContextName();
            String allowedContexts=(String)component.getClientProperty("guibase.contexts");
            if (allowedContexts!=null) {
                if (contextName==null)
                    return false;
                else
                    return allowedContexts.indexOf(contextName)!=-1;
            }
        }
        return true;
    }
    /**
     * Returns the action for given ID.
     *
     * @param actionID              the acion ID
     * @return                      the action
     */
    protected SmartAction getAction(String actionID) {
        return m_actions.getAction(actionID);
    }

    /**
     * Implements a menu item that may be hidden when not selected.
     */
    public static class SmartMenuItem extends JMenuItem {
        /** The manager. */
        protected SmartActionManager m_smartActionManager;

        /**
         * Creates an instance of this class.
         *
         * @param smartActionManager        the manager
         */
        public SmartMenuItem(SmartActionManager smartActionManager) {
            m_smartActionManager=smartActionManager;
        }
        /**
         * Creates a property change listener for the button.
         *
         * @param action                    action for which the listener is created
         * @return                          property change listener
         */
        protected PropertyChangeListener createActionPropertyChangeListener(Action action) {
            return new ButtonActionPropertyChangeListener(this,m_smartActionManager);
        }
        /**
         * Configures properties from given action.
         *
         * @param action                    the action
         */
        protected void configurePropertiesFromAction(Action action) {
            if (action!=null) {
                this.setIcon((Icon)action.getValue(Action.SMALL_ICON));
                this.setText((String)action.getValue(Action.NAME));
                this.setEnabled(action.isEnabled());
                Integer mnemonic=(Integer)action.getValue(Action.MNEMONIC_KEY);
                if (mnemonic!=null)
                    this.setMnemonic(mnemonic.intValue());
                this.setToolTipText((String)action.getValue(Action.SHORT_DESCRIPTION));
                this.setAccelerator((KeyStroke)action.getValue(Action.ACCELERATOR_KEY));
                boolean visible=isVisibleInContext(this,m_smartActionManager);
                if (hideWhenDisabled(this) && !action.isEnabled())
                    visible=false;
                this.setVisible(visible);
            }
        }
    }

    /**
     * Implements a menu item that mimics action's selected state.
     */
    public static class SmartCheckBoxMenuItem extends JCheckBoxMenuItem {
        /** The manager. */
        protected SmartActionManager m_smartActionManager;

        /**
         * Creates an instance of this class.
         *
         * @param smartActionManager        the manager
         */
        public SmartCheckBoxMenuItem(SmartActionManager smartActionManager) {
            m_smartActionManager=smartActionManager;
        }
        /**
         * Creates a property change listener for the button.
         *
         * @param action                    action for which the listener is created
         * @return                          property change listener
         */
        protected PropertyChangeListener createActionPropertyChangeListener(Action action) {
            return new ButtonActionPropertyChangeListener(this,m_smartActionManager);
        }
        /**
         * Configures properties from given action.
         *
         * @param action                    the action
         */
        protected void configurePropertiesFromAction(Action action) {
            if (action!=null) {
                this.setIcon((Icon)action.getValue(Action.SMALL_ICON));
                this.setText((String)action.getValue(Action.NAME));
                this.setEnabled(action.isEnabled());
                Integer mnemonic=(Integer)action.getValue(Action.MNEMONIC_KEY);
                if (mnemonic!=null)
                    this.setMnemonic(mnemonic.intValue());
                this.setToolTipText((String)action.getValue(Action.SHORT_DESCRIPTION));
                this.setSelected(((SmartAction)action).isSelected());
                this.setAccelerator((KeyStroke)action.getValue(Action.ACCELERATOR_KEY));
                boolean visible=isVisibleInContext(this,m_smartActionManager);
                if (hideWhenDisabled(this) && !action.isEnabled())
                    visible=false;
                this.setVisible(visible);
            }
        }
    }

    /**
     * Implements a menu item that mimics action's selected state.
     */
    public static class SmartRadioButtonMenuItem extends JRadioButtonMenuItem {
        /** The manager. */
        protected SmartActionManager m_smartActionManager;

        /**
         * Creates an instance of this class.
         *
         * @param smartActionManager        the manager
         */
        public SmartRadioButtonMenuItem(SmartActionManager smartActionManager) {
            m_smartActionManager=smartActionManager;
        }
        /**
         * Creates menu item.
         */
        public SmartRadioButtonMenuItem() {
            setModel(new DefaultButtonModel());
        }
        /**
         * Creates a property change listener for the button.
         *
         * @param action                    action for which the listener is created
         * @return                          property change listener
         */
        protected PropertyChangeListener createActionPropertyChangeListener(Action action) {
            return new ButtonActionPropertyChangeListener(this,m_smartActionManager);
        }
        /**
         * Configures properties from given action.
         *
         * @param action                    the action
         */
        protected void configurePropertiesFromAction(Action action) {
            if (action!=null) {
                this.setIcon((Icon)action.getValue(Action.SMALL_ICON));
                this.setText((String)action.getValue(Action.NAME));
                this.setEnabled(action.isEnabled());
                Integer mnemonic=(Integer)action.getValue(Action.MNEMONIC_KEY);
                if (mnemonic!=null)
                    this.setMnemonic(mnemonic.intValue());
                this.setToolTipText((String)action.getValue(Action.SHORT_DESCRIPTION));
                this.setSelected(((SmartAction)action).isSelected());
                this.setAccelerator((KeyStroke)action.getValue(Action.ACCELERATOR_KEY));
                boolean visible=isVisibleInContext(this,m_smartActionManager);
                if (hideWhenDisabled(this) && !action.isEnabled())
                    visible=false;
                this.setVisible(visible);
            }
        }
    }

    /**
     * Implements a menu that may be hidden when not selected.
     */
    public static class SmartMenu extends JMenu {
        /** The manager. */
        protected SmartActionManager m_smartActionManager;

        /**
         * Creates an instance of this class.
         *
         * @param smartActionManager        the manager
         */
        public SmartMenu(SmartActionManager smartActionManager) {
            m_smartActionManager=smartActionManager;
        }
        /**
         * Creates a property change listener for the button.
         *
         * @param action                    action for which the listener is created
         * @return                          property change listener
         */
        protected PropertyChangeListener createActionPropertyChangeListener(Action action) {
            return new ButtonActionPropertyChangeListener(this,m_smartActionManager);
        }
        /**
         * Configures properties from given action.
         *
         * @param action                    the action
         */
        protected void configurePropertiesFromAction(Action action) {
            if (action!=null) {
                this.setIcon((Icon)action.getValue(Action.SMALL_ICON));
                this.setText((String)action.getValue(Action.NAME));
                this.setEnabled(action.isEnabled());
                Integer mnemonic=(Integer)action.getValue(Action.MNEMONIC_KEY);
                if (mnemonic!=null)
                    this.setMnemonic(mnemonic.intValue());
                this.setToolTipText((String)action.getValue(Action.SHORT_DESCRIPTION));
                boolean visible=isVisibleInContext(this,m_smartActionManager);
                if (hideWhenDisabled(this) && !action.isEnabled())
                    visible=false;
                this.setVisible(visible);
            }
        }
    }

    /**
     * Property change listener for changes to action properties.
     */
    protected static class ButtonActionPropertyChangeListener implements PropertyChangeListener {
        /** Target of this listener. */
        protected AbstractButton m_target;
        /** The smart action manager. */
        protected SmartActionManager m_smartActionManager;

        /**
         * Creates this listener and attaches it to a button.
         *
         * @param target                    the button
         * @param smartActionManager        the smart action manager
         */
        public ButtonActionPropertyChangeListener(AbstractButton target,SmartActionManager smartActionManager) {
            m_target=target;
            m_smartActionManager=smartActionManager;
        }
        /**
         * Called when property of an action has changed.
         *
         * @param event                     event descripbing the type of change
         */
        public void propertyChange(PropertyChangeEvent event) {
            String propertyName=event.getPropertyName();
            if (Action.NAME.equals(propertyName)) {
                String text=(String)event.getNewValue();
                m_target.setText(text);
                m_target.repaint();
            }
            else if (Action.SHORT_DESCRIPTION.equals(propertyName)) {
                String text=(String)event.getNewValue();
                m_target.setToolTipText(text);
            }
            else if ("enabled".equals(propertyName)) {
                Boolean enabledState=(Boolean)event.getNewValue();
                m_target.setEnabled(enabledState.booleanValue());
                boolean visible=isVisibleInContext(m_target,m_smartActionManager);
                if (hideWhenDisabled(m_target) && !enabledState.booleanValue())
                    visible=false;
                m_target.setVisible(visible);
                m_target.repaint();
            }
            else if (Action.SMALL_ICON.equals(propertyName)) {
                Icon icon=(Icon)event.getNewValue();
                m_target.setIcon(icon);
                m_target.invalidate();
                m_target.repaint();
            }
            else if (Action.MNEMONIC_KEY.equals(propertyName)) {
                Integer mnemonic=(Integer)event.getNewValue();
                m_target.setMnemonic(mnemonic.intValue());
                m_target.invalidate();
                m_target.repaint();
            }
            else if ("selected".equals(propertyName)) {
                Boolean selectedState=(Boolean)event.getNewValue();
                m_target.setSelected(selectedState.booleanValue());
                m_target.repaint();
            }
        }
    }

    /**
     * The manager of the modal menu.
     */
    public static class ModalMenuManager extends SmartActionDecorator {
        /** The modal smart action. */
        protected ModalSmartAction m_modalSmartAction;
        /** The localization manager. */
        protected LocalizationManager m_localizationManager;
        /** The listener for popup menus. */
        protected PopupMenuListener m_popupMenuListener;
        /** Determines the type of the options. */
        protected int m_optionsType;

        /**
         * Creates an instance of this class.
         *
         * @param actionID                      the action ID
         * @param localizationManager           the localization manager
         * @param modalSmartAction              the modal smart action
         * @param popupMenu                     the popup menu
         * @param optionsType                   determines the type of the options
         */
        public ModalMenuManager(String actionID,LocalizationManager localizationManager,ModalSmartAction modalSmartAction,JPopupMenu popupMenu,int optionsType) {
            super(actionID,localizationManager);
            m_localizationManager=localizationManager;
            m_modalSmartAction=modalSmartAction;
            m_optionsType=optionsType;
            m_popupMenuListener=new PopupMenuHandler();
            popupMenu.addPopupMenuListener(m_popupMenuListener);
            updateCurrentAction();
        }
        /**
         * Returns the decorated action.
         *
         * @return                              decorated action
         */
        protected SmartAction getDecoratedAction() {
            return m_modalSmartAction;
        }
        /**
         * Called when the action is performed.
         *
         * @param e                                 the event
         */
        public void actionPerformed(ActionEvent e) {
        }

        protected class PopupMenuHandler implements PopupMenuListener,ActionListener {
            public void popupMenuCanceled(PopupMenuEvent e) {
            }
            public void popupMenuWillBecomeInvisible(PopupMenuEvent e) {
            }
            public void popupMenuWillBecomeVisible(PopupMenuEvent e) {
                JPopupMenu popupMenu=(JPopupMenu)e.getSource();
                String[][] modalOptions=m_modalSmartAction.getModalOptions();
                if (modalOptions==null) {
                    popupMenu.removeAll();
                    return;
                }
                int numberOfOptions=modalOptions.length;
                int numberOfComponents=popupMenu.getComponentCount();
                for (int i=0;i<numberOfOptions;i++) {
                    JMenuItem menuItem;
                    if (i<numberOfComponents)
                        menuItem=(JMenuItem)popupMenu.getComponent(i);
                    else {
                        if (m_optionsType==1)
                            menuItem=new JCheckBoxMenuItem();
                        else if (m_optionsType==2)
                            menuItem=new JRadioButtonMenuItem();
                        else
                            menuItem=new JMenuItem();
                        menuItem.addActionListener(this);
                        popupMenu.add(menuItem);
                    }
                    menuItem.setActionCommand(modalOptions[i][0]);
                    menuItem.setText(m_localizationManager.getPhrase(modalOptions[i][1]));
                    menuItem.setSelected(m_modalSmartAction.isOptionSelected(i));
                }
                for (int i=numberOfComponents-1;i>=numberOfOptions;i--) {
                    JMenuItem menuItem=(JMenuItem)popupMenu.getComponent(i);
                    menuItem.removeActionListener(this);
                    popupMenu.remove(i);
                }
            }
            public void actionPerformed(ActionEvent e) {
                m_currentAction.actionPerformed(e);
            }
        }
    }
}
