package de.fzi.wim.guibase.localization;

import java.util.Map;
import java.util.HashMap;
import java.util.Locale;
import java.io.InputStream;
import java.io.IOException;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.ParserConfigurationException;
import org.xml.sax.SAXException;
import org.xml.sax.InputSource;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * This class loads data from a localization manager file. It hashes
 * the loaded LocalizationManager objects which can be queried by specifying their Locale.
 */
public class LocalizationManagerLoader {
    /** Map of LocalizationManager objects indexed by the hashcode of their Locale. */
    protected Map m_localesMap;

    /**
     * Creates a new LanguageLoader.
     */
    public LocalizationManagerLoader() {
        m_localesMap=new HashMap();
    }
    /**
     * Loads the data from an input source connected to a locale setting. Files specified by the
     * <code>include</code> tag will be parsed here as well.
     *
     * @param resourceName                          name of the resource
     * @throws SAXException                         thrown if there is a problem
     */
    public void loadLocalizationManagers(String resourceName) throws SAXException {
        Document document=null;
        InputStream inputStream=LocalizationManagerLoader.class.getClassLoader().getResourceAsStream(resourceName);
        try {
            DocumentBuilder builder=DocumentBuilderFactory.newInstance().newDocumentBuilder();
            document=builder.parse(new InputSource(inputStream));
        }
        catch (ParserConfigurationException e) {
            throw new SAXException("Parser configuration error",e);
        }
        catch (IOException e) {
            throw new SAXException("I/O exception reading file",e);
        }
        finally {
            try {
                inputStream.close();
            }
            catch (IOException e) {
                throw new SAXException("I/O exception closing file",e);
            }
        }
        NodeList list=document.getElementsByTagName("include");
        for (int i=0;i<list.getLength();i++) {
            Element node=(Element)list.item(i);
            String href=node.getAttribute("href");
            if (href==null)
                throw new SAXException("Attribute href is missing on the include element.");
            loadLocalizationManagers(href);
        }
        list=document.getElementsByTagName("language");
        for (int i=0;i<list.getLength();i++) {
            Node node=list.item(i);
            handleLanguage((Element)node);
        }
    }
    /**
     * Handles the loading of a language node.
     *
     * @param node                                  node to handle
     * @throws SAXException                         thrown if there is an error
     */
    protected void handleLanguage(Element node) throws SAXException {
        LocalizationManager localizationManager=getLocalizationManager(node);
        NodeList children=node.getChildNodes();
        for (int i=0;i<children.getLength();i++) {
            Node child=children.item(i);
            String childNodeName=child.getNodeName();
            if (child.getNodeType()==Node.ELEMENT_NODE && ("phrase".equals(childNodeName) || "image".equals(childNodeName)))
                handlePhraseOrImage(localizationManager,(Element)child);
        }
    }
    /**
     * Handles the loading of a phrase node. It adds the loaded key and associated phrase to
     * the provided Language object.
     *
     * @param localizationManager                   localization manager to fill in
     * @param node                                  phrase node to handler
     */
    protected void handlePhraseOrImage(LocalizationManager localizationManager,Element node) {
        String key=node.getAttribute("key");
        if (key!=null ) {
            String value=node.getAttribute("value");
            if (value==null) {
                Node firstChild=node.getFirstChild();
                if (firstChild!=null && firstChild.getNodeType()==Node.TEXT_NODE)
                    value=firstChild.getNodeValue();
            }
            if (key!=null && value!=null && key.length()>0)
                localizationManager.addPhrase(key,value);
        }
    }
    /**
     * Returns a <code>Locale</code> object from given node and attribute names.
     *
     * @param node                                      node that is processed
     * @param languageAttributeName                     name of the attribute determining the language
     * @param countryAttributeName                      name of the attribute determining the country
     * @return                                          locale for given node (or <code>null</code>)
     */
    protected Locale getLocaleFromNode(Element node,String languageAttributeName,String countryAttributeName) {
        Locale result=null;
        String language=node.getAttribute(languageAttributeName);
        if (language!=null) {
            String country=node.getAttribute(countryAttributeName);
            if (country==null)
                result=new Locale(language);
            else
                result=new Locale(language,country);
        }
        return result;
    }

    /**
     * Returns the localization manager from given attributes.
     *
     * @param node                                  node to handle
     * @return                                      localization manager with parameters from given node
     * @throws SAXException                         thrown if there is an error
     */
    protected LocalizationManager getLocalizationManager(Element node) throws SAXException {
        Locale locale=getLocaleFromNode(node,"name","country");
        if (locale==null)
            throw new SAXException("Attribute 'language' must be present.");
        LocalizationManager localizationManager=(LocalizationManager)m_localesMap.get(locale);
        if (localizationManager==null) {
            LocalizationManager parentLocalizationManager=null;
            Locale parentLocale=getLocaleFromNode(node,"parentlanguage","parentcountry");
            if (parentLocale!=null)
                parentLocalizationManager=getLocalizationManager(parentLocale);
            localizationManager=new LocalizationManager(locale,parentLocalizationManager);
            m_localesMap.put(locale,localizationManager);
        }
        return localizationManager;
    }
    /**
     * Returns the Language object associated with the provided Locale.
     *
     * @param locale                                    the locale for which localization manager is requested
     * @return                                          localization manager for given locale
     */
    public LocalizationManager getLocalizationManager(Locale locale) {
        LocalizationManager result=(LocalizationManager)m_localesMap.get(new Locale(locale.getLanguage(),locale.getCountry()));
        if (result==null)
            result=(LocalizationManager)m_localesMap.get(new Locale(locale.getLanguage()));
        if (result==null)
            result=(LocalizationManager)m_localesMap.get(new Locale("en"));
        return result;
    }
}
